package org.fastfilter.tools;

import java.io.*;
import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.util.Scanner;

import org.fastfilter.xorplus.XorPlus8;

public class PasswordLookup {

    public static void main(String... args) throws Exception {
        if (args.length != 1) {
            System.out.println("Usage: java " + PasswordLookup.class.getName() + " <filterFile> \n"
                    + "Requires a filter file generated by " + BuildFilterFile.class.getName());
            return;
        }
        String filterFile = args[0];
        DataInputStream in = openFile(filterFile, new File(filterFile).length() - (8 << BuildFilterFile.SEGMENT_BITS));
        long[] segmentStarts = new long[1 << BuildFilterFile.SEGMENT_BITS];
        for (int i = 0; i < segmentStarts.length; i++) {
            segmentStarts[i] = in.readLong();
        }
        in.close();
        Scanner scanner = new Scanner(System.in);
        while (true) {
            Console console = System.console();
            String password;
            if (console != null) {
                password = new String(console.readPassword("Password? "));
            } else {
                System.out.println("Password? ");
                password = scanner.nextLine();
            }
            if (password.length() == 0) {
                break;
            }
            testPassword(filterFile, segmentStarts, password);
        }
        scanner.close();
    }

    private static void testPassword(String filterFile, long[] segmentStarts, String password) throws Exception {
        Result utf8 = testPassword(filterFile, segmentStarts, password, "UTF-8");
        Result ascii = testPassword(filterFile, segmentStarts, password, "ASCII");
        Result max = utf8.compareTo(ascii) >= 0 ? utf8 : ascii;
        System.out.println(max);
    }

    private static Result testPassword(String filterFile, long[] segmentStarts, String password, String charset) throws Exception {
        byte[] passwordBytes = password.getBytes(Charset.forName(charset));
        MessageDigest md = MessageDigest.getInstance("SHA-1");
        byte[] sha1 = md.digest(passwordBytes);
        long hash = 0;
        for (int i = 0; i < 8; i++) {
            hash = (hash << 8) | (sha1[i] & 0xff);
        }
        // set the lowest bit to 0
        long key = hash ^ (hash & 1);
        int segment = (int) (key >>> (64 - BuildFilterFile.SEGMENT_BITS));
        DataInputStream in = openFile(filterFile, segmentStarts[segment]);
        XorPlus8 filter = new XorPlus8(in);
        in.close();
        if (filter.mayContain(key)) {
            return Result.FOUND;
        } else if (filter.mayContain(key | 1)) {
            return Result.FOUND_COMMON;
        } else {
            return Result.NOT_FOUND;
        }
    }

    enum Result {
        NOT_FOUND("Not found"),
        FOUND("Found"),
        FOUND_COMMON("Found; common");
        private final String message;
        Result(String message) {
            this.message = message;
        }
        public String toString() {
            return message;
        }
    }

    static DataInputStream openFile(String fileName, long skip) throws Exception {
        DataInputStream in = new DataInputStream(new BufferedInputStream(new FileInputStream(fileName)));
        while (skip > 0) {
            long skipped = in.skip(skip);
            if (skipped <= 0) {
                break;
            }
            skip -= skipped;
        }
        return in;
    }

}
