package org.fastfilter.xor;

import org.fastfilter.Filter;
import org.fastfilter.utils.Hash;

/**
 * The Xor Filter, a new algorithm that can replace a Bloom filter.
 *
 * It needs 1.23 log(1/fpp) bits per key. It is related to the BDZ algorithm [1]
 * (a minimal perfect hash function algorithm).
 *
 * [1] paper: Simple and Space-Efficient Minimal Perfect Hash Functions -
 * http://cmph.sourceforge.net/papers/wads07.pdf
 */
public class Xor16 implements Filter {

    private static final int BITS_PER_FINGERPRINT = 16;
    private static final int HASHES = 3;
    private static final int FACTOR_TIMES_100 = 123;
    private final int blockLength;
    private long seed;
    private short[] fingerprints;
    private final int bitCount;

    public long getBitCount() {
        return bitCount;
    }

    private static int getArrayLength(int size) {
        return (int) (HASHES + (long) FACTOR_TIMES_100 * size / 100);
    }

    public static Xor16 construct(long[] keys) {
        return new Xor16(keys);
    }

    public Xor16(long[] keys) {
        int size = keys.length;
        int arrayLength = getArrayLength(size);
        bitCount = arrayLength * BITS_PER_FINGERPRINT;
        blockLength = arrayLength / HASHES;
        long[] reverseOrder = new long[size];
        byte[] reverseH = new byte[size];
        int reverseOrderPos;
        long seed;
        do {
            seed = Hash.randomSeed();
            byte[] t2count = new byte[arrayLength];
            long[] t2 = new long[arrayLength];
            for (long k : keys) {
                for (int hi = 0; hi < HASHES; hi++) {
                    int h = getHash(k, seed, hi);
                    t2[h] ^= k;
                    if (t2count[h] > 120) {
                        throw new IllegalArgumentException();
                    }
                    t2count[h]++;
                }
            }
            int[] alone = new int[arrayLength];
            int alonePos = 0;
            reverseOrderPos = 0;
            for (int nextAloneCheck = 0; nextAloneCheck < arrayLength; ) {
                while (nextAloneCheck < arrayLength) {
                    if (t2count[nextAloneCheck] == 1) {
                        alone[alonePos++] = nextAloneCheck;
                        // break;
                    }
                    nextAloneCheck++;
                }
                while (alonePos > 0) {
                    int i = alone[--alonePos];
                    if (t2count[i] == 0) {
                        continue;
                    }
                    long k = t2[i];
                    byte found = -1;
                    for (int hi = 0; hi < HASHES; hi++) {
                        int h = getHash(k, seed, hi);
                        int newCount = --t2count[h];
                        if (newCount == 0) {
                            found = (byte) hi;
                        } else {
                            if (newCount == 1) {
                                alone[alonePos++] = h;
                            }
                            t2[h] ^= k;
                        }
                    }
                    reverseOrder[reverseOrderPos] = k;
                    reverseH[reverseOrderPos] = found;
                    reverseOrderPos++;
                }
            }
        } while (reverseOrderPos != size);
        this.seed = seed;
        short[] fp = new short[arrayLength];
        for (int i = reverseOrderPos - 1; i >= 0; i--) {
            long k = reverseOrder[i];
            int found = reverseH[i];
            int change = -1;
            long hash = Hash.hash64(k, seed);
            int xor = fingerprint(hash);
            for (int hi = 0; hi < HASHES; hi++) {
                int h = getHash(k, seed, hi);
                if (found == hi) {
                    change = h;
                } else {
                    xor ^= fp[h];
                }
            }
            fp[change] = (short) xor;
        }
        fingerprints = new short[arrayLength];
        System.arraycopy(fp, 0, fingerprints, 0, fp.length);
    }

    @Override
    public boolean mayContain(long key) {
        long hash = Hash.hash64(key, seed);
        int f = fingerprint(hash);
        int r0 = (int) hash;
        int r1 = (int) Long.rotateLeft(hash, 21);
        int r2 = (int) Long.rotateLeft(hash, 42);
        int h0 = Hash.reduce(r0, blockLength);
        int h1 = Hash.reduce(r1, blockLength) + blockLength;
        int h2 = Hash.reduce(r2, blockLength) + 2 * blockLength;
        f ^= fingerprints[h0] ^ fingerprints[h1] ^ fingerprints[h2];
        return (f & 0xffff) == 0;
    }

    private int getHash(long key, long seed, int index) {
        long r = Long.rotateLeft(Hash.hash64(key, seed), 21 * index);
        r = Hash.reduce((int) r, blockLength);
        r = r + index * blockLength;
        return (int) r;
    }

    private int fingerprint(long hash) {
        return (int) (hash & ((1 << BITS_PER_FINGERPRINT) - 1));
    }

}
