/*
 * Decompiled with CFR 0.152.
 */
package reactor.fn.timer;

import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import reactor.bus.registry.Registration;
import reactor.bus.selector.Selector;
import reactor.core.support.Assert;
import reactor.core.support.NamedDaemonThreadFactory;
import reactor.fn.Consumer;
import reactor.fn.Pausable;
import reactor.fn.timer.TimeUtils;
import reactor.fn.timer.Timer;
import reactor.jarjar.com.lmax.disruptor.EventFactory;
import reactor.jarjar.com.lmax.disruptor.RingBuffer;

public class HashWheelTimer
implements Timer {
    public static final int DEFAULT_WHEEL_SIZE = 512;
    private static final String DEFAULT_TIMER_NAME = "hash-wheel-timer";
    private final RingBuffer<Set<TimerRegistration>> wheel;
    private final int resolution;
    private final Thread loop;
    private final Executor executor;
    private final WaitStrategy waitStrategy;

    public HashWheelTimer() {
        this(100, 512, new SleepWait());
    }

    public HashWheelTimer(int resolution) {
        this(resolution, 512, new SleepWait());
    }

    public HashWheelTimer(int res, int wheelSize, WaitStrategy waitStrategy) {
        this(DEFAULT_TIMER_NAME, res, wheelSize, waitStrategy, Executors.newFixedThreadPool(1));
    }

    public HashWheelTimer(String name, int res, int wheelSize, WaitStrategy strategy, Executor exec) {
        this.waitStrategy = strategy;
        this.wheel = RingBuffer.createSingleProducer(new EventFactory<Set<TimerRegistration>>(){

            @Override
            public Set<TimerRegistration> newInstance() {
                return new ConcurrentSkipListSet<TimerRegistration>();
            }
        }, wheelSize);
        this.resolution = res;
        this.loop = new NamedDaemonThreadFactory(name).newThread(new Runnable(){

            @Override
            public void run() {
                long deadline = System.currentTimeMillis();
                while (true) {
                    Set registrations = (Set)HashWheelTimer.this.wheel.get(HashWheelTimer.this.wheel.getCursor());
                    for (TimerRegistration r : registrations) {
                        if (r.isCancelled()) {
                            registrations.remove(r);
                            continue;
                        }
                        if (r.ready()) {
                            HashWheelTimer.this.executor.execute(r);
                            registrations.remove(r);
                            if (r.isCancelAfterUse()) continue;
                            HashWheelTimer.this.reschedule(r);
                            continue;
                        }
                        if (r.isPaused()) {
                            HashWheelTimer.this.reschedule(r);
                            continue;
                        }
                        r.decrement();
                    }
                    deadline += (long)HashWheelTimer.this.resolution;
                    try {
                        HashWheelTimer.this.waitStrategy.waitUntil(deadline);
                    }
                    catch (InterruptedException e) {
                        return;
                    }
                    HashWheelTimer.this.wheel.publish(HashWheelTimer.this.wheel.next());
                }
            }
        });
        this.executor = exec;
        this.start();
    }

    @Override
    public long getResolution() {
        return this.resolution;
    }

    @Override
    public Registration<? extends Consumer<Long>> schedule(Consumer<Long> consumer, long period, TimeUnit timeUnit, long delayInMilliseconds) {
        Assert.isTrue(!this.loop.isInterrupted(), "Cannot submit tasks to this timer as it has been cancelled.");
        return this.schedule(TimeUnit.MILLISECONDS.convert(period, timeUnit), delayInMilliseconds, consumer);
    }

    @Override
    public Registration<? extends Consumer<Long>> submit(Consumer<Long> consumer, long period, TimeUnit timeUnit) {
        Assert.isTrue(!this.loop.isInterrupted(), "Cannot submit tasks to this timer as it has been cancelled.");
        long ms = TimeUnit.MILLISECONDS.convert(period, timeUnit);
        return this.schedule(ms, ms, consumer).cancelAfterUse();
    }

    @Override
    public Registration<? extends Consumer<Long>> submit(Consumer<Long> consumer) {
        return this.submit(consumer, this.resolution, TimeUnit.MILLISECONDS);
    }

    @Override
    public Registration<? extends Consumer<Long>> schedule(Consumer<Long> consumer, long period, TimeUnit timeUnit) {
        return this.schedule(TimeUnit.MILLISECONDS.convert(period, timeUnit), 0L, consumer);
    }

    private TimerRegistration<? extends Consumer<Long>> schedule(long recurringTimeout, long firstDelay, Consumer<Long> consumer) {
        Assert.isTrue(recurringTimeout >= (long)this.resolution, "Cannot schedule tasks for amount of time less than timer precision.");
        long offset = recurringTimeout / (long)this.resolution;
        long rounds = offset / (long)this.wheel.getBufferSize();
        long firstFireOffset = firstDelay / (long)this.resolution;
        long firstFireRounds = firstFireOffset / (long)this.wheel.getBufferSize();
        TimerRegistration<Consumer<Long>> r = new TimerRegistration<Consumer<Long>>(firstFireRounds, offset, consumer, rounds);
        this.wheel.get(this.wheel.getCursor() + firstFireOffset + 1L).add(r);
        return r;
    }

    private void reschedule(TimerRegistration registration) {
        registration.reset();
        this.wheel.get(this.wheel.getCursor() + registration.getOffset()).add(registration);
    }

    public void start() {
        this.loop.start();
        this.wheel.publish(0L);
    }

    @Override
    public void cancel() {
        this.loop.interrupt();
    }

    public String toString() {
        return String.format("HashWheelTimer { Buffer Size: %d, Resolution: %d }", this.wheel.getBufferSize(), this.resolution);
    }

    public static class SleepWait
    implements WaitStrategy {
        @Override
        public void waitUntil(long deadlineMilliseconds) throws InterruptedException {
            long sleepTimeMs = deadlineMilliseconds - System.currentTimeMillis();
            if (sleepTimeMs > 0L) {
                Thread.sleep(sleepTimeMs);
            }
        }
    }

    public static class BusySpinWait
    implements WaitStrategy {
        @Override
        public void waitUntil(long deadlineMilliseconds) throws InterruptedException {
            while (deadlineMilliseconds >= System.currentTimeMillis()) {
                if (!Thread.currentThread().isInterrupted()) continue;
                throw new InterruptedException();
            }
        }
    }

    public static class YieldingWait
    implements WaitStrategy {
        @Override
        public void waitUntil(long deadlineMilliseconds) throws InterruptedException {
            while (deadlineMilliseconds >= System.currentTimeMillis()) {
                Thread.yield();
                if (!Thread.currentThread().isInterrupted()) continue;
                throw new InterruptedException();
            }
        }
    }

    public static class TimerRegistration<T extends Consumer<Long>>
    implements Runnable,
    Comparable,
    Pausable,
    Registration {
        public static int STATUS_PAUSED = 1;
        public static int STATUS_CANCELLED = -1;
        public static int STATUS_READY = 0;
        private final T delegate;
        private final long rescheduleRounds;
        private final long scheduleOffset;
        private final AtomicLong rounds;
        private final AtomicInteger status;
        private final AtomicBoolean cancelAfterUse;
        private final boolean lifecycle;

        public TimerRegistration(long rounds, long offset, T delegate, long rescheduleRounds) {
            Assert.notNull(delegate, "Delegate cannot be null");
            this.rescheduleRounds = rescheduleRounds;
            this.scheduleOffset = offset;
            this.delegate = delegate;
            this.rounds = new AtomicLong(rounds);
            this.status = new AtomicInteger(STATUS_READY);
            this.cancelAfterUse = new AtomicBoolean(false);
            this.lifecycle = Pausable.class.isAssignableFrom(delegate.getClass());
        }

        public void decrement() {
            this.rounds.decrementAndGet();
        }

        public boolean ready() {
            return this.status.get() == STATUS_READY && this.rounds.get() == 0L;
        }

        @Override
        public void run() {
            this.delegate.accept(TimeUtils.approxCurrentTimeMillis());
        }

        public void reset() {
            this.status.set(STATUS_READY);
            this.rounds.set(this.rescheduleRounds);
        }

        @Override
        public Registration cancel() {
            if (!this.isCancelled()) {
                if (this.lifecycle) {
                    ((Pausable)this.delegate).cancel();
                }
                this.status.set(STATUS_CANCELLED);
            }
            return this;
        }

        @Override
        public boolean isCancelled() {
            return this.status.get() == STATUS_CANCELLED;
        }

        @Override
        public Registration pause() {
            if (!this.isPaused()) {
                if (this.lifecycle) {
                    ((Pausable)this.delegate).pause();
                }
                this.status.set(STATUS_PAUSED);
            }
            return this;
        }

        @Override
        public boolean isPaused() {
            return this.status.get() == STATUS_PAUSED;
        }

        @Override
        public Registration resume() {
            if (this.isPaused()) {
                if (this.lifecycle) {
                    ((Pausable)this.delegate).resume();
                }
                this.reset();
            }
            return this;
        }

        private long getOffset() {
            return this.scheduleOffset;
        }

        @Override
        public Selector getSelector() {
            return null;
        }

        public Object getObject() {
            return this.delegate;
        }

        public TimerRegistration<T> cancelAfterUse() {
            this.cancelAfterUse.set(true);
            return this;
        }

        @Override
        public boolean isCancelAfterUse() {
            return this.cancelAfterUse.get();
        }

        public int compareTo(Object o) {
            TimerRegistration other = (TimerRegistration)o;
            if (this.rounds.get() == other.rounds.get()) {
                return other == this ? 0 : -1;
            }
            return Long.compare(this.rounds.get(), other.rounds.get());
        }

        public String toString() {
            return String.format("HashWheelTimer { Rounds left: %d, Status: %d }", this.rounds.get(), this.status.get());
        }
    }

    public static interface WaitStrategy {
        public void waitUntil(long var1) throws InterruptedException;
    }
}

