/*
 * Copyright (c) 2011-2016 Pivotal Software Inc, All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package reactor.core.publisher;

import java.util.NoSuchElementException;
import java.util.Objects;

import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;
import reactor.core.Fuseable;
import reactor.core.Receiver;

/**
 * Take the very last value from a Publisher source and and emit that one.
 *
 * @param <T> the value type
 */
final class MonoTakeLastOne<T> extends MonoSource<T, T> implements Fuseable {

	final T defaultValue;

    public MonoTakeLastOne(Publisher<? extends T> source) {
        super(source);
	    this.defaultValue = null;
    }

	public MonoTakeLastOne(Publisher<? extends T> source, T defaultValue) {
		super(source);
		this.defaultValue = Objects.requireNonNull(defaultValue, "defaultValue");
	}

    @Override
    public void subscribe(Subscriber<? super T> s) {
        source.subscribe(new TakeLastOneSubscriber<>(s, defaultValue, true));
    }

	static final class TakeLastOneSubscriber<T>
			extends Operators.MonoSubscriber<T, T>
			implements Receiver {

		final boolean mustEmit;
		final T       defaultValue;
		Subscription s;

		public TakeLastOneSubscriber(Subscriber<? super T> actual,
				T defaultValue,
				boolean mustEmit) {
			super(actual);
			this.defaultValue = defaultValue;
			this.mustEmit = mustEmit;
		}

		@Override
		public void onSubscribe(Subscription s) {
			if (Operators.validate(this.s, s)) {
				this.s = s;

				actual.onSubscribe(this);

				s.request(Long.MAX_VALUE);
			}

		}

		@Override
		public void onNext(T t) {
			value = t;
		}

		@Override
		public void onComplete() {
			T v = value;
			if (v == null) {
				if (mustEmit) {
					if(defaultValue != null){
						complete(defaultValue);
					}
					else {
						actual.onError(Operators.onOperatorError(new NoSuchElementException(
								"Flux#last() didn't observe any " + "onNext signal")));
					}
				}
				else {
					actual.onComplete();
				}
				return;
			}
			complete(v);
		}

		@Override
		public void cancel() {
			super.cancel();
			s.cancel();
		}

		@Override
		public void setValue(T value) {
			// value is always in a field
		}

		@Override
		public Object upstream() {
			return s;
		}
	}
}
