/*
 * Copyright (c) 2011-2016 Pivotal Software Inc, All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package reactor.core.publisher;

import org.reactivestreams.Subscriber;

/**
 * Hides the identities of the upstream Publisher object and its Subscription as well.
 *
 * @param <T> the value type
 * @see <a href="https://github.com/reactor/reactive-streams-commons">Reactive-Streams-Commons</a>
 */
final class ParallelFluxHide<T> extends ParallelFlux<T> {

	final ParallelFlux<T> source;

	public ParallelFluxHide(ParallelFlux<T> source) {
		this.source = source;
	}

	@Override
	public boolean isOrdered() {
		return source.isOrdered();
	}

	@Override
	public int parallelism() {
		return source.parallelism();
	}

	@Override
	public void subscribe(Subscriber<? super T>[] subscribers) {
		if (!validate(subscribers)) {
			return;
		}

		int n = subscribers.length;
		@SuppressWarnings("unchecked") Subscriber<? super T>[] parents =
				new Subscriber[n];
		for (int i = 0; i < n; i++) {
			parents[i] = new FluxHide.HideSubscriber<>(subscribers[i]);
		}

		source.subscribe(parents);
	}
}
