//
// [The "BSD license"]
//  Copyright (c) 2012 Terence Parr
//  Copyright (c) 2012 Sam Harwell
//  Copyright (c) 2014 Eric Vergnaud
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The name of the author may not be used to endorse or promote products
//     derived from this software without specific prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
//  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
//  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

var DFAState = require('./DFAState').DFAState;
var ATNConfigSet = require('./../atn/ATNConfigSet').ATNConfigSet;
var DFASerializer = require('./DFASerializer').DFASerializer;
var LexerDFASerializer = require('./DFASerializer').LexerDFASerializer;

function DFAStatesSet() {
    return this;
}

Object.defineProperty(DFAStatesSet.prototype, "length", {
    get: function () {
        return Object.keys(this).length;
    }
});

function DFA(atnStartState, decision) {
    if (decision === undefined) {
        decision = 0;
    }
    // From which ATN state did we create this DFA?
    this.atnStartState = atnStartState;
    this.decision = decision;
    // A set of all DFA states. Use {@link Map} so we can get old state back
    // ({@link Set} only allows you to see if it's there).
    this._states = new DFAStatesSet();
    this.s0 = null;
    // {@code true} if this DFA is for a precedence decision; otherwise,
    // {@code false}. This is the backing field for {@link //isPrecedenceDfa},
    // {@link //setPrecedenceDfa}.
    this.precedenceDfa = false;
    return this;
}

// Get the start state for a specific precedence value.
//
// @param precedence The current precedence.
// @return The start state corresponding to the specified precedence, or
// {@code null} if no start state exists for the specified precedence.
//
// @throws IllegalStateException if this is not a precedence DFA.
// @see //isPrecedenceDfa()

DFA.prototype.getPrecedenceStartState = function (precedence) {
    if (!(this.precedenceDfa)) {
        throw ("Only precedence DFAs may contain a precedence start state.");
    }
    // s0.edges is never null for a precedence DFA
    if (precedence < 0 || precedence >= this.s0.edges.length) {
        return null;
    }
    return this.s0.edges[precedence] || null;
};

// Set the start state for a specific precedence value.
//
// @param precedence The current precedence.
// @param startState The start state corresponding to the specified
// precedence.
//
// @throws IllegalStateException if this is not a precedence DFA.
// @see //isPrecedenceDfa()
//
DFA.prototype.setPrecedenceStartState = function (precedence, startState) {
    if (!(this.precedenceDfa)) {
        throw ("Only precedence DFAs may contain a precedence start state.");
    }
    if (precedence < 0) {
        return;
    }

    // synchronization on s0 here is ok. when the DFA is turned into a
    // precedence DFA, s0 will be initialized once and not updated again
    // s0.edges is never null for a precedence DFA
    this.s0.edges[precedence] = startState;
};

//
// Sets whether this is a precedence DFA. If the specified value differs
// from the current DFA configuration, the following actions are taken;
// otherwise no changes are made to the current DFA.
//
// <ul>
// <li>The {@link //states} map is cleared</li>
// <li>If {@code precedenceDfa} is {@code false}, the initial state
// {@link //s0} is set to {@code null}; otherwise, it is initialized to a new
// {@link DFAState} with an empty outgoing {@link DFAState//edges} array to
// store the start states for individual precedence values.</li>
// <li>The {@link //precedenceDfa} field is updated</li>
// </ul>
//
// @param precedenceDfa {@code true} if this is a precedence DFA; otherwise,
// {@code false}

DFA.prototype.setPrecedenceDfa = function (precedenceDfa) {
    if (this.precedenceDfa !== precedenceDfa) {
        this._states = new DFAStatesSet();
        if (precedenceDfa) {
            var precedenceState = new DFAState(new ATNConfigSet());
            precedenceState.edges = [];
            precedenceState.isAcceptState = false;
            precedenceState.requiresFullContext = false;
            this.s0 = precedenceState;
        } else {
            this.s0 = null;
        }
        this.precedenceDfa = precedenceDfa;
    }
};

Object.defineProperty(DFA.prototype, "states", {
    get: function () {
        return this._states;
    }
});

// Return a list of all states in this DFA, ordered by state number.
DFA.prototype.sortedStates = function () {
    // states_ is a map of state/state, where key=value
    var keys = Object.keys(this._states);
    var list = [];
    for (var i = 0; i < keys.length; i++) {
        list.push(this._states[keys[i]]);
    }
    return list.sort(function (a, b) {
        return a.stateNumber - b.stateNumber;
    });
};

DFA.prototype.toString = function (literalNames, symbolicNames) {
    literalNames = literalNames || null;
    symbolicNames = symbolicNames || null;
    if (this.s0 === null) {
        return "";
    }
    var serializer = new DFASerializer(this, literalNames, symbolicNames);
    return serializer.toString();
};

DFA.prototype.toLexerString = function () {
    if (this.s0 === null) {
        return "";
    }
    var serializer = new LexerDFASerializer(this);
    return serializer.toString();
};

exports.DFA = DFA;
