// [The "BSD license"]
//  Copyright (c) 2012 Terence Parr
//  Copyright (c) 2012 Sam Harwell
//  Copyright (c) 2014 Eric Vergnaud
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The name of the author may not be used to endorse or promote products
//     derived from this software without specific prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
//  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
//  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
///

// The basic notion of a tree has a parent, a payload, and a list of children.
//  It is the most abstract interface for all the trees used by ANTLR.
///

var Token = require('./../Token').Token;
var Interval = require('./../IntervalSet').Interval;
var INVALID_INTERVAL = new Interval(-1, -2);
var Utils = require('../Utils.js');


function Tree() {
    return this;
}

function SyntaxTree() {
    Tree.call(this);
    return this;
}

SyntaxTree.prototype = Object.create(Tree.prototype);
SyntaxTree.prototype.constructor = SyntaxTree;

function ParseTree() {
    SyntaxTree.call(this);
    return this;
}

ParseTree.prototype = Object.create(SyntaxTree.prototype);
ParseTree.prototype.constructor = ParseTree;

function RuleNode() {
    ParseTree.call(this);
    return this;
}

RuleNode.prototype = Object.create(ParseTree.prototype);
RuleNode.prototype.constructor = RuleNode;

function TerminalNode() {
    ParseTree.call(this);
    return this;
}

TerminalNode.prototype = Object.create(ParseTree.prototype);
TerminalNode.prototype.constructor = TerminalNode;

function ErrorNode() {
    TerminalNode.call(this);
    return this;
}

ErrorNode.prototype = Object.create(TerminalNode.prototype);
ErrorNode.prototype.constructor = ErrorNode;

function ParseTreeVisitor() {
    return this;
}

ParseTreeVisitor.prototype.visit = function (ctx) {
    if (Utils.isArray(ctx)) {
        var self = this;
        return ctx.map(function (child) {
            return visitAtom(self, child)
        });
    } else {
        return visitAtom(this, ctx);
    }
};

ParseTreeVisitor.prototype.visitTerminal = function (node) {
};

ParseTreeVisitor.prototype.visitErrorNode = function (node) {
};


var visitAtom = function (visitor, ctx) {
    if (ctx.parser === undefined) { //is terminal
        return;
    }

    var name = ctx.parser.ruleNames[ctx.ruleIndex];
    var funcName = "visit" + Utils.titleCase(name);

    return visitor[funcName](ctx);
};

function ParseTreeListener() {
    return this;
}

ParseTreeListener.prototype.visitTerminal = function (node) {
};

ParseTreeListener.prototype.visitErrorNode = function (node) {
};

ParseTreeListener.prototype.enterEveryRule = function (node) {
};

ParseTreeListener.prototype.exitEveryRule = function (node) {
};

function TerminalNodeImpl(symbol) {
    TerminalNode.call(this);
    this.parentCtx = null;
    this.symbol = symbol;
    return this;
}

TerminalNodeImpl.prototype = Object.create(TerminalNode.prototype);
TerminalNodeImpl.prototype.constructor = TerminalNodeImpl;

TerminalNodeImpl.prototype.getChild = function (i) {
    return null;
};

TerminalNodeImpl.prototype.getSymbol = function () {
    return this.symbol;
};

TerminalNodeImpl.prototype.getParent = function () {
    return this.parentCtx;
};

TerminalNodeImpl.prototype.getPayload = function () {
    return this.symbol;
};

TerminalNodeImpl.prototype.getSourceInterval = function () {
    if (this.symbol === null) {
        return INVALID_INTERVAL;
    }
    var tokenIndex = this.symbol.tokenIndex;
    return new Interval(tokenIndex, tokenIndex);
};

TerminalNodeImpl.prototype.getChildCount = function () {
    return 0;
};

TerminalNodeImpl.prototype.accept = function (visitor) {
    return visitor.visitTerminal(this);
};

TerminalNodeImpl.prototype.getText = function () {
    return this.symbol.text;
};

TerminalNodeImpl.prototype.toString = function () {
    if (this.symbol.type === Token.EOF) {
        return "<EOF>";
    } else {
        return this.symbol.text;
    }
};

// Represents a token that was consumed during resynchronization
// rather than during a valid match operation. For example,
// we will create this kind of a node during single token insertion
// and deletion as well as during "consume until error recovery set"
// upon no viable alternative exceptions.

function ErrorNodeImpl(token) {
    TerminalNodeImpl.call(this, token);
    return this;
}

ErrorNodeImpl.prototype = Object.create(TerminalNodeImpl.prototype);
ErrorNodeImpl.prototype.constructor = ErrorNodeImpl;

ErrorNodeImpl.prototype.isErrorNode = function () {
    return true;
};

ErrorNodeImpl.prototype.accept = function (visitor) {
    return visitor.visitErrorNode(this);
};

function ParseTreeWalker() {
    return this;
}

ParseTreeWalker.prototype.walk = function (listener, t) {
    var errorNode = t instanceof ErrorNode ||
        (t.isErrorNode !== undefined && t.isErrorNode());
    if (errorNode) {
        listener.visitErrorNode(t);
    } else if (t instanceof TerminalNode) {
        listener.visitTerminal(t);
    } else {
        this.enterRule(listener, t);
        for (var i = 0; i < t.getChildCount(); i++) {
            var child = t.getChild(i);
            this.walk(listener, child);
        }
        this.exitRule(listener, t);
    }
};
//
// The discovery of a rule node, involves sending two events: the generic
// {@link ParseTreeListener//enterEveryRule} and a
// {@link RuleContext}-specific event. First we trigger the generic and then
// the rule specific. We to them in reverse order upon finishing the node.
//
ParseTreeWalker.prototype.enterRule = function (listener, r) {
    var ctx = r.getRuleContext();
    listener.enterEveryRule(ctx);
    ctx.enterRule(listener);
};

ParseTreeWalker.prototype.exitRule = function (listener, r) {
    var ctx = r.getRuleContext();
    ctx.exitRule(listener);
    listener.exitEveryRule(ctx);
};

ParseTreeWalker.DEFAULT = new ParseTreeWalker();

exports.RuleNode = RuleNode;
exports.ErrorNode = ErrorNode;
exports.TerminalNode = TerminalNode;
exports.ErrorNodeImpl = ErrorNodeImpl;
exports.TerminalNodeImpl = TerminalNodeImpl;
exports.ParseTreeListener = ParseTreeListener;
exports.ParseTreeVisitor = ParseTreeVisitor;
exports.ParseTreeWalker = ParseTreeWalker;
exports.INVALID_INTERVAL = INVALID_INTERVAL;