/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.oracle.logminer;

import io.debezium.annotation.NotThreadSafe;
import io.debezium.connector.oracle.OracleConnector;
import io.debezium.connector.oracle.OracleOffsetContext;
import io.debezium.connector.oracle.logminer.LogMinerHelper;
import io.debezium.connector.oracle.logminer.TransactionalBufferMetrics;
import io.debezium.pipeline.ErrorHandler;
import io.debezium.pipeline.source.spi.ChangeEventSource;
import io.debezium.util.Threads;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import org.apache.kafka.connect.errors.DataException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public final class TransactionalBuffer {
    private static final Logger LOGGER = LoggerFactory.getLogger(TransactionalBuffer.class);
    private final Map<String, Transaction> transactions = new HashMap<String, Transaction>();
    private final ExecutorService executor;
    private final AtomicInteger taskCounter;
    private final ErrorHandler errorHandler;
    private final Supplier<Integer> commitQueueCapacity;
    private TransactionalBufferMetrics metrics;
    private final Set<String> abandonedTransactionIds;
    private final Set<String> rolledBackTransactionIds;
    private BigDecimal largestScn;
    private BigDecimal lastCommittedScn;

    TransactionalBuffer(String logicalName, ErrorHandler errorHandler, TransactionalBufferMetrics metrics, int inCommitQueueCapacity) {
        ArrayBlockingQueue<Runnable> workQueue = new ArrayBlockingQueue<Runnable>(inCommitQueueCapacity);
        this.executor = new ThreadPoolExecutor(1, 1, Integer.MAX_VALUE, TimeUnit.MILLISECONDS, workQueue, Threads.threadFactory(OracleConnector.class, logicalName, "transactional-buffer", true, false), new ThreadPoolExecutor.CallerRunsPolicy());
        this.commitQueueCapacity = workQueue::remainingCapacity;
        this.taskCounter = new AtomicInteger();
        this.errorHandler = errorHandler;
        this.metrics = metrics;
        this.largestScn = BigDecimal.ZERO;
        this.lastCommittedScn = BigDecimal.ZERO;
        this.abandonedTransactionIds = new HashSet<String>();
        this.rolledBackTransactionIds = new HashSet<String>();
    }

    BigDecimal getLargestScn() {
        return this.largestScn;
    }

    Set<String> getRolledBackTransactionIds() {
        return new HashSet<String>(this.rolledBackTransactionIds);
    }

    void resetLargestScn(Long value) {
        this.largestScn = value != null ? new BigDecimal(value) : BigDecimal.ZERO;
    }

    void registerCommitCallback(String transactionId, BigDecimal scn, Instant changeTime, CommitCallback callback) {
        if (this.abandonedTransactionIds.contains(transactionId)) {
            LogMinerHelper.logWarn(this.metrics, "Captured DML for abandoned transaction {}, ignored", transactionId);
            return;
        }
        if (this.rolledBackTransactionIds.contains(transactionId)) {
            LogMinerHelper.logWarn(this.metrics, "Captured DML for rolled-back transaction {}, ignored", transactionId);
            return;
        }
        this.transactions.computeIfAbsent(transactionId, s -> new Transaction(scn));
        this.metrics.setActiveTransactions(this.transactions.size());
        this.metrics.incrementRegisteredDmlCounter();
        this.metrics.calculateLagMetrics(changeTime);
        Transaction transaction = this.transactions.get(transactionId);
        if (transaction != null) {
            transaction.commitCallbacks.add(callback);
        }
        if (scn.compareTo(this.largestScn) > 0) {
            this.largestScn = scn;
        }
    }

    boolean commit(String transactionId, BigDecimal scn, OracleOffsetContext offsetContext, Timestamp timestamp, ChangeEventSource.ChangeEventSourceContext context, String debugMessage) {
        Transaction transaction = this.transactions.get(transactionId);
        if (transaction == null) {
            return false;
        }
        Instant start = Instant.now();
        this.calculateLargestScn();
        transaction = this.transactions.remove(transactionId);
        BigDecimal smallestScn = this.calculateSmallestScn();
        this.taskCounter.incrementAndGet();
        this.abandonedTransactionIds.remove(transactionId);
        if (offsetContext.getCommitScn() != null && offsetContext.getCommitScn() > scn.longValue() || this.lastCommittedScn.longValue() > scn.longValue()) {
            LogMinerHelper.logWarn(this.metrics, "Transaction {} was already processed, ignore. Committed SCN in offset is {}, commit SCN of the transaction is {}, last committed SCN is {}", transactionId, offsetContext.getCommitScn(), scn, this.lastCommittedScn);
            this.metrics.setActiveTransactions(this.transactions.size());
            return false;
        }
        List commitCallbacks = transaction.commitCallbacks;
        LOGGER.trace("COMMIT, {}, smallest SCN: {}, largest SCN {}", new Object[]{debugMessage, smallestScn, this.largestScn});
        this.executor.execute(() -> {
            try {
                int counter = commitCallbacks.size();
                for (CommitCallback callback : commitCallbacks) {
                    if (!context.isRunning()) {
                        return;
                    }
                    callback.execute(timestamp, smallestScn, scn, --counter);
                }
                this.lastCommittedScn = new BigDecimal(scn.longValue());
            }
            catch (InterruptedException e) {
                LogMinerHelper.logError(this.metrics, "Thread interrupted during running", e);
                Thread.currentThread().interrupt();
            }
            catch (Exception e) {
                this.errorHandler.setProducerThrowable(e);
            }
            finally {
                this.metrics.incrementCommittedTransactions();
                this.metrics.setActiveTransactions(this.transactions.size());
                this.metrics.incrementCommittedDmlCounter(commitCallbacks.size());
                this.metrics.setCommittedScn(scn.longValue());
                this.metrics.setOffsetScn(offsetContext.getScn());
                this.metrics.setCommitQueueCapacity(this.commitQueueCapacity.get());
                this.metrics.setLastCommitDuration(Duration.between(start, Instant.now()).toMillis());
                this.taskCounter.decrementAndGet();
            }
        });
        this.metrics.setCommitQueueCapacity(this.commitQueueCapacity.get());
        return true;
    }

    boolean rollback(String transactionId, String debugMessage) {
        Transaction transaction = this.transactions.get(transactionId);
        if (transaction != null) {
            LOGGER.debug("Transaction rolled back: {}", (Object)debugMessage);
            this.calculateLargestScn();
            this.transactions.remove(transactionId);
            this.abandonedTransactionIds.remove(transactionId);
            this.rolledBackTransactionIds.add(transactionId);
            this.metrics.setActiveTransactions(this.transactions.size());
            this.metrics.incrementRolledBackTransactions();
            this.metrics.addRolledBackTransactionId(transactionId);
            return true;
        }
        return false;
    }

    void abandonLongTransactions(Long thresholdScn) {
        BigDecimal threshold = new BigDecimal(thresholdScn);
        BigDecimal smallestScn = this.calculateSmallestScn();
        if (smallestScn == null) {
            return;
        }
        if (threshold.compareTo(smallestScn) < 0) {
            threshold = smallestScn;
        }
        Iterator<Map.Entry<String, Transaction>> iter = this.transactions.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, Transaction> transaction = iter.next();
            if (transaction.getValue().firstScn.compareTo(threshold) > 0) continue;
            LogMinerHelper.logWarn(this.metrics, "Following long running transaction {} will be abandoned and ignored: {} ", transaction.getKey(), transaction.getValue().toString());
            this.abandonedTransactionIds.add(transaction.getKey());
            iter.remove();
            this.calculateLargestScn();
            this.metrics.addAbandonedTransactionId(transaction.getKey());
            this.metrics.setActiveTransactions(this.transactions.size());
        }
    }

    boolean isTransactionRegistered(String txId) {
        return this.transactions.get(txId) != null;
    }

    private BigDecimal calculateSmallestScn() {
        BigDecimal scn = this.transactions.isEmpty() ? null : this.transactions.values().stream().map(transaction -> ((Transaction)transaction).firstScn).min(BigDecimal::compareTo).orElseThrow(() -> new DataException("Cannot calculate smallest SCN"));
        this.metrics.setOldestScn(scn == null ? -1L : scn.longValue());
        return scn;
    }

    private void calculateLargestScn() {
        this.largestScn = this.transactions.isEmpty() ? BigDecimal.ZERO : this.transactions.values().stream().map(transaction -> ((Transaction)transaction).lastScn).max(BigDecimal::compareTo).orElseThrow(() -> new DataException("Cannot calculate largest SCN"));
    }

    boolean isEmpty() {
        return this.transactions.isEmpty() && this.taskCounter.get() == 0;
    }

    public String toString() {
        StringBuilder result = new StringBuilder();
        this.transactions.values().forEach(t -> result.append(t.toString()));
        return result.toString();
    }

    void close() {
        this.transactions.clear();
        this.executor.shutdown();
        try {
            if (!this.executor.awaitTermination(1000L, TimeUnit.MILLISECONDS)) {
                this.executor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            LogMinerHelper.logError(this.metrics, "Thread interrupted during shutdown", e);
        }
    }

    @NotThreadSafe
    private static final class Transaction {
        private final BigDecimal firstScn;
        private BigDecimal lastScn;
        private final List<CommitCallback> commitCallbacks;

        private Transaction(BigDecimal firstScn) {
            this.firstScn = firstScn;
            this.commitCallbacks = new ArrayList<CommitCallback>();
            this.lastScn = firstScn;
        }

        public String toString() {
            return "Transaction{firstScn=" + this.firstScn + ", lastScn=" + this.lastScn + '}';
        }
    }

    public static interface CommitCallback {
        public void execute(Timestamp var1, BigDecimal var2, BigDecimal var3, int var4) throws InterruptedException;
    }
}

