/**
 * Copyright 2004 Fabrizio Giustina.
 *
 * Licensed under the Artistic License; you may not use this file
 * except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://maven-taglib.sourceforge.net/license.html
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package net.sf.maventaglib;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.CharArrayReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.lang.SystemUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xalan.processor.TransformerFactoryImpl;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;


/**
 * Base class for the maven taglib plugin.
 * @author Fabrizio Giustina
 * @version $Id: TaglibPlugin.java,v 1.13 2005/08/23 12:29:52 fgiust Exp $
 */
public abstract class TaglibPlugin
{

    /**
     * logger.
     */
    protected Log log = LogFactory.getLog(TaglibPlugin.class);

    /**
     * tld directory path.
     */
    protected String srcDir;

    /**
     * tld directory path.
     */
    protected String tldSrc;

    /**
     * output dir for converted tlds.
     */
    protected String outputDir;

    /**
     * converted tld file name.
     */
    protected String tldOut;

    /**
     * Instantiate a new Plugin and set up transformer factory.
     */
    public TaglibPlugin()
    {
        if (SystemUtils.JAVA_VERSION_INT >= 150)
        {
            System.setProperty(
                "javax.xml.transform.TransformerFactory",
                "com.sun.org.apache.xalan.internal.xsltc.trax.TransformerFactoryImpl");
        }
        else
        {
            System.setProperty(
                "javax.xml.transform.TransformerFactory",
                "org.apache.xalan.processor.TransformerFactoryImpl");
        }
    }

    /**
     * @param dir The outputDir to set.
     */
    public void setOutputDir(String dir)
    {
        this.outputDir = dir;
    }

    /**
     * @param file The tldOut to set.
     */
    public void setTldOut(String file)
    {
        this.tldOut = file;
    }

    /**
     * @param file The tldSrc to set.
     */
    public void setTldSrc(String file)
    {
        this.tldSrc = file;
    }

    /**
     * @param path The full path for the tld directory.
     */
    public void setSrcDir(String path)
    {
        this.srcDir = path;
    }

    /**
     * Executes the plugin goal.
     * @throws Exception any specific exception thrown by a plugin implementation.
     */
    public abstract void execute() throws Exception;

    /**
     * Transform a file using the given xsl stylesheet.
     * @param inputFile input file
     * @param stylesheet xslt used for transformation
     * @param outputFile output file
     * @throws Exception xml parsing/transforming exceptions
     */
    protected void applyXslt(File inputFile, String stylesheet, File outputFile) throws Exception
    {

        InputStream fis = null;
        try
        {
            fis = new BufferedInputStream(new FileInputStream(inputFile));
            XMLReader reader = getReader();
            Source src = new SAXSource(reader, new InputSource(fis));
            applyXslt(src, stylesheet, outputFile);
        }
        catch (TransformerException e)
        {
            echo("Unable to complete xslt transformation from "
                + inputFile.getAbsolutePath()
                + " to "
                + outputFile.getAbsolutePath()
                + " using "
                + stylesheet
                + ": "
                + e.getMessage());
        }
        finally
        {
            try
            {
                if (fis != null)
                {
                    fis.close();
                }
            }
            catch (IOException ignored)
            {
                // ignore
            }
        }
    }

    /**
     * Apply an xsl stylesheet to a java.xml.tranform.Source.
     * @param src Source
     * @param stylesheet xslt used for transformation
     * @param outputFile output file
     * @throws Exception xml parsing/transforming exceptions
     */
    protected void applyXslt(Source src, String stylesheet, File outputFile) throws Exception
    {
        OutputStream fos = null;
        outputFile.getParentFile().mkdirs();

        try
        {
            fos = new BufferedOutputStream(new FileOutputStream(outputFile));

            // result
            StreamResult res = new StreamResult(fos);

            // transformer
            InputStream xsl = TaglibPlugin.class.getResourceAsStream(stylesheet);
            if (xsl == null)
            {
                throw new Exception("Can't find stylesheet " + stylesheet);
            }
            Transformer transformer = getTransformerFactory().newTransformer(new StreamSource(xsl));
            // go
            transformer.transform(src, res);
        }
        finally
        {
            try
            {
                if (fos != null)
                {
                    fos.close();
                }
            }
            catch (IOException ignored)
            {
                // ignore
            }
        }

        log.info("applyXslt done");
    }

    /**
     * Returns a XMLReader instance.
     * @return XMLReader instance
     * @throws SAXException for errors on building the sax parser
     * @throws ParserConfigurationException if a SAX parser is not configured
     */
    protected XMLReader getReader() throws SAXException, ParserConfigurationException
    {
        SAXParserFactory factory = SAXParserFactory.newInstance();
        factory.setNamespaceAware(true);
        factory.setValidating(false);
        XMLReader reader = factory.newSAXParser().getXMLReader();
        reader.setEntityResolver(new EntityResolver()
        {

            public InputSource resolveEntity(String publicId, String systemId)
            {
                return new InputSource(new CharArrayReader(new char[0]));
            }
        });
        return reader;
    }

    /**
     * Returns a DocumentBuilder instance.
     * @return DocumentBuilder instance
     * @throws FactoryConfigurationError if the parser is not configured
     * @throws ParserConfigurationException if the parser is not configured
     */
    protected DocumentBuilder getDocumentBuilder() throws ParserConfigurationException, FactoryConfigurationError
    {
        DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();

        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setValidating(false);
        factory.setNamespaceAware(false);
        factory.setExpandEntityReferences(false);
        builder.setEntityResolver(new EntityResolver()
        {

            public InputSource resolveEntity(String publicId, String systemId)
            {
                return new InputSource(new CharArrayReader(new char[0]));
            }
        });

        return builder;
    }

    /**
     * Returns a transformer factory instance.
     * @return TransformerFactory instance
     */
    private TransformerFactory getTransformerFactory()
    {
        // this seems the only way to make maven and xalan work for java 1.3-1.4-1.5
        return new TransformerFactoryImpl();
    }

    /**
     * echo a message (using System.out).
     * @param message output text.
     */
    protected void echo(String message)
    {
        System.out//
            .println(message);
    }

}