/**
 * Copyright 2004 Fabrizio Giustina.
 *
 * Licensed under the Artistic License; you may not use this file
 * except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://maven-taglib.sourceforge.net/license.html
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package net.sf.maventaglib.checker;

import javax.servlet.jsp.tagext.TagExtraInfo;
import javax.servlet.jsp.tagext.TagSupport;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.tree.DefaultDocument;
import org.dom4j.tree.DefaultElement;


/**
 * Validates tag handler classes fount in tlds.
 * @author Fabrizio Giustina
 * @version $Revision $ ($Author $)
 */
public class TldChecker
{

    /**
     * logger.
     */
    private static Log log = LogFactory.getLog(TldChecker.class);

    /**
     * Check the given tld. Assure then:
     * <ul>
     * <li>Any tag class is loadable</li>
     * <li>the tag class has a setter for any of the declared attribute</li>
     * <li>the type declared in the dtd for an attribute (if any) matches the type accepted by the getter</li>
     * </ul>
     * @param tlds list of Tld to check.
     * @return Document containing validation results
     */
    public Document check(Tld[] tlds)
    {
        Document doc = new DefaultDocument();
        Element document = new DefaultElement("document");

        doc.add(document);

        for (int j = 0; j < tlds.length; j++)
        {
            Element tldNode = checkTld(tlds[j]);
            document.add(tldNode);
        }

        return doc;
    }

    /**
     * Checks a single tld and returns validation results.
     * @param tld Tld
     * @return tld element
     */
    private Element checkTld(Tld tld)
    {
        // new section for each tld
        Element section = new DefaultElement("tld");
        section.addAttribute("name", tld.getName());
        section.addAttribute("file", tld.getFilename());
        Tag[] tags = tld.getTags();
        if (tags != null)
        {
            for (int j = 0; j < tags.length; j++)
            {
                Element tagNode = checkTag(tags[j]);
                section.add(tagNode);
            }
        }

        return section;
    }

    /**
     * Checks a single tag and returns validation results.
     * @param tag Tag
     * @return tag element
     */
    private Element checkTag(Tag tag)
    {
        // new subsection for each tag
        Element subsection = new DefaultElement("tag");

        String className = tag.getTagClass();
        subsection.addAttribute("name", tag.getName());

        Element tagImpl = new DefaultElement("tagclass");
        tagImpl.addAttribute("name", className);

        subsection.add(tagImpl);
        Object tagObject = null;

        try
        {
            Class tagClass = Class.forName(className);

            if (!TagSupport.class.isAssignableFrom(tagClass))
            {
                tagImpl.addAttribute("extend", "error");
            }

            try
            {
                tagObject = tagClass.newInstance();
            }
            catch (Throwable e)
            {
                tagImpl.addAttribute("loadable", "error");
            }
        }
        catch (ClassNotFoundException e)
        {
            tagImpl.addAttribute("found", "error");
            tagImpl.addAttribute("extend", "error");
            tagImpl.addAttribute("loadable", "error");
        }

        if (tag.getTeiClass() != null)
        {
            Element teiImpl = checkTeiClass(tag.getTeiClass());
            subsection.add(teiImpl);
        }

        TagAttribute[] attributes = tag.getAttributes();

        if (tagObject != null && attributes.length > 0)
        {
            Element attributesNode = new DefaultElement("attributes");
            subsection.add(attributesNode);

            for (int j = 0; j < attributes.length; j++)
            {
                Element attributeNode = checkAttribute(tagObject, attributes[j]);
                attributesNode.add(attributeNode);
            }
        }

        return subsection;
    }

    /**
     * Check a declared TagExtraInfo class.
     * @param className TEI class name
     * @return teiclass Element
     */
    private Element checkTeiClass(String className)
    {
        Element teiImpl = new DefaultElement("teiclass");
        teiImpl.addAttribute("name", className);

        Class teiClass = null;
        try
        {
            teiClass = Class.forName(className);

            if (!TagExtraInfo.class.isAssignableFrom(teiClass))
            {
                teiImpl.addAttribute("extend", "error");
            }

            try
            {
                teiClass.newInstance();
            }
            catch (Throwable e)
            {
                teiImpl.addAttribute("loadable", "error");
            }
        }
        catch (ClassNotFoundException e)
        {
            teiImpl.addAttribute("found", "error");
            teiImpl.addAttribute("extend", "error");
            teiImpl.addAttribute("loadable", "error");
        }
        return teiImpl;
    }

    /**
     * Checks a single attribute and returns validation results.
     * @param tag tag handler instance
     * @param attribute TagAttribute
     * @return attribute element
     */
    private Element checkAttribute(Object tag, TagAttribute attribute)
    {
        String tldType = attribute.getAttributeType();
        String tldName = attribute.getAttributeName();

        Element attributeNode = new DefaultElement("attribute");
        attributeNode.addAttribute("tldname", tldName);
        attributeNode.addAttribute("tldtype", StringUtils.defaultString(tldType));

        if (!PropertyUtils.isWriteable(tag, tldName))
        {
            Element errorNode = new DefaultElement("error");
            errorNode.addAttribute("level", "error");
            errorNode.addText("Setter not found");
            attributeNode.add(errorNode);
            return attributeNode;
        }

        Class tagType = null;
        try
        {
            tagType = PropertyUtils.getPropertyType(tag, tldName);
        }
        catch (Throwable e)
        {
            // should never happen, since we already checked the writable property
        }

        if (tldType != null && tagType != null)
        {
            Class tldTypeClass = getClassFromName(tldType);

            if (!tagType.isAssignableFrom(tldTypeClass))
            {
                //out.write(this.errorTd);

                Element errorNode = new DefaultElement("error");
                errorNode.addAttribute("level", "error");
                errorNode.addText("Error in attribute type: tld declares ["
                    + tldType
                    + "], class declares ["
                    + tagType.getName()
                    + "]");
                attributeNode.add(errorNode);
            }

        }

        attributeNode.addAttribute("tagtype", tagType == null ? "" : tagType.getName());

        if (tldType != null && !tldType.equals(tagType.getName()))
        {
            Element errorNode = new DefaultElement("error");
            errorNode.addAttribute("level", "warning");
            errorNode.addText("Inexact match: tld declares ["
                + tldType
                + "], class declares ["
                + tagType.getName()
                + "]");
            attributeNode.add(errorNode);
        }
        else if (tldType == null && !java.lang.String.class.equals(tagType))
        {
            Element errorNode = new DefaultElement("error");
            errorNode.addAttribute("level", "info");
            errorNode.addText("Attribute type different from String and not declared in tld.");
            attributeNode.add(errorNode);
        }

        return attributeNode;

    }

    /**
     * returns a class from its name, handling primitives.
     * @param className clss name
     * @return Class istantiated using Class.forName or the matching primitive.
     */
    private Class getClassFromName(String className)
    {

        Class tldTypeClass = null;

        if ("int".equals(className))
        {
            tldTypeClass = int.class;
        }
        else if ("long".equals(className))
        {
            tldTypeClass = long.class;
        }
        else if ("double".equals(className))
        {
            tldTypeClass = double.class;
        }
        else if ("boolean".equals(className))
        {
            tldTypeClass = boolean.class;
        }
        else if ("char".equals(className))
        {
            tldTypeClass = char.class;
        }
        else if ("byte".equals(className))
        {
            tldTypeClass = byte.class;
        }

        if (tldTypeClass == null)
        {
            // not a primitive type
            try
            {
                tldTypeClass = Class.forName(className);
            }
            catch (ClassNotFoundException e)
            {
                log.error("unable to find class [" + className + "] declared in 'type' attribute");
            }
        }
        return tldTypeClass;
    }

}