/**
 * Copyright 2004 Fabrizio Giustina.
 *
 * Licensed under the Artistic License; you may not use this file
 * except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://maven-taglib.sourceforge.net/license.html
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package net.sf.maventaglib.checker;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * Reads a tld file and generates a Tld object.
 * @author Fabrizio Giustina
 * @version $Revision: 1.4 $ ($Author: fgiust $)
 */
public final class TldParser
{

    /**
     * logger.
     */
    private static Log log = LogFactory.getLog(TldParser.class);

    /**
     * don't instantiate.
     */
    private TldParser()
    {
        // unused
    }

    /**
     * Parse a Tld object from a document.
     * @param tldDoc Document for the parsed tld
     * @param tldName of the tld file
     * @return Tld instance
     */
    public static Tld parse(Document tldDoc, String tldName)
    {

        Tld tld = new Tld();
        tld.setFilename(tldName);
        List tags = new ArrayList();

        // @todo NPE
        NodeList tagList = tldDoc.getElementsByTagName("taglib").item(0).getChildNodes();

        for (int i = 0; i < tagList.getLength(); i++)
        {
            Node tagNode = tagList.item(i);

            if ("shortname".equals(tagNode.getNodeName()) || "short-name".equals(tagNode.getNodeName()))
            {
                Node child = tagNode.getFirstChild();
                if (child != null)
                {
                    tld.setName(child.getNodeValue());
                }
            }
            else if ("tag".equals(tagNode.getNodeName()))
            {
                Tag tag = parseTag(tagNode);
                tags.add(tag);
            }
            tld.setTags((Tag[]) tags.toArray(new Tag[tags.size()]));
        }

        return tld;
    }

    /**
     * Parse a <code>tag</code> element.
     * @param tagNode Node
     * @return a Tag instance
     */
    private static Tag parseTag(Node tagNode)
    {
        Tag tag = new Tag();
        List attributes = new ArrayList();
        NodeList tagAttributes = tagNode.getChildNodes();

        for (int k = 0; k < tagAttributes.getLength(); k++)
        {
            Node tagAttribute = tagAttributes.item(k);
            String nodeName = tagAttribute.getNodeName();

            if ("name".equals(nodeName))
            {
                // tag class name
                tag.setName(tagAttribute.getFirstChild().getNodeValue());
            }
            else if ("tag-class".equals(nodeName) || "tagclass".equals(nodeName))
            {
                // tag class name
                tag.setTagClass(tagAttribute.getFirstChild().getNodeValue());
            }
            else if ("tei-class".equals(nodeName) || "teiclass".equals(nodeName))
            {
                // tei class name
                tag.setTeiClass(tagAttribute.getFirstChild().getNodeValue());
            }
            else if ("attribute".equals(nodeName))
            {
                TagAttribute attribute = parseTagAttribute(tagAttribute);
                attributes.add(attribute);
            }
            tag.setAttributes((TagAttribute[]) attributes.toArray(new TagAttribute[attributes.size()]));
        }
        return tag;
    }

    /**
     * Parse an <code>attribute</code> element.
     * @param tagAttribute Node
     * @return TagAttribute instance
     */
    private static TagAttribute parseTagAttribute(Node tagAttribute)
    {
        TagAttribute attribute = new TagAttribute();

        NodeList attributeParams = tagAttribute.getChildNodes();
        for (int z = 0; z < attributeParams.getLength(); z++)
        {
            Node param = attributeParams.item(z);
            if (param.getNodeType() != Node.TEXT_NODE && param.hasChildNodes())
            {
                if ("name".equals(param.getNodeName()))
                {
                    attribute.setAttributeName(param.getFirstChild().getNodeValue());
                }
                else if ("type".equals(param.getNodeName()))
                {
                    attribute.setAttributeType(param.getFirstChild().getNodeValue());
                }
            }
        }
        if (log.isDebugEnabled())
        {
            log.debug(attribute);
        }
        return attribute;
    }

}