/**
 *  Copyright 2014 - 17 by Riccardo Massera (TheCoder4.Eu) and Stephan Rauh (http://www.beyondjava.net).
 *
 *  This file is part of BootsFaces.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.bootsfaces.component.breadcrumbs;

import java.io.IOException;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.render.FacesRenderer;
import net.bootsfaces.render.CoreRenderer;
import net.bootsfaces.render.Tooltip;

/**
 * This class generates the HTML code of &lt;b:breadcrumbs /&gt;.
 */
@FacesRenderer(componentFamily = "net.bootsfaces.component", rendererType = "net.bootsfaces.component.breadcrumbs.Breadcrumbs")
public class BreadcrumbsRenderer extends CoreRenderer {

	/**
	 * This methods generates the HTML code of the current b:breadcrumbs. <code>encodeBegin</code> generates the
	 * start of the component. After the, the JSF framework calls <code>encodeChildren()</code> to generate the HTML
	 * code between the beginning and the end of the component. For instance, in the case of a panel component the
	 * content of the panel is generated by <code>encodeChildren()</code>. After that, <code>encodeEnd()</code> is
	 * called to generate the rest of the HTML code.
	 *
	 * @param context the FacesContext.
	 * @param component the current b:breadcrumbs.
	 * @throws IOException thrown if something goes wrong when writing the HTML code.
	 */
	@Override
	public void encodeBegin(FacesContext context, UIComponent component) throws IOException {
		if (!component.isRendered()) {
			return;
		}
		Breadcrumbs breadcrumbs = (Breadcrumbs) component;
		ResponseWriter rw = context.getResponseWriter();
		String clientId = breadcrumbs.getClientId();

		rw.startElement("ol", breadcrumbs);

		rw.writeAttribute("id", clientId, "id");

		String style = breadcrumbs.getStyle();
		if (null != style && style.length() > 0) {
			rw.writeAttribute("style", style, "style");
		}

		Tooltip.generateTooltip(context, breadcrumbs, rw);

		String styleClass = breadcrumbs.getStyleClass();
		if (null == styleClass) {
			styleClass = "";
		}
		rw.writeAttribute("class", "breadcrumb ".concat(styleClass), "class");

		// Root node
		UIComponent root = component.getFacet("root");
		if (root != null) {
			rw.startElement("li", component);
			root.encodeAll(context);
			rw.endElement("li");
		}
	}

	/**
	 * This methods generates the HTML code of the current b:breadcrumbs. <code>encodeBegin</code> generates the
	 * start of the component. After the, the JSF framework calls <code>encodeChildren()</code> to generate the HTML
	 * code between the beginning and the end of the component. For instance, in the case of a panel component the
	 * content of the panel is generated by <code>encodeChildren()</code>. After that, <code>encodeEnd()</code> is
	 * called to generate the rest of the HTML code.
	 *
	 * @param context the FacesContext.
	 * @param component the current b:breadcrumbs.
	 * @throws IOException thrown if something goes wrong when writing the HTML code.
	 */
	@Override
	public void encodeEnd(FacesContext context, UIComponent component) throws IOException {
		if (!component.isRendered()) {
			return;
		}
		ResponseWriter rw = context.getResponseWriter();

		// End point / current page
		UIComponent end = component.getFacet("end");
		if (end != null) {
			rw.startElement("li", component);
			end.encodeAll(context);
			rw.endElement("li");
		}

		rw.endElement("ol");
		Tooltip.activateTooltips(context, component);
	}
}
