/**
 *  Copyright 2014-2017 Riccardo Massera (TheCoder4.Eu) and Stephan Rauh (http://www.beyondjava.net).
 *
 *  This file is part of BootsFaces.
 *
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
 */

package net.bootsfaces.component.buttonGroup;

import java.io.IOException;

import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.render.FacesRenderer;

import net.bootsfaces.render.CoreRenderer;
import net.bootsfaces.render.Responsive;
import net.bootsfaces.render.Tooltip;

/** This class generates the HTML code of &lt;b:buttonGroup /&gt;. */
@FacesRenderer(componentFamily = "net.bootsfaces.component", rendererType = "net.bootsfaces.component.buttonGroup.ButtonGroup")
public class ButtonGroupRenderer extends CoreRenderer {

	/**
	 * This methods generates the HTML code of the current b:buttonGroup.
	 * <code>encodeBegin</code> generates the start of the component. After the,
	 * the JSF framework calls <code>encodeChildren()</code> to generate the
	 * HTML code between the beginning and the end of the component. For
	 * instance, in the case of a panel component the content of the panel is
	 * generated by <code>encodeChildren()</code>. After that,
	 * <code>encodeEnd()</code> is called to generate the rest of the HTML code.
	 *
	 * @param context
	 *            the FacesContext.
	 * @param component
	 *            the current b:buttonGroup.
	 * @throws IOException
	 *             thrown if something goes wrong when writing the HTML code.
	 */
	@Override
	public void encodeBegin(FacesContext context, UIComponent component) throws IOException {
		if (!component.isRendered()) {
			return;
		}
		ButtonGroup buttonGroup = (ButtonGroup) component;
		ResponseWriter rw = context.getResponseWriter();
		String clientId = buttonGroup.getClientId();
		rw.startElement("div", buttonGroup);
		rw.writeAttribute("id", clientId, "id");

		String responsive = Responsive.getResponsiveStyleClass(buttonGroup, false).trim();
		if(responsive.length() > 0) {
			rw.writeAttribute("class", responsive, null);
			rw.startElement("div", buttonGroup);
		}


		StringBuilder styleClasses = new StringBuilder("btn-group");
		styleClasses.append("vertical".equals(buttonGroup.getOrientation()) ? "-vertical" : "");
		String s = buttonGroup.getSize();
		if (s != null) {
			styleClasses.append(" btn-group-").append(s);
		}
		String sc = buttonGroup.getStyleClass();
		if (null != sc) {
			styleClasses.append(" ").append(sc);
		}

		String pull = buttonGroup.getPull();
		if ("right".equals(pull) || "left".equals(pull)){
			styleClasses.append(" pull-").append(pull);
		}

		rw.writeAttribute("class", styleClasses.toString(), "class");

		if (null != buttonGroup.getStyle()) {
			rw.writeAttribute("style", buttonGroup.getStyle(), "style");
		}

		Tooltip.generateTooltip(context, buttonGroup, rw);
		beginDisabledFieldset(buttonGroup, rw);
	}

	/**
	 * This methods generates the HTML code of the current b:buttonGroup.
	 * <code>encodeBegin</code> generates the start of the component. After the,
	 * the JSF framework calls <code>encodeChildren()</code> to generate the
	 * HTML code between the beginning and the end of the component. For
	 * instance, in the case of a panel component the content of the panel is
	 * generated by <code>encodeChildren()</code>. After that,
	 * <code>encodeEnd()</code> is called to generate the rest of the HTML code.
	 *
	 * @param context
	 *            the FacesContext.
	 * @param component
	 *            the current b:buttonGroup.
	 * @throws IOException
	 *             thrown if something goes wrong when writing the HTML code.
	 */
	@Override
	public void encodeEnd(FacesContext context, UIComponent component) throws IOException {
		if (!component.isRendered()) {
			return;
		}
		ButtonGroup buttonGroup = (ButtonGroup) component;
		ResponseWriter rw = context.getResponseWriter();
		endDisabledFieldset(buttonGroup, rw);
		rw.endElement("div");
		String responsive = Responsive.getResponsiveStyleClass(buttonGroup, false).trim();
		if(responsive.length() > 0) {
			rw.endElement("div");
		}
		Tooltip.activateTooltips(context, buttonGroup);
	}
}
