/**
 *  Copyright 2014-2017 Riccardo Massera (TheCoder4.Eu) and Stephan Rauh (http://www.beyondjava.net).
 *  
 *  This file is part of BootsFaces.
 *  
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
 */

package net.bootsfaces.component.dropButton;

import java.io.IOException;

import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.render.FacesRenderer;

import net.bootsfaces.component.icon.IconRenderer;
import net.bootsfaces.component.navLink.AbstractNavLink;
import net.bootsfaces.render.CoreRenderer;
import net.bootsfaces.render.Responsive;
import net.bootsfaces.render.Tooltip;

/** This class generates the HTML code of &lt;b:dropButton /&gt;. */
@FacesRenderer(componentFamily = "net.bootsfaces.component", rendererType = "net.bootsfaces.component.dropButton.DropButton")
public class DropButtonRenderer extends CoreRenderer {

	/**
	 * This methods generates the HTML code of the current b:dropButton.
	 * <code>encodeBegin</code> generates the start of the component. After the, the
	 * JSF framework calls <code>encodeChildren()</code> to generate the HTML code
	 * between the beginning and the end of the component. For instance, in the case
	 * of a panel component the content of the panel is generated by
	 * <code>encodeChildren()</code>. After that, <code>encodeEnd()</code> is called
	 * to generate the rest of the HTML code.
	 * 
	 * @param context
	 *            the FacesContext.
	 * @param component
	 *            the current b:dropButton.
	 * @throws IOException
	 *             thrown if something goes wrong when writing the HTML code.
	 */
	@Override
	public void encodeBegin(FacesContext context, UIComponent component) throws IOException {
		if (!component.isRendered()) {
			return;
		}
		DropButton dropButton = (DropButton) component;
		ResponseWriter rw = context.getResponseWriter();

		String styleClass = getStyleclass(dropButton);
		String style = dropButton.getStyle() != null ? dropButton.getStyle() : null;

		rw.startElement("div", dropButton);
		String clientId = dropButton.getClientId(context);
		rw.writeAttribute("id", clientId, "id");
		rw.writeAttribute("name", clientId, "name");
		rw.writeAttribute("class", styleClass, "class");
		if (style != null)
			rw.writeAttribute("style", style, "style");
		Tooltip.generateTooltip(context, dropButton, rw);

		String ts = "btn btn-";
		String look = dropButton.getLook();
		if (look != null) {
			ts += look + " ";
		} else {
			ts += "default ";
		}
		String size = dropButton.getSize();
		if (size != null) {
			ts += "btn-" + size + " ";
		}

		String value = (String) dropButton.getAttributes().get("value");

		rw.startElement("button", dropButton);
		rw.writeAttribute("id", "dtL" + clientId, "id");
		rw.writeAttribute("class", ts + "dropdown-toggle", "class");
		rw.writeAttribute("type", "button", null);
		rw.writeAttribute("role", "button", null);
		rw.writeAttribute("data-toggle", "dropdown", null);
		String tabindex = dropButton.getTabindex();
		if (!"0".equals(tabindex)) {
			writeAttribute(rw, "tabindex", tabindex, null);
		}

		String icon = dropButton.getIcon();
		String faicon = dropButton.getIconAwesome();
		boolean fa = false; // flag to indicate whether the selected icon
							// set is Font Awesome or not.
		if (faicon != null) {
			icon = faicon;
			fa = true;
		}
		if (icon != null) {
			Object ialign = dropButton.getIconAlign();
			if (ialign != null && ialign.equals("right")) {
				rw.writeText(value + " ", null);
				IconRenderer.encodeIcon(rw, dropButton, icon, fa, dropButton.getIconSize(), dropButton.getIconRotate(),
						dropButton.getIconFlip(), dropButton.isIconSpin(), null, null, false, false, false, false,
						dropButton.isIconBrand(), dropButton.isIconInverse(), dropButton.isIconLight(),
						dropButton.isIconPulse(), dropButton.isIconRegular(), dropButton.isIconRegular());

				rw.writeText(" ", null);
			} else {
				IconRenderer.encodeIcon(rw, dropButton, icon, fa, dropButton.getIconSize(), dropButton.getIconRotate(),
						dropButton.getIconFlip(), dropButton.isIconSpin(), null, null, false, false, false, false,
						dropButton.isIconBrand(), dropButton.isIconInverse(), dropButton.isIconLight(),
						dropButton.isIconPulse(), dropButton.isIconRegular(), dropButton.isIconRegular());
				rw.writeText(" " + value, null);
			}
		} else {
			rw.writeText(value, null);
		}
		// Encode Caret
		{
			rw.startElement("b", dropButton);
			rw.writeAttribute("class", "caret", "class");
			rw.endElement("b");
		}
		rw.endElement("button");

		rw.startElement("ul", dropButton);
		rw.writeAttribute("class", "dropdown-menu", "class");
		rw.writeAttribute("role", "menu", null);
		rw.writeAttribute("aria-labelledby", "dtL" + clientId, null);
	}

	private String getStyleclass(DropButton dropButton) {
		String direction = getDirection(dropButton);
		String styleClass = "btn-group";
		if (direction.equals("up")) {
			styleClass += " drop" + direction;
		}
		if (dropButton.getStyleClass() != null)
			styleClass += (" " + dropButton.getStyleClass());

		styleClass += Responsive.getResponsiveStyleClass(dropButton, false);
		return styleClass;
	}

	private String getDirection(DropButton dropButton) {
		String direction = dropButton.getDrop();
		if (direction == null) {
			direction = "down";
		}
		if (!direction.equals("up") && !direction.equals("down")) {
			direction = "down";
		}
		return direction;
	}

	/**
	 * This methods generates the HTML code of the current b:dropButton.
	 * <code>encodeBegin</code> generates the start of the component. After the, the
	 * JSF framework calls <code>encodeChildren()</code> to generate the HTML code
	 * between the beginning and the end of the component. For instance, in the case
	 * of a panel component the content of the panel is generated by
	 * <code>encodeChildren()</code>. After that, <code>encodeEnd()</code> is called
	 * to generate the rest of the HTML code.
	 * 
	 * @param context
	 *            the FacesContext.
	 * @param component
	 *            the current b:dropButton.
	 * @throws IOException
	 *             thrown if something goes wrong when writing the HTML code.
	 */
	@Override
	public void encodeEnd(FacesContext context, UIComponent component) throws IOException {
		if (!component.isRendered()) {
			return;
		}
		DropButton dropButton = (DropButton) component;
		ResponseWriter rw = context.getResponseWriter();
		rw.endElement("ul");

		rw.endElement("div"); // btn-group
		Tooltip.activateTooltips(context, dropButton);
	}

}
