/*
 * Decompiled with CFR 0.152.
 */
package net.dataforte.cassandra.pool;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import net.dataforte.cassandra.pool.CassandraRing;
import net.dataforte.cassandra.pool.FairBlockingQueue;
import net.dataforte.cassandra.pool.PoolConfiguration;
import net.dataforte.cassandra.pool.PoolProperties;
import net.dataforte.cassandra.pool.PooledConnection;
import net.dataforte.cassandra.pool.jmx.ConnectionPoolMBean;
import org.apache.cassandra.thrift.Cassandra;
import org.apache.thrift.TException;
import org.apache.thrift.transport.TTransportException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionPool {
    public static final String POOL_JMX_PREFIX = "cassandra.pool";
    public static final String POOL_JMX_TYPE_PREFIX = "cassandra.pool:type=";
    private static final Logger log = LoggerFactory.getLogger(ConnectionPool.class);
    private AtomicInteger size = new AtomicInteger(0);
    private PoolConfiguration poolProperties;
    private BlockingQueue<PooledConnection> busy;
    private BlockingQueue<PooledConnection> idle;
    private Map<Cassandra.Client, PooledConnection> connectionMap;
    private volatile PoolMaintenance poolMaintenance;
    private volatile boolean closed = false;
    protected ConnectionPoolMBean jmxPool = null;
    private AtomicInteger waitcount = new AtomicInteger(0);
    private CassandraRing cassandraRing = null;

    public ConnectionPool(PoolConfiguration prop) throws TException {
        this.init(prop);
    }

    public Cassandra.Client getConnection() throws TException {
        PooledConnection con = this.borrowConnection(-1);
        return con.getConnection();
    }

    public String getName() {
        return this.getPoolProperties().getPoolName();
    }

    public int getWaitCount() {
        return this.waitcount.get();
    }

    public PoolConfiguration getPoolProperties() {
        return this.poolProperties;
    }

    public int getSize() {
        return this.size.get();
    }

    public int getActive() {
        return this.busy.size();
    }

    public int getIdle() {
        return this.idle.size();
    }

    public boolean isClosed() {
        return this.closed;
    }

    public void close() {
        this.close(false);
    }

    public void close(boolean force) {
        BlockingQueue<PooledConnection> pool;
        if (this.closed) {
            return;
        }
        this.closed = true;
        if (this.poolMaintenance != null) {
            this.poolMaintenance.stopRunning();
        }
        BlockingQueue<PooledConnection> blockingQueue = this.idle.size() > 0 ? this.idle : (pool = force ? this.busy : this.idle);
        while (pool.size() > 0) {
            try {
                PooledConnection con = pool.poll(1000L, TimeUnit.MILLISECONDS);
                while (con != null) {
                    if (pool == this.idle) {
                        this.release(con);
                    } else {
                        this.abandon(con);
                    }
                    con = pool.poll(1000L, TimeUnit.MILLISECONDS);
                }
            }
            catch (InterruptedException ex) {
                Thread.interrupted();
            }
            if (pool.size() != 0 || !force || pool == this.busy) continue;
            pool = this.busy;
        }
        if (this.getPoolProperties().isJmxEnabled()) {
            this.jmxPool = null;
        }
    }

    protected void init(PoolConfiguration properties) throws TException {
        this.poolProperties = properties;
        this.connectionMap = new HashMap<Cassandra.Client, PooledConnection>();
        this.cassandraRing = new CassandraRing(this.poolProperties.getConfiguredHosts());
        this.busy = new ArrayBlockingQueue<PooledConnection>(properties.getMaxActive(), false);
        this.idle = properties.isFairQueue() ? new FairBlockingQueue<PooledConnection>() : new ArrayBlockingQueue<PooledConnection>(properties.getMaxActive(), properties.isFairQueue());
        if (properties.isPoolSweeperEnabled()) {
            if (log.isDebugEnabled()) {
                log.debug("Starting pool maintenance thread");
            }
            this.poolMaintenance = new PoolMaintenance("[Pool-Maintenance]:" + properties.getName(), this, (long)properties.getTimeBetweenEvictionRunsMillis());
            this.poolMaintenance.start();
        }
        if (properties.getMaxActive() < properties.getInitialSize()) {
            log.warn("initialSize is larger than maxActive, setting initialSize to: " + properties.getMaxActive());
            properties.setInitialSize(properties.getMaxActive());
        }
        if (properties.getMinIdle() > properties.getMaxActive()) {
            log.warn("minIdle is larger than maxActive, setting minIdle to: " + properties.getMaxActive());
            properties.setMinIdle(properties.getMaxActive());
        }
        if (properties.getMaxIdle() > properties.getMaxActive()) {
            log.warn("maxIdle is larger than maxActive, setting maxIdle to: " + properties.getMaxActive());
            properties.setMaxIdle(properties.getMaxActive());
        }
        if (properties.getMaxIdle() < properties.getMinIdle()) {
            log.warn("maxIdle is smaller than minIdle, setting maxIdle to: " + properties.getMinIdle());
            properties.setMaxIdle(properties.getMinIdle());
        }
        if (this.getPoolProperties().isJmxEnabled()) {
            if (log.isDebugEnabled()) {
                log.debug("Creating JMX MBean");
            }
            this.createMBean();
        }
        PooledConnection[] initialPool = new PooledConnection[this.poolProperties.getInitialSize()];
        try {
            for (int i = 0; i < initialPool.length; ++i) {
                initialPool[i] = this.borrowConnection(0);
            }
        }
        catch (Exception x) {
            if (this.jmxPool != null) {
                this.jmxPool.notify("INIT FAILED", ConnectionPool.getStackTrace(x));
            }
            this.close(true);
            throw new TException((Throwable)x);
        }
        finally {
            for (int i = 0; i < initialPool.length; ++i) {
                if (initialPool[i] == null) continue;
                try {
                    this.returnConnection(initialPool[i]);
                    continue;
                }
                catch (Exception x) {}
            }
        }
        this.closed = false;
        if (log.isInfoEnabled()) {
            log.info("ConnectionPool initialized.");
        }
        if (log.isTraceEnabled()) {
            for (String p : PoolProperties.getPropertyNames()) {
                log.trace("[" + this.getName() + "] ConnectionPool: " + p + "=" + this.poolProperties.get(p));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void abandon(PooledConnection con) {
        if (con == null) {
            return;
        }
        try {
            con.lock();
            String trace = con.getStackTrace();
            if (this.getPoolProperties().isLogAbandoned()) {
                log.warn("[" + this.getName() + "] Connection has been abandoned " + con + ":" + trace);
            }
            if (this.jmxPool != null) {
                this.jmxPool.notify("CONNECTION ABANDONED", trace);
            }
            this.release(con);
            if (this.waitcount.get() > 0) {
                this.idle.offer(new PooledConnection(this.poolProperties, this));
            }
        }
        finally {
            con.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void suspect(PooledConnection con) {
        if (con == null) {
            return;
        }
        if (con.isSuspect()) {
            return;
        }
        try {
            con.lock();
            String trace = con.getStackTrace();
            if (this.getPoolProperties().isLogAbandoned()) {
                log.warn("[" + this.getName() + "] Connection has been marked suspect, possibly abandoned " + con + "[" + (System.currentTimeMillis() - con.getTimestamp()) + " ms.]:" + trace);
            }
            if (this.jmxPool != null) {
                this.jmxPool.notify("SUSPECT CONNECTION ABANDONED", trace);
            }
            con.setSuspect(true);
        }
        finally {
            con.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void release(PooledConnection con) {
        if (con == null) {
            return;
        }
        try {
            con.lock();
            if (con.release()) {
                this.size.addAndGet(-1);
            }
        }
        finally {
            con.unlock();
        }
    }

    private PooledConnection borrowConnection(int wait) throws TException {
        long maxWait;
        if (this.isClosed()) {
            throw new TException("[" + this.getName() + "] Connection pool closed.");
        }
        long now = System.currentTimeMillis();
        PooledConnection con = (PooledConnection)this.idle.poll();
        do {
            PooledConnection result;
            if (con != null && (result = this.borrowConnection(now, con)) != null) {
                return result;
            }
            if (this.size.get() < this.getPoolProperties().getMaxActive()) {
                if (this.size.addAndGet(1) > this.getPoolProperties().getMaxActive()) {
                    this.size.decrementAndGet();
                } else {
                    return this.createConnection(now, con);
                }
            }
            maxWait = wait;
            if (wait == -1) {
                maxWait = this.getPoolProperties().getMaxWait() <= 0 ? Long.MAX_VALUE : (long)this.getPoolProperties().getMaxWait();
            }
            long timetowait = Math.max(0L, maxWait - (System.currentTimeMillis() - now));
            this.waitcount.incrementAndGet();
            try {
                con = this.idle.poll(timetowait, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException ex) {
                Thread.interrupted();
                TException sx = new TException("[" + this.getName() + "] Pool wait interrupted.");
                sx.initCause((Throwable)ex);
                throw sx;
            }
            finally {
                this.waitcount.decrementAndGet();
            }
            if (maxWait != 0L || con != null) continue;
            throw new TException("[" + this.getName() + "] NoWait: Pool empty. Unable to fetch a connection, none available[" + this.busy.size() + " in use].");
        } while (con != null || System.currentTimeMillis() - now < maxWait);
        if (log.isDebugEnabled()) {
            int counter = 0;
            for (PooledConnection connection : this.busy) {
                log.debug("[" + this.getName() + "] Busy connection " + counter + " borrowed at " + connection.getStackTrace());
                ++counter;
            }
        }
        throw new TException("[" + this.getName() + "] Timeout: Pool empty. Unable to fetch a connection in " + maxWait / 1000L + " seconds, none available[" + this.busy.size() + " in use].");
    }

    protected PooledConnection createConnection(long now, PooledConnection con) throws TException {
        boolean error = false;
        try {
            con = this.create();
            con.lock();
            con.connect();
            if (con.validate(4)) {
                this.connectionMap.put(con.getConnection(), con);
                con.setTimestamp(now);
                if (this.getPoolProperties().isLogAbandoned()) {
                    con.setStackTrace(ConnectionPool.getThreadDump());
                }
                if (!this.busy.offer(con)) {
                    log.debug("[" + this.getName() + "] Connection doesn't fit into busy array, connection will not be traceable.");
                }
                PooledConnection pooledConnection = con;
                return pooledConnection;
            }
            error = true;
        }
        catch (Exception e) {
            error = true;
            if (log.isDebugEnabled()) {
                log.debug("[" + this.getName() + "] Unable to create a new Cassandra connection.", (Throwable)e);
            }
            if (e instanceof TException) {
                throw (TException)e;
            }
            TException ex = new TException(e.getMessage());
            ex.initCause((Throwable)e);
            throw ex;
        }
        finally {
            if (error) {
                this.release(con);
            }
            con.unlock();
        }
        return null;
    }

    protected PooledConnection borrowConnection(long now, PooledConnection con) throws TException {
        boolean setToNull = false;
        try {
            con.lock();
            if (con.isReleased()) {
                PooledConnection pooledConnection = null;
                return pooledConnection;
            }
            if (!con.isDiscarded() && !con.isInitialized()) {
                con.connect();
            }
            if (!con.isDiscarded() && con.validate(1)) {
                con.setTimestamp(now);
                if (this.getPoolProperties().isLogAbandoned()) {
                    con.setStackTrace(ConnectionPool.getThreadDump());
                }
                if (!this.busy.offer(con)) {
                    log.debug("[" + this.getName() + "] Connection doesn't fit into busy array, connection will not be traceable.");
                }
                PooledConnection pooledConnection = con;
                return pooledConnection;
            }
            con.reconnect();
            if (con.validate(4)) {
                con.setTimestamp(now);
                if (this.getPoolProperties().isLogAbandoned()) {
                    con.setStackTrace(ConnectionPool.getThreadDump());
                }
                if (!this.busy.offer(con)) {
                    log.debug("[" + this.getName() + "] Connection doesn't fit into busy array, connection will not be traceable.");
                }
                PooledConnection pooledConnection = con;
                return pooledConnection;
            }
            try {
                this.release(con);
                setToNull = true;
                throw new TException("[" + this.getName() + "] Failed to validate a newly established connection.");
            }
            catch (Exception x) {
                this.release(con);
                setToNull = true;
                if (x instanceof TException) {
                    throw (TException)x;
                }
                TException ex = new TException(x.getMessage());
                ex.initCause((Throwable)x);
                throw ex;
            }
        }
        finally {
            con.unlock();
            if (setToNull) {
                con = null;
            }
        }
    }

    protected boolean shouldClose(PooledConnection con, int action) {
        if (con.isDiscarded()) {
            return true;
        }
        if (this.isClosed()) {
            return true;
        }
        if (!con.validate(action)) {
            return true;
        }
        if (this.getPoolProperties().getMaxAge() > 0L) {
            return System.currentTimeMillis() - con.getLastConnected() > this.getPoolProperties().getMaxAge();
        }
        return false;
    }

    public void release(Cassandra.Client connection) {
        PooledConnection pooledConnection = this.connectionMap.get(connection);
        this.returnConnection(pooledConnection);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void returnConnection(PooledConnection con) {
        if (this.isClosed()) {
            this.release(con);
            return;
        }
        if (con != null) {
            try {
                con.lock();
                if (this.busy.remove(con)) {
                    if (!this.shouldClose(con, 2)) {
                        con.setStackTrace(null);
                        con.setTimestamp(System.currentTimeMillis());
                        if (this.idle.size() >= this.poolProperties.getMaxIdle() && !this.poolProperties.isPoolSweeperEnabled() || !this.idle.offer(con)) {
                            if (log.isDebugEnabled()) {
                                log.debug("[" + this.getName() + "] Connection [" + con + "] will be closed and not returned to the pool, idle[" + this.idle.size() + "]>=maxIdle[" + this.poolProperties.getMaxIdle() + "] idle.offer failed.");
                            }
                            this.release(con);
                        }
                    } else {
                        if (log.isDebugEnabled()) {
                            log.debug("[" + this.getName() + "] Connection [" + con + "] will be closed and not returned to the pool.");
                        }
                        this.release(con);
                    }
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("[" + this.getName() + "] Connection [" + con + "] will be closed and not returned to the pool, busy.remove failed.");
                    }
                    this.release(con);
                }
            }
            finally {
                con.unlock();
            }
        }
    }

    protected boolean shouldAbandon() {
        float perc;
        float max;
        if (this.poolProperties.getAbandonWhenPercentageFull() == 0) {
            return true;
        }
        float used = this.busy.size();
        return used / (max = (float)this.poolProperties.getMaxActive()) * 100.0f >= (perc = (float)this.poolProperties.getAbandonWhenPercentageFull());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void checkAbandoned() {
        if (log.isTraceEnabled()) {
            log.trace("[" + this.getName() + "] checking for abandoned connections");
        }
        try {
            if (this.busy.size() == 0) {
                return;
            }
            Iterator locked = this.busy.iterator();
            int sto = this.getPoolProperties().getSuspectTimeout();
            while (locked.hasNext()) {
                PooledConnection con = (PooledConnection)locked.next();
                boolean setToNull = false;
                try {
                    con.lock();
                    if (this.idle.contains(con)) continue;
                    long time = con.getTimestamp();
                    long now = System.currentTimeMillis();
                    if (this.shouldAbandon() && now - time > con.getAbandonTimeout()) {
                        this.busy.remove(con);
                        this.abandon(con);
                        setToNull = true;
                        continue;
                    }
                    if (sto <= 0 || now - time <= (long)(sto * 1000)) continue;
                    this.suspect(con);
                }
                finally {
                    con.unlock();
                    if (!setToNull) continue;
                    con = null;
                }
            }
        }
        catch (ConcurrentModificationException e) {
            log.debug("checkAbandoned failed.", (Throwable)e);
        }
        catch (Exception e) {
            log.warn("checkAbandoned failed, it will be retried.", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void checkIdle() {
        try {
            if (this.idle.size() == 0) {
                return;
            }
            long now = System.currentTimeMillis();
            Iterator unlocked = this.idle.iterator();
            while (this.idle.size() >= this.getPoolProperties().getMinIdle() && unlocked.hasNext()) {
                PooledConnection con = (PooledConnection)unlocked.next();
                boolean setToNull = false;
                try {
                    con.lock();
                    if (this.busy.contains(con)) continue;
                    long time = con.getTimestamp();
                    if (con.getReleaseTime() <= 0L || now - time <= con.getReleaseTime() || this.getSize() <= this.getPoolProperties().getMinIdle()) continue;
                    if (log.isDebugEnabled()) {
                        log.debug("[" + this.getName() + "] Releasing idle connection " + con);
                    }
                    this.release(con);
                    this.idle.remove(con);
                    setToNull = true;
                }
                finally {
                    con.unlock();
                    if (!setToNull) continue;
                    con = null;
                }
            }
        }
        catch (ConcurrentModificationException e) {
            log.debug("[" + this.getName() + "] checkIdle failed.", (Throwable)e);
        }
        catch (Exception e) {
            log.warn("[" + this.getName() + "] checkIdle failed, it will be retried.", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testAllIdle() {
        try {
            if (this.idle.size() == 0) {
                return;
            }
            for (PooledConnection con : this.idle) {
                try {
                    con.lock();
                    if (this.busy.contains(con) || con.validate(3)) continue;
                    this.idle.remove(con);
                    this.release(con);
                }
                finally {
                    con.unlock();
                }
            }
        }
        catch (ConcurrentModificationException e) {
            log.debug("[" + this.getName() + "] testAllIdle failed.", (Throwable)e);
        }
        catch (Exception e) {
            log.warn("[" + this.getName() + "] testAllIdle failed, it will be retried.", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void refreshRing() {
        try {
            if (this.idle.size() == 0) {
                return;
            }
            for (PooledConnection con : this.idle) {
                try {
                    con.lock();
                    if (this.busy.contains(con)) continue;
                    this.cassandraRing.refresh((Cassandra.Iface)con.getConnection());
                    log.debug("[" + this.getName() + "] refreshRing success, ring = " + this.cassandraRing);
                    return;
                }
                catch (TTransportException t) {
                    log.warn("[" + this.getName() + "] removing connection to non-responding host ");
                    this.idle.remove(con);
                    this.release(con);
                }
                finally {
                    con.unlock();
                }
            }
        }
        catch (ConcurrentModificationException e) {
            log.debug("[" + this.getName() + "] refreshRing failed.", (Throwable)e);
        }
        catch (Exception e) {
            log.warn("[" + this.getName() + "] refreshRing failed, it will be retried.", (Throwable)e);
        }
    }

    protected static String getThreadDump() {
        Exception x = new Exception();
        x.fillInStackTrace();
        return ConnectionPool.getStackTrace(x);
    }

    public static String getStackTrace(Throwable x) {
        if (x == null) {
            return null;
        }
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        PrintStream writer = new PrintStream(bout);
        x.printStackTrace(writer);
        String result = bout.toString();
        return x.getMessage() != null && x.getMessage().length() > 0 ? x.getMessage() + ";" + result : result;
    }

    protected PooledConnection create() {
        PooledConnection con = new PooledConnection(this.getPoolProperties(), this);
        return con;
    }

    protected void finalize(PooledConnection con) {
    }

    protected void disconnectEvent(PooledConnection con, boolean finalizing) {
        this.connectionMap.remove(con.getConnection());
    }

    public ConnectionPoolMBean getJmxPool() {
        return this.jmxPool;
    }

    public CassandraRing getCassandraRing() {
        return this.cassandraRing;
    }

    protected void createMBean() {
        try {
            this.jmxPool = new ConnectionPoolMBean(this);
        }
        catch (Exception x) {
            log.warn("Unable to start JMX integration for connection pool. Instance[" + this.getName() + "] can't be monitored.", (Throwable)x);
        }
    }

    protected class PoolMaintenance
    extends Thread {
        protected ConnectionPool pool;
        protected long sleepTime;
        protected volatile boolean run;

        PoolMaintenance(String name, ConnectionPool pool, long sleepTime) {
            super(name);
            this.run = true;
            this.setDaemon(true);
            this.pool = pool;
            this.sleepTime = sleepTime;
            if (sleepTime <= 0L) {
                log.warn("[" + this.getName() + "] Database connection pool maintenance thread interval is set to 0, defaulting to 30 seconds");
                this.sleepTime = 30000L;
            } else if (sleepTime < 1000L) {
                log.warn("[" + this.getName() + "] Database connection pool maintenance thread interval is set to lower than 1 second.");
            }
        }

        public void run() {
            while (this.run) {
                try {
                    PoolMaintenance.sleep(this.sleepTime);
                }
                catch (InterruptedException e) {
                    Thread.interrupted();
                    continue;
                }
                if (this.pool.isClosed()) {
                    if (this.pool.getSize() > 0) continue;
                    this.run = false;
                    continue;
                }
                try {
                    if (this.pool.getPoolProperties().isRemoveAbandoned()) {
                        this.pool.checkAbandoned();
                    }
                    if (this.pool.getPoolProperties().getMinIdle() < this.pool.idle.size()) {
                        this.pool.checkIdle();
                    }
                    if (this.pool.getPoolProperties().isTestWhileIdle()) {
                        this.pool.testAllIdle();
                    }
                    if (!this.pool.getPoolProperties().isAutomaticHostDiscovery()) continue;
                    this.pool.refreshRing();
                }
                catch (Exception x) {
                    log.error("", (Throwable)x);
                }
            }
        }

        public void stopRunning() {
            this.run = false;
            this.interrupt();
        }
    }
}

