#set( $symbol_pound = '#' )
#set( $symbol_dollar = '$' )
#set( $symbol_escape = '\' )
package ${package}.${artifactId};

import java.io.File;
import static java.lang.System.*;
import java.util.*;
import javax.annotation.concurrent.Immutable;
import javax.swing.*;
import net.java.truelicense.core.*;
import net.java.truelicense.swing.LicenseWizard;

/**
 * Installs, views, verifies or uninstalls license keys for ${subject}.
 * A real application should do something useful besides managing license keys.
 *
 * @author Christian Schlichtherle
 */
@Immutable
public enum Main {

    USAGE {
        @Override protected void run(final Deque<String> params) {
            throw new IllegalArgumentException();
        }
    },

    HELP {
        @Override protected void run(final Deque<String> params) {
            out.printf(valueOf(params.pop().toUpperCase(Locale.ROOT)).help());
        }
    },

    VERSION {
        @Override protected void run(final Deque<String> params) {
            out.printf(message("version"), Main.class.getSimpleName());
        }
    },

    INSTALL {
        @Override protected void run(Deque<String> params) throws Exception {
            out.println(manager.install(context.fileStore(new File(params.pop()))));
        }
    },

    VERIFY {
        @Override protected void run(Deque<String> params) throws Exception {
            manager.verify();
        }
    },

    VIEW {
        @Override protected void run(Deque<String> params) throws Exception {
            out.println(manager.view());
        }
    },

    UNINSTALL {
        @Override protected void run(Deque<String> params) throws Exception {
            manager.uninstall();
        }
    },

    WIZARD {
        @Override protected void run(Deque<String> params) throws Exception {
            SwingUtilities.invokeAndWait(new Runnable() {
                @Override public void run() {
                    try {
                        UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
                    } catch (Exception ex) {
                        throw new AssertionError(ex);
                    }
                    final LicenseWizard wizard = new LicenseWizard(manager);
                    wizard.setUninstallButtonVisible(true);
                    System.exit(wizard.showModalDialog());
                }
            });
        }
    };

    final LicenseConsumerManager manager = LicensingSchema.manager();
    final LicenseConsumerContext context = manager.context();

    /**
     * Runs this command.
     * Implementations are free to modify the given deque.
     *
     * @param params the command parameters.
     */
    // This method needs to be protected or otherwise ProGuard 4.8 may generate
    // invalid code when --repackageclasses is set, even if
    // -allowaccessmodification is set, too.
    // See https://sourceforge.net/p/proguard/bugs/367/ for details.
    protected abstract void run(Deque<String> params) throws Exception;

    public static void main(String... args) {
        System.exit(processAndHandleExceptions(args));
    }

    @SuppressWarnings("CallToThreadDumpStack")
    static int processAndHandleExceptions(final String... args) {
        int status;
        try {
            process(args);
            status = 0;
        } catch (final IllegalArgumentException ex) {
            printUsage();
            status = 1;
        } catch (final NoSuchElementException ex) {
            printUsage();
            status = 1;
        } catch (final Throwable ex) {
            ex.printStackTrace();
            status = 2;
        }
        return status;
    }

    static void process(final String... args) throws Exception {
        final Deque<String> params = new LinkedList<String>(Arrays.asList(args));
        final String command = upperCase(params.pop());
        valueOf(command).run(params);
    }

    private static String upperCase(String s) {
        return s.toUpperCase(Locale.ROOT);
    }

    private static void printUsage() {
        final StringBuilder builder = new StringBuilder(25 * 80);
        for (final Main main : values()) builder.append(main.usage());
        err.println(builder.toString());
    }

    String usage() {
        return String.format(message("usage"), Main.class.getSimpleName());
    }

    String help() { return message("help"); }

    String message(String key) {
        return ResourceBundle
                .getBundle(Main.class.getName())
                .getString(name() + "." + key);
    }
}
