package net.objectlab.qalab.m2;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;

import net.objectlab.qalab.interfaces.QALabExporter;
import net.objectlab.qalab.parser.StatMerger;
import net.objectlab.qalab.util.TaskLogger;
import net.objectlab.qalab.m2.util.Utils;
import net.objectlab.qalab.m2.util.Maven2TaskLogger;

import org.xml.sax.InputSource;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.FileNotFoundException;

import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

/**
 * Goal that handles the merge of statistics into qalab.xml.
 *
 * @author <a href="http://www.davesag.com">Dave Sag</a>.
 * @goal merge
 * @phase deploy
 */
public class BuildStatMergeMojo extends AbstractMojo {

    // ~ Instance fields -------------------------------------------------------
    /**
     * The file the whose values will be incorporated in the qalab.xml.
     * This file will have been generated by checkstyle, or pmd etc.
     *
     * @parameter
     * @required
     */
    private File inputFile = null;

    /**
     * The merged properties file. By default this is generated into the
     * root of the project as it is often checked into source control.
     *
     * @parameter expression="${project.basedir}/qalab.xml"
     */
    private File outputFile = null;

    /**
     * If true then any debug logging output will be suppressed.
     * @parameter default-value=false
     */
    private boolean quiet = false;

    /**
     * The directory where the source code is.
     * @parameter expression="${project.basedir}/src/"
     */
    private String srcDir;

    /**
     * The fully qualified class name for the handler for the given statistics.
     *<ul>The built-in handlers are:
     *<li>net.objectlab.qalab.parser.CheckstyleStatMerge</li>
     *<li>net.objectlab.qalab.parser.PMDStatMerge</li>
     *<li>net.objectlab.qalab.parser.FindBugsStatMerge</li>
     *<li>net.objectlab.qalab.parser.SimianStatMerge</li>
     *</ul>
     * @parameter
     * @required
     */
    private String handler;

    /**
     * The timestamp for the stats.
     * @parameter
     */
    private String mergerTimeStamp;

    /**
     * An exporter class name.
     * @parameter default-value="net.objectlab.qalab.exporter.QALabXMLExporter"
     */
    private String exporterClassName =
        "net.objectlab.qalab.exporter.QALabXMLExporter";

    /**
     * The properties file to use instead of setting them one by one.
     * @parameter
     */
    private File propertiesFile;

    /**
     * the loaded properties from the properties file declared above.
     */
    private Properties theProperties = null;

    // ~ Methods

    /**
     * Method invoked by the ant framework to execute the action associated
     * with this task.  This will validate the input parameters then merge
     * the statistics.
     * @throws MojoExecutionException if anything goes wrong.
     */
    public final void execute() throws MojoExecutionException {

        theProperties = createOverridingProperties();

        // validate the provided parameters
        validate();

        getLog().info("Starting QALab Merge");

        // Display the files being processed
        if (!quiet) {
            getLog().info("inputFile='" + inputFile.getPath()
                    + "', outputFile='"
                    + outputFile.getPath());
            getLog().info("srcDir='" + srcDir + "', mergerTimeStamp="
                    + mergerTimeStamp);
            String proppath = null;
            if (propertiesFile != null) {
                proppath = propertiesFile.getPath();
            }
            getLog().info("quiet='" + quiet + "', propertiesFile='"
                    + proppath + "'.");
            for (final Iterator it = theProperties.entrySet().iterator();
                    it.hasNext();) {

                final Map.Entry entry = (Map.Entry) it.next();
                final String key = entry.getKey().toString();
                if (key.indexOf("qalab") >= 0) {
                    getLog().info(key + " = '" + entry.getValue() + "'");
                }
            }
        }

        mergeFiles();
    }

    /**
     * Merge the statistics into the qalab.xml.
     * @throws MojoExecutionException if anything goes wrong.
     */
    private void mergeFiles() throws MojoExecutionException {
        try {
            final TaskLogger logger = new Maven2TaskLogger(this);

            // create the exporter
            final QALabExporter exporter = (QALabExporter)
                Class.forName(exporterClassName).newInstance();

            theProperties.setProperty("qalab.merge.output.file",
                outputFile.getAbsolutePath());

            exporter.setQuiet(quiet);
            exporter.setTaskLogger(logger);

            final StatMerger merger = (StatMerger) Class.forName(handler)
                                                  .newInstance();

            merger.setQuiet(quiet);
            merger.setSrcDir(srcDir);
            merger.setTaskLogger(logger);
            final String tsOnly =
                    theProperties.getProperty("qalab.merge.timestampdateonly");
            merger.setMergerTimeStamp(
                    mergerTimeStamp, tsOnly != null
                    && (tsOnly.equalsIgnoreCase("yes")
                            || tsOnly.equalsIgnoreCase("true"))
                            );

            theProperties.setProperty("qalab.merge.output.timestamp",
                    merger.getMergerTimeStamp());
            theProperties.setProperty("qalab.merge.type",
                    merger.getType());
            exporter.configure(theProperties);

            merger.mergeStats(
                    new InputSource(
                            new FileInputStream(inputFile)), exporter);
            getLog().info("Files: "
                + merger.getFileCount()
                + " Violations:"
                + merger.getTotalViolations());

            exporter.save();
        } catch (IllegalAccessException e) {
            throw new MojoExecutionException(e.toString());
        } catch (FileNotFoundException e) {
            throw new MojoExecutionException(e.toString());
        } catch (IOException e) {
            throw new MojoExecutionException(e.toString());
        } catch (ClassNotFoundException e) {
            throw new MojoExecutionException(e.toString());
        } catch (InstantiationException e) {
            throw new MojoExecutionException(e.toString());
        }
    }

    /**
     * Validates the parameters supplied by maven 2.
     * @throws MojoExecutionException if anything goes wrong.
     */
    private void validate() throws MojoExecutionException {
        try {
            Utils.checkFile(inputFile, "inputFile");
        } catch (IOException ioex) {
            throw new MojoExecutionException(ioex.getMessage(), ioex);
        }
    }

    /**
     * Create the Properties object based on the arguments specified
     * to the Mojo in your <code>pom.xml</code> file.
     * @return the properties for property expansion expansion
     * @throws MojoExecutionException if anything goes wrong.
     */
    private Properties createOverridingProperties()
            throws MojoExecutionException {

        final Properties result = new Properties();

        // Load the properties file if specified
        if (propertiesFile != null && propertiesFile.canRead()
                && propertiesFile.isFile()) {
            FileInputStream inStream = null;
            if (!quiet) {
                getLog().debug("loading " + propertiesFile);
            }

            try {
                inStream = new FileInputStream(propertiesFile);
                result.load(inStream);
            } catch (FileNotFoundException fnfex) {
                throw new MojoExecutionException(
                        "Could not find Properties file '"
                        + propertiesFile + "'", fnfex);
            } catch (IOException ioex) {
                throw new MojoExecutionException(
                        "Error loading Properties file '"
                        + propertiesFile + "'", ioex);
            } finally {
                try {
                    if (inStream != null) {
                        inStream.close();
                    }
                } catch (IOException ioex) {
                    throw new MojoExecutionException(
                            "Error closing Properties file '"
                            + propertiesFile + "'", ioex);
                }
            }
        }

        // override with Maven properties as apropriate.
        final Map projectContext = getPluginContext();

        for (final Iterator it = projectContext.entrySet().iterator();
                it.hasNext();) {

            final Map.Entry entry = (Map.Entry) it.next();
            final String value = String.valueOf(entry.getValue());
            if (entry.getKey().toString().indexOf("qalab") >= 0) {
                getLog().info("Adding " + entry.getKey() + " / " + value);
            }
            result.put(entry.getKey(), value);
        }

        return result;
    }
}
