package net.objectlab.qalab.m2;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.FileNotFoundException;

import net.objectlab.qalab.parser.BuildStatForChartParser;
import net.objectlab.qalab.m2.util.Utils;
import net.objectlab.qalab.m2.util.Maven2TaskLogger;

/**
 * Goal which generates the QALab BuildStat chart.
 *
 * @author <a href="http://www.davesag.com">Dave Sag</a>.
 * @goal chart
 * @phase deploy
 */
public class BuildStatChartMojo extends AbstractMojo {

    // ~ Static fields/initializers
    // ------------------------------------------------------------------------

    /** Default SAX parser. **/
    private static final String DEFAULT_PARSER_NAME =
        "org.apache.xerces.parsers.SAXParser";

    /** Default Chart width. **/
    private static final int DEFAULT_WIDTH = 950;

    /** Default Chart height. **/
    private static final int DEFAULT_HEIGHT = 600;

    // ~ Instance fields -------------------------------------------------------

    /**
     * The QALab properties file.
     * @parameter expression="${project.basedir}/qalab.xml"
     * @required
     */
    private File qalabFile = null;

    /**
     * The directory to put the chart.
     * @parameter expression="${project.build.directory}/site/qalab"
     * @required
     */
    private File toDir = null;

    /**
     * If true then any debug logging output will be suppressed.
     * @parameter default-value=false
     */
    private boolean quiet = false;

    /**
     * The Chart width.
     * @parameter default-value=950
     */
    private int width = DEFAULT_WIDTH;

    /**
     * The Chart height.
     * @parameter default-value=600
     */
    private int height = DEFAULT_HEIGHT;

    /**
     * If the movingAverage is <= 0 then there is no moving average,
     * otherwise it shows the average based on the last n points, where
     * n is the value of this field.
     *
     * @parameter default-value=0
     */
    private int movingAverage = 0;

    /**
     * If true then generate a summary chart only.
     * @parameter alias="summaryOnly" default-value=true
     */
    private boolean summaryOnly = true;

    /**
     * Statistic type, defaulted to 'checkstyle'.
     * @parameter default-value="checkstyle"
     */
    private String types = "checkstyle";

    /**
     * validates the input then generates the charts.
     * @throws MojoExecutionException if anything goes wrong.
     */
    public final void execute() throws MojoExecutionException {
        // validate the provided parameters
        validate();

        // Display the files being processed
        if (!quiet) {
            getLog().debug("qalabFile='" + qalabFile.getPath() + "', toDir='"
                + toDir.getPath() + "' width='" + width
                + "' height='" + height
                + "' summaryOnly='" + summaryOnly + "'");
        }

        generateCharts();

    }

    /**
     * Generate the charts by creating a SAX parser specialised in finding the
     * elements of given type and creating charts in a given directory.
     * @throws MojoExecutionException if anything goes wrong.
     */
    private void generateCharts() throws MojoExecutionException {
        // define a sax stuff...
        final BuildStatForChartParser myForChartParser =
            new BuildStatForChartParser(new Maven2TaskLogger(this));

        myForChartParser.setChartHeight(height);
        myForChartParser.setChartWidth(width);
        myForChartParser.setToDir(toDir.getAbsolutePath() + "/");
        myForChartParser.setMovingAverage(movingAverage);
        myForChartParser.setSummaryOnly(summaryOnly);
        myForChartParser.setAcceptedStyle(types);
        myForChartParser.setQuiet(quiet);

        XMLReader parser = null;

        try {
            parser = XMLReaderFactory.createXMLReader(DEFAULT_PARSER_NAME);
            parser.setContentHandler(myForChartParser);
            parser.setErrorHandler(myForChartParser);

            if (!quiet) {
                getLog().debug("Parsing " + qalabFile);
            }

            parser.parse(new InputSource(new FileInputStream(qalabFile)));
        } catch (SAXException sex) {
            getLog().error(sex.toString());
            throw new MojoExecutionException("Error generating charts.", sex);
        } catch (FileNotFoundException fnfex) {
            getLog().error(fnfex.toString());
            throw new MojoExecutionException("Error generating charts.", fnfex);
        } catch (IOException ioex) {
            getLog().error(ioex.toString());
            throw new MojoExecutionException("Error generating charts.", ioex);
        }
    }

    /**
     * Validates the parameters supplied by maven.
     * @throws MojoExecutionException if anything goes wrong.
     */
    private void validate() throws MojoExecutionException {
        try {
            Utils.checkFile(qalabFile, "qalabFile");
        } catch (IOException ioex) {
            throw new MojoExecutionException(ioex.getMessage(), ioex);
        }

        if (toDir == null) {
            throw new MojoExecutionException("toDir is mandatory");
        }

        if (toDir.exists() && !toDir.isDirectory()) {
            final String message = "toDir must be a directory ("
                    + toDir + ")";
            throw new MojoExecutionException(message);
        }

        if (!toDir.exists()) {
            if (!quiet) {
                getLog().debug("Creating directory: " + toDir.toString());
            }

            toDir.mkdir();
        }
    }

}
