package net.objectlab.qalab.m2;

import net.objectlab.qalab.parser.BuildStatMoverHandler;
import net.objectlab.qalab.util.QALabTags;
import net.objectlab.qalab.util.TaskLogger;
import net.objectlab.qalab.m2.util.Utils;
import net.objectlab.qalab.m2.util.Maven2TaskLogger;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;

import java.text.ParseException;

/**
 * Goal to create the mover XML file given the correct parameters.
 * The movers report shows what has changed over the last "x" hours.
 *
 * @author <a href="http://www.davesag.com">Dave Sag</a>.
 * @goal movers
 * @phase deploy
 */
public class BuildStatMoversMojo extends AbstractMojo {

    // ~ Static final fields ---------------------------------------------------
    /**The size of the string based on ISO 8601 date format. */
    private static final int MIN_DATE_SIZE = 18;

    /**
     * The qalabFile, typically qalab.xml.
     * @parameter expression="${project.basedir}/qalab.xml"
     */
    private File qalabFile = null;

    /**
     * The number of hours to define the time window from now.
     * @parameter
     */
    private String startTimeHoursOffset = null;

    /**
     * Statistic types to be taken into account
     * (checkstyle|findbugs|pmd|simian).
     * Or comma separated combinations. Default is checkstyle only for now
     * as there are no working Maven2 plugins for the others.
     *
     * @parameter default-value="checkstyle"
     */
    private String types = "checkstyle";

    /**
     * The output XML file with ups and downs.
     * @parameter expression="${project.build.directory}/qalab/qalab-movers.xml"
     */
    private File outputXMLfile;

    /**
     * If true then any debug logging output will be suppressed.
     * @parameter default-value=false
     */
    private boolean quiet = false;

    /**
     * if true, adjust for weekend ie add 48h if start time falls on a weekend.
     * @parameter default-value=true
     */
    private boolean weekendAdjustment = true;

    /**
     * Start Time Window (instead of using startTimeHoursOffset).
     * parameter
     */
    private String startTimeWindow = null;

    /**
     * End Time Window (instead of assuming "now").
     * parameter
     */
    private String endTimeWindow = null;

    /**
     * Method invoked by the maven 2 framework to execute the compilation of
     * statistics movers withing the time window.
     * @throws MojoExecutionException if anything went wrong.
     */
    public final void execute() throws MojoExecutionException {
        validate();

        if (!quiet) {
            getLog().debug("qalabFile='" + qalabFile.getPath()
                + "', type='" + types
                + "'" + " output='" + outputXMLfile.getPath()
                + "' adjustForWeekend='" + weekendAdjustment + "'"
                + " startTimeOffset='" + startTimeHoursOffset + "'"
                + " startTimeWindow='" + startTimeWindow + "'"
                + " endTimeWindow='" + endTimeWindow + "'");
        }

        try {
            final TaskLogger logger = new Maven2TaskLogger(this);

            final BuildStatMoverHandler handler =
                new BuildStatMoverHandler(new InputSource(
                        new FileReader(qalabFile)), types,
                    new FileWriter(outputXMLfile), weekendAdjustment,
                    logger, quiet);

            if (startTimeWindow != null) {
                handler.setStartTimeWindow(QALabTags.DEFAULT_DATETIME_FORMAT
                    .parse(fixTime(startTimeWindow)));
            }

            if (endTimeWindow != null) {
                handler.setEndTimeWindow(QALabTags.DEFAULT_DATETIME_FORMAT
                    .parse(fixTime(endTimeWindow)));
            }

            if (startTimeHoursOffset != null) {
                handler.setOffsetHours(startTimeHoursOffset);
            }

            handler.process();
        } catch (ParserConfigurationException pcex) {
            getLog().error(pcex.toString());
        } catch (SAXException sex) {
            getLog().error(sex.toString());
        } catch (IOException ioex) {
            getLog().error(ioex.toString());
        } catch (ParseException pex) {
            throw new MojoExecutionException("Date Parse issue", pex);
        }
    }

    /**
     * Do some basic parameter validation.
     * @throws MojoExecutionException if anything goes wrong.
     */
    private void validate() throws MojoExecutionException {
        try {
            Utils.checkFile(qalabFile, "qalabFile");
        } catch (IOException ioex) {
            throw new MojoExecutionException(ioex.getMessage(), ioex);
        }

        if ((startTimeHoursOffset == null) && (startTimeWindow == null)) {
            throw new MojoExecutionException(
                    "Please set startTimeHoursOffset or startTimeWindow.");
        }

        if ((startTimeHoursOffset != null) && (startTimeWindow != null)) {
            throw new MojoExecutionException(
                    "Use startTimeHoursOffset OR startTimeWindow, not both.");
        }

        final File dir = outputXMLfile.getParentFile();
        if (!dir.exists()) {
            dir.mkdirs();
        }
    }

    /**
     * @param aTime The time string to adjust.
     * @return the adjusted time string - now with full hh:mm:ss
     */
    private String fixTime(final String aTime) {
        String result = aTime.trim();

        if (result.length() < MIN_DATE_SIZE) {
            result += " 00:00:00";
        }

        return result;
    }

}
