package net.objectlab.qalab.m2.report;

import java.io.File;
import java.io.IOException;
import java.util.Locale;
import java.util.ResourceBundle;
import java.io.FileInputStream;
import java.io.InputStream;

import net.objectlab.qalab.m2.util.Utils;
import net.objectlab.qalab.m2.util.XmlTransformer;

import org.apache.maven.project.MavenProject;
import org.apache.maven.reporting.AbstractMavenReport;
import org.apache.maven.reporting.MavenReportException;

import org.codehaus.doxia.site.renderer.SiteRenderer;
import javax.xml.transform.TransformerException;

/**
 * Goal that handles the generation of a movers report based on the data in
 * the qalab-movers.xml file.
 *
 * @author <a href="http://www.davesag.com">Dave Sag</a>.
 * @goal movers-report
 * @phase deploy
 */
public class MoversReport extends AbstractMavenReport {

    // ~ Instance fields -------------------------------------------------------
    /**
     * The generated qalab-movers.xml file.
     *
     * @parameter expression="${project.build.directory}/qalab/qalab-movers.xml"
     */
    private File inputMoversFile = null;

    /**
     * The xml input stream.
     */
    private InputStream theXmlStream = null;

    /**
     * The xslt style sheet.
     *
     * @parameter
     */
    private File moversStyleSheet = null;

    /**
     * The xslt style sheet input stream.
     */
    private InputStream theStyleSheetStream = null;

    /*
     * If true then any debug logging output will be suppressed.
     * @parameter default-value=false
     */
//    private boolean quiet = false;

    /**
     * The directory where the generated html report will go.
     * @parameter expression="${project.build.directory}/site/qalab"
     * @required
     */
    private String outputDirectory;

    /**
     * Not sure what this is.
     * @component
     */
    private SiteRenderer siteRenderer;

    /**
     * The maven project.
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;

    /**
     * generate the actual report.
     * @param aLocale ignored.
     * @throws MavenReportException if anything goes wrong.
     */
    protected final void executeReport(final Locale aLocale)
        throws MavenReportException {

        validate();
        final File outdir = new File(getOutputDirectory());
        if (!outdir.exists()) {
            outdir.mkdirs();
        }
        assert outdir.isDirectory()
                : "the output directory was not a directory";

        final File output;
        try {
            output = new File(outdir, "movers.html");
            output.createNewFile();
        } catch (IOException ioex) {
            throw new MavenReportException("could not create output file.",
                    ioex);
        }

        final XmlTransformer t = new XmlTransformer(theXmlStream,
                theStyleSheetStream, output);
        try {
            t.transform();
        } catch (TransformerException tex) {
            throw new MavenReportException("transformation failed.", tex);
        }

    }

    /**
     * @return "qalab/index"
     * @see org.apache.maven.reporting.MavenReport#getOutputName()
     */
    public final String getOutputName() {
        return "qalab/movers";
    }

    /**
     * @return The output directory as configured in your <code>pom.xml</code>.
     * @see org.apache.maven.reporting.AbstractMavenReport#getOutputDirectory()
     */
    protected final String getOutputDirectory() {
        return outputDirectory;
    }

    /**
     * @return The Maven Project itself.  Used internally to get access to
     * Config params etc.
     * @see org.apache.maven.reporting.AbstractMavenReport#getProject()
     */
    protected final MavenProject getProject() {
        return project;
    }

    /**
     * @return a direct reference to the site renderer.
     * @see org.apache.maven.reporting.AbstractMavenReport#getSiteRenderer()
     */
    protected final SiteRenderer getSiteRenderer() {
        return siteRenderer;
    }

    /**
     * @param aLocale The locale.
     * @return The locale specific report name.
     * @see org.apache.maven.reporting.MavenReport#getName(java.util.Locale)
     */
    public final String getName(final Locale aLocale) {
        return getBundle(aLocale).getString("report.qalab.movers.name");
    }

    /**
     * @param aLocale The locale.
     * @return The locale specific report description.
     * @see
     * org.apache.maven.reporting.MavenReport#getDescription(java.util.Locale)
     */
    public final String getDescription(final Locale aLocale) {
        return getBundle(aLocale).getString("report.qalab.movers.description");
    }

    /**
     * @return true.
     * @see org.apache.maven.reporting.MavenReport#isExternalReport
     */
    public final boolean isExternalReport() {
        return true;
    }

    /**
     * @param aLocale The locale.
     * @return the resource bundle for this report.
     */
    private static ResourceBundle getBundle(final Locale aLocale) {
        return ResourceBundle.getBundle("qalab-report",
                aLocale, MainReport.class.getClassLoader());
    }

    /**
     * Validates the parameters supplied by maven 2.
     * @throws MavenReportException if any supplied params are wrong.
     */
    private void validate() throws MavenReportException {
        try {
            Utils.checkFile(inputMoversFile, "inputMoversFile");
            theXmlStream = new FileInputStream(inputMoversFile);
        } catch (IOException ioex) {
            throw new MavenReportException(ioex.getMessage(), ioex);
        }

        if (moversStyleSheet == null) {
            try {
                theStyleSheetStream = Utils.extractAsInputStream(
                        "qalab-movers-html.xsl");
            } catch (IOException ioex) {
                throw new MavenReportException(
                        "Could not find the default stylesheet. "
                        + ioex.getMessage(), ioex);
            }
        } else {
           try {
                Utils.checkFile(moversStyleSheet, "moversStyleSheet");
                theStyleSheetStream = new FileInputStream(moversStyleSheet);
            } catch (IOException ioex) {
                throw new MavenReportException(ioex.getMessage(), ioex);
            }
        }
    }
}
