////////////////////////////////////////////////////////////////////////////////
//
//                  ObjectLab is sponsoring QALab
//
// Based in London, we are world leaders in the design and development
// of bespoke applications for the Securities Financing markets.
//
// <a href="http://www.objectlab.co.uk/open">Click here to learn more</a>
//           ___  _     _           _   _          _
//          / _ \| |__ (_) ___  ___| |_| |    __ _| |__
//         | | | | '_ \| |/ _ \/ __| __| |   / _` | '_ \
//         | |_| | |_) | |  __/ (__| |_| |__| (_| | |_) |
//          \___/|_.__// |\___|\___|\__|_____\__,_|_.__/
//                   |__/
//
//                   http://www.ObjectLab.co.uk
// ---------------------------------------------------------------------------
//
//QALab is released under the GNU General Public License.
//
//QALab: Collects QA Statistics from your build over time.
//2005+, ObjectLab Ltd
//
//This library is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public
//License as published by the Free Software Foundation; either
//version 2.1 of the License, or (at your option) any later version.
//
//This library is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//General Public License for more details.
//
//You should have received a copy of the GNU General Public
//License along with this library; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
////////////////////////////////////////////////////////////////////////////////
package net.objectlab.qalab.m2;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;

import net.objectlab.qalab.interfaces.QALabExporter;
import net.objectlab.qalab.parser.StatMerger;
import net.objectlab.qalab.util.TaskLogger;
import net.objectlab.qalab.m2.util.Utils;
import net.objectlab.qalab.m2.util.Maven2TaskLogger;

import org.xml.sax.InputSource;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.FileNotFoundException;

import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

/**
 * Use this goal to merge all available statistics, easiest use, supported:
 * Checkstyle, PMD, PMD CPD, Simian, FindBugs, Cobertua Line and Cobertura
 * Branch; it should not require any tailoring if you are using the Maven default
 * for all those reports.
 * 
 * @author Benoit Xhenseval
 * @goal merge-all
 * @phase site
 */
public class QALabStatAllMergeMojo extends AbstractMojo {

    // ~ Instance fields -------------------------------------------------------
    /**
     * The input file generated by Checkstyle.
     * 
     * @parameter expression="${project.build.directory}/checkstyle-result.xml"
     */
    private File checkstyleInputFile = null;

    /**
     * The fully qualified class name for the handler for Checkstyle.
     * 
     * @parameter default-value="net.objectlab.qalab.parser.CheckstyleStatMerge"
     */
    private String checkstyleHandler;

    /**
     * The input file generated by PMD.
     * 
     * @parameter expression="${project.build.directory}/pmd.xml"
     */
    private File pmdInputFile = null;

    /**
     * The fully qualified class name for the handler for PMD.
     * 
     * @parameter default-value="net.objectlab.qalab.parser.PMDStatMerge"
     */
    private String pmdHandler;

    /**
     * The input file generated by PMD CPD.
     * 
     * @parameter expression="${project.build.directory}/cpd.xml"
     */
    private File pmdCpdInputFile = null;

    /**
     * The fully qualified class name for the handler for PMD CPD.
     * 
     * @parameter default-value="net.objectlab.qalab.parser.PMDCPDStatMerge"
     */
    private String pmdCpdHandler;

    /**
     * The input file generated by FindBugs.
     * 
     * @parameter expression="${project.build.directory}/findbugs.xml"
     */
    private File findbugsInputFile = null;

    /**
     * The fully qualified class name for the handler for FindBugs.
     * 
     * @parameter default-value="net.objectlab.qalab.parser.FindBugsStatMerge"
     */
    private String findbugsHandler;

    /**
     * The input file generated by Simian.
     * 
     * @parameter expression="${project.build.directory}/simian-raw-report.xml"
     */
    private File simianInputFile = null;

    /**
     * The fully qualified class name for the handler for Simian.
     * 
     * @parameter default-value="net.objectlab.qalab.parser.SimianStatMerge"
     */
    private String simianHandler;

    /**
     * The input file generated by Cobertura.
     * 
     * @parameter expression="${project.reporting.outputDirectory}/cobertura/coverage.xml"
     */
    private File coberturaInputFile = null;

    /**
     * The fully qualified class name for the handler for Cobertura Line.
     * 
     * @parameter default-value="net.objectlab.qalab.parser.CoberturaLineStatMerge"
     */
    private String coberturaLineHandler;

    /**
     * The fully qualified class name for the handler for Cobertura Branch.
     * 
     * @parameter default-value="net.objectlab.qalab.parser.CoberturaBranchStatMerge"
     */
    private String coberturaBranchHandler;

    /**
     * The merged properties file. By default this is generated into the root of
     * the project as it is often checked into source control.
     * 
     * @parameter expression="${project.basedir}/qalab.xml"
     */
    private File outputFile = null;

    /**
     * If true then any debug logging output will be suppressed.
     * 
     * @parameter default-value=false
     */
    private boolean quiet = false;

    /**
     * If true then use ONLY DATE for timestamp (use in conjunction action
     * replace).
     * 
     * @parameter default-value=true
     */
    private boolean dateOnly = true;

    /**
     * The directory where the source code is.
     * 
     * @parameter expression="${project.build.sourceDirectory}"
     */
    private String srcDir;

    /**
     * The timestamp for the stats.
     * 
     * @parameter
     */
    private String mergerTimeStamp;

    /**
     * An exporter class name.
     * 
     * @parameter default-value="net.objectlab.qalab.exporter.QALabXMLExporter"
     */
    private String exporterClassName = "net.objectlab.qalab.exporter.QALabXMLExporter";

    /**
     * The properties file to use instead of setting them one by one.
     * 
     * @parameter
     */
    private File propertiesFile;

    /**
     * the loaded properties from the properties file declared above.
     */
    private Properties theProperties = null;

    // ~ Methods

    /**
     * Method invoked by the Maven framework to execute the action associated with
     * this task. This will validate the input parameters then merge the
     * statistics.
     * 
     * @throws MojoExecutionException
     *             if anything goes wrong.
     */
    public final void execute() throws MojoExecutionException {

        theProperties = createOverridingProperties();

        getLog().info("Starting QALab ALL Merge");

        doMerge(checkstyleInputFile, checkstyleHandler);
        doMerge(pmdInputFile, pmdHandler);
        doMerge(pmdCpdInputFile, pmdCpdHandler);
        doMerge(findbugsInputFile, findbugsHandler);
        doMerge(simianInputFile, simianHandler);
        doMerge(coberturaInputFile, coberturaLineHandler);
        doMerge(coberturaInputFile, coberturaBranchHandler);

        // If there was nothing to merge, create an empty qalab.xml in order to avoid reporting problems
        // For instance, an empty qalab.xml is necessary for the root POM, when using <packaging>pom</packaging
        if (!getOutputFile().exists()) {
        	try {
				QALabExporter exporter = (QALabExporter) Class.forName(getExporterClassName()).newInstance();

				getTheProperties().setProperty("qalab.merge.output.file", getOutputFile().getAbsolutePath());
	            exporter.setQuiet(isQuiet());
	            exporter.setTaskLogger(new Maven2TaskLogger(this));
	            exporter.configure(getTheProperties());

				exporter.save();
			} catch (Exception excn) {
				excn.printStackTrace();
			}
        }
    }

    private void doMerge(File inputFile, String handler) throws MojoExecutionException {
        // validate the provided parameters
        if (!validate(inputFile)) {
            return;
        }

        // Display the files being processed
        if (!quiet) {
            getLog().info("----------------------------------------------");
            getLog().info("QALab Merge:");
            getLog().info("checkstyleInputFile='" + inputFile.getPath());
            getLog().info("outputFile='" + outputFile.getPath());
            getLog().info("srcDir='" + srcDir + "', mergerTimeStamp=" + mergerTimeStamp);
            String proppath = null;
            if (propertiesFile != null) {
                proppath = propertiesFile.getPath();
            }
            getLog().info("quiet='" + quiet + "', propertiesFile='" + proppath + "'.");
            for (final Iterator it = theProperties.entrySet().iterator(); it.hasNext();) {

                final Map.Entry entry = (Map.Entry) it.next();
                final String key = entry.getKey().toString();
                if (key.indexOf("qalab") >= 0) {
                    getLog().info(key + " = '" + entry.getValue() + "'");
                }
            }
        }

        mergeFiles(inputFile, handler);
    }

    /**
     * Merge the statistics into the qalab.xml.
     * 
     * @throws MojoExecutionException
     *             if anything goes wrong.
     */
    private void mergeFiles(File inputFile, String handler) throws MojoExecutionException {
        try {
            final TaskLogger logger = new Maven2TaskLogger(this);

            // create the exporter
            final QALabExporter exporter = (QALabExporter) Class.forName(getExporterClassName()).newInstance();

            getTheProperties().setProperty("qalab.merge.output.file", getOutputFile().getAbsolutePath());

            exporter.setQuiet(isQuiet());
            exporter.setTaskLogger(logger);

            final StatMerger merger = (StatMerger) Class.forName(handler).newInstance();

            merger.setQuiet(isQuiet());
            merger.setSrcDir(getSrcDir());
            merger.setTaskLogger(logger);
            merger.setMergerTimeStamp(getMergerTimeStamp(), dateOnly);

            getTheProperties().setProperty("qalab.merge.output.timestamp", merger.getMergerTimeStamp());
            getTheProperties().setProperty("qalab.merge.type", merger.getType());
            exporter.configure(getTheProperties());

            merger.mergeStats(new InputSource(new FileInputStream(inputFile)), exporter);
            getLog().info("Files: " + merger.getFileCount() + " Violations:" + merger.getTotalStatistics());

            exporter.save();
        } catch (IllegalAccessException e) {
            throw new MojoExecutionException(e.toString());
        } catch (FileNotFoundException e) {
            throw new MojoExecutionException(e.toString());
        } catch (IOException e) {
            throw new MojoExecutionException(e.toString());
        } catch (ClassNotFoundException e) {
            throw new MojoExecutionException(e.toString());
        } catch (InstantiationException e) {
            throw new MojoExecutionException(e.toString());
        }
    }

    /**
     * Validates the parameters supplied by maven 2.
     */
    private boolean validate(final File file) {
        try {
            Utils.checkFile(file, "inputFile");
            return true;
        } catch (IOException ioex) {
            getLog().warn("\n\nQALab ==> The file " + file.getPath() + " cannot be accessed. SKIPPING....\n\n");
        }
        return false;
    }

    /**
     * Create the Properties object based on the arguments specified to the Mojo
     * in your <code>pom.xml</code> file.
     * 
     * @return the properties for property expansion expansion
     * @throws MojoExecutionException
     *             if anything goes wrong.
     */
    private Properties createOverridingProperties() throws MojoExecutionException {

        final Properties result = new Properties();

        // Load the properties file if specified
        if (propertiesFile != null && propertiesFile.canRead() && propertiesFile.isFile()) {
            FileInputStream inStream = null;
            if (!quiet) {
                getLog().debug("loading " + propertiesFile);
            }

            try {
                inStream = new FileInputStream(propertiesFile);
                result.load(inStream);
            } catch (FileNotFoundException fnfex) {
                throw new MojoExecutionException("Could not find Properties file '" + propertiesFile + "'", fnfex);
            } catch (IOException ioex) {
                throw new MojoExecutionException("Error loading Properties file '" + propertiesFile + "'", ioex);
            } finally {
                try {
                    if (inStream != null) {
                        inStream.close();
                    }
                } catch (IOException ioex) {
                    throw new MojoExecutionException("Error closing Properties file '" + propertiesFile + "'", ioex);
                }
            }
        }

        // override with Maven properties as apropriate.
        final Map projectContext = getPluginContext();

        if (projectContext != null) {
            for (final Iterator it = projectContext.entrySet().iterator(); it.hasNext();) {

                final Map.Entry entry = (Map.Entry) it.next();
                final String value = String.valueOf(entry.getValue());
                if (entry.getKey().toString().indexOf("qalab") >= 0) {
                    getLog().info("Adding " + entry.getKey() + " / " + value);
                }
                result.put(entry.getKey(), value);
            }
        }
        
        return result;
    }

    public String getExporterClassName() {
        return exporterClassName;
    }

    public void setExporterClassName(String exporterClassName) {
        this.exporterClassName = exporterClassName;
    }

    public File getInputFile() {
        return checkstyleInputFile;
    }

    public void setInputFile(File inputFile) {
        this.checkstyleInputFile = inputFile;
    }

    public String getMergerTimeStamp() {
        return mergerTimeStamp;
    }

    public void setMergerTimeStamp(String mergerTimeStamp) {
        this.mergerTimeStamp = mergerTimeStamp;
    }

    public File getOutputFile() {
        return outputFile;
    }

    public void setOutputFile(File outputFile) {
        this.outputFile = outputFile;
    }

    public File getPropertiesFile() {
        return propertiesFile;
    }

    public void setPropertiesFile(File propertiesFile) {
        this.propertiesFile = propertiesFile;
    }

    public boolean isQuiet() {
        return quiet;
    }

    public void setQuiet(boolean quiet) {
        this.quiet = quiet;
    }

    public String getSrcDir() {
        return srcDir;
    }

    public void setSrcDir(String srcDir) {
        this.srcDir = srcDir;
    }

    public Properties getTheProperties() {
        return theProperties;
    }

    public void setTheProperties(Properties theProperties) {
        this.theProperties = theProperties;
    }

    /**
     * @return the checkstyleHandler
     */
    public String getCheckstyleHandler() {
        return checkstyleHandler;
    }

    /**
     * @param checkstyleHandler the checkstyleHandler to set
     */
    public void setCheckstyleHandler(String checkstyleHandler) {
        this.checkstyleHandler = checkstyleHandler;
    }

    /**
     * @return the checkstyleInputFile
     */
    public File getCheckstyleInputFile() {
        return checkstyleInputFile;
    }

    /**
     * @param checkstyleInputFile the checkstyleInputFile to set
     */
    public void setCheckstyleInputFile(File checkstyleInputFile) {
        this.checkstyleInputFile = checkstyleInputFile;
    }

    /**
     * @return the coberturaBranchHandler
     */
    public String getCoberturaBranchHandler() {
        return coberturaBranchHandler;
    }

    /**
     * @param coberturaBranchHandler the coberturaBranchHandler to set
     */
    public void setCoberturaBranchHandler(String coberturaBranchHandler) {
        this.coberturaBranchHandler = coberturaBranchHandler;
    }

    /**
     * @return the coberturaInputFile
     */
    public File getCoberturaInputFile() {
        return coberturaInputFile;
    }

    /**
     * @param coberturaInputFile the coberturaInputFile to set
     */
    public void setCoberturaInputFile(File coberturaInputFile) {
        this.coberturaInputFile = coberturaInputFile;
    }

    /**
     * @return the coberturaLineHandler
     */
    public String getCoberturaLineHandler() {
        return coberturaLineHandler;
    }

    /**
     * @param coberturaLineHandler the coberturaLineHandler to set
     */
    public void setCoberturaLineHandler(String coberturaLineHandler) {
        this.coberturaLineHandler = coberturaLineHandler;
    }

    /**
     * @return the dateOnly
     */
    public boolean isDateOnly() {
        return dateOnly;
    }

    /**
     * @param dateOnly the dateOnly to set
     */
    public void setDateOnly(boolean dateOnly) {
        this.dateOnly = dateOnly;
    }

    /**
     * @return the findbugsHandler
     */
    public String getFindbugsHandler() {
        return findbugsHandler;
    }

    /**
     * @param findbugsHandler the findbugsHandler to set
     */
    public void setFindbugsHandler(String findbugsHandler) {
        this.findbugsHandler = findbugsHandler;
    }

    /**
     * @return the findbugsInputFile
     */
    public File getFindbugsInputFile() {
        return findbugsInputFile;
    }

    /**
     * @param findbugsInputFile the findbugsInputFile to set
     */
    public void setFindbugsInputFile(File findbugsInputFile) {
        this.findbugsInputFile = findbugsInputFile;
    }

    /**
     * @return the pmdCpdHandler
     */
    public String getPmdCpdHandler() {
        return pmdCpdHandler;
    }

    /**
     * @param pmdCpdHandler the pmdCpdHandler to set
     */
    public void setPmdCpdHandler(String pmdCpdHandler) {
        this.pmdCpdHandler = pmdCpdHandler;
    }

    /**
     * @return the pmdCpdInputFile
     */
    public File getPmdCpdInputFile() {
        return pmdCpdInputFile;
    }

    /**
     * @param pmdCpdInputFile the pmdCpdInputFile to set
     */
    public void setPmdCpdInputFile(File pmdCpdInputFile) {
        this.pmdCpdInputFile = pmdCpdInputFile;
    }

    /**
     * @return the pmdHandler
     */
    public String getPmdHandler() {
        return pmdHandler;
    }

    /**
     * @param pmdHandler the pmdHandler to set
     */
    public void setPmdHandler(String pmdHandler) {
        this.pmdHandler = pmdHandler;
    }

    /**
     * @return the pmdInputFile
     */
    public File getPmdInputFile() {
        return pmdInputFile;
    }

    /**
     * @param pmdInputFile the pmdInputFile to set
     */
    public void setPmdInputFile(File pmdInputFile) {
        this.pmdInputFile = pmdInputFile;
    }

    /**
     * @return the simianHandler
     */
    public String getSimianHandler() {
        return simianHandler;
    }

    /**
     * @param simianHandler the simianHandler to set
     */
    public void setSimianHandler(String simianHandler) {
        this.simianHandler = simianHandler;
    }

    /**
     * @return the simianInputFile
     */
    public File getSimianInputFile() {
        return simianInputFile;
    }

    /**
     * @param simianInputFile the simianInputFile to set
     */
    public void setSimianInputFile(File simianInputFile) {
        this.simianInputFile = simianInputFile;
    }
}
/*
 *                   ObjectLab is sponsoring QALab
 * 
 * Based in London, we are world leaders in the design and development 
 * of bespoke applications for the securities financing markets.
 * 
 * <a href="http://www.objectlab.co.uk/open">Click here to learn more about us</a>
 *           ___  _     _           _   _          _
 *          / _ \| |__ (_) ___  ___| |_| |    __ _| |__
 *         | | | | '_ \| |/ _ \/ __| __| |   / _` | '_ \
 *         | |_| | |_) | |  __/ (__| |_| |__| (_| | |_) |
 *          \___/|_.__// |\___|\___|\__|_____\__,_|_.__/
 *                   |__/
 *
 *                     www.ObjectLab.co.uk
 */
