////////////////////////////////////////////////////////////////////////////////
//
//                  ObjectLab is sponsoring QALab
//
// Based in London, we are world leaders in the design and development
// of bespoke applications for the Securities Financing markets.
//
// <a href="http://www.objectlab.co.uk/open">Click here to learn more</a>
//           ___  _     _           _   _          _
//          / _ \| |__ (_) ___  ___| |_| |    __ _| |__
//         | | | | '_ \| |/ _ \/ __| __| |   / _` | '_ \
//         | |_| | |_) | |  __/ (__| |_| |__| (_| | |_) |
//          \___/|_.__// |\___|\___|\__|_____\__,_|_.__/
//                   |__/
//
//                   http://www.ObjectLab.co.uk
// ---------------------------------------------------------------------------
//
//QALab is released under the GNU General Public License.
//
//QALab: Collects QA Statistics from your build over time.
//2005+, ObjectLab Ltd
//
//This library is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public
//License as published by the Free Software Foundation; either
//version 2.1 of the License, or (at your option) any later version.
//
//This library is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//General Public License for more details.
//
//You should have received a copy of the GNU General Public
//License along with this library; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
////////////////////////////////////////////////////////////////////////////////
package net.objectlab.qalab.m2.report;

import java.io.File;
import java.io.IOException;
import java.util.Locale;
import java.util.ResourceBundle;
import java.io.FileInputStream;
import java.io.InputStream;

import net.objectlab.qalab.m2.util.Utils;
import net.objectlab.qalab.m2.util.XmlTransformer;

import org.apache.maven.project.MavenProject;
import org.apache.maven.reporting.AbstractMavenReport;
import org.apache.maven.reporting.MavenReportException;

import org.codehaus.doxia.site.renderer.SiteRenderer;
import javax.xml.transform.TransformerException;

/**
 * Goal that handles the generation of a site report based on the data in the
 * qalab.xml file.
 * 
 * @author Dave Sag and Benoit Xhenseval.
 * @goal report
 * @phase deploy
 */
public class MainReport extends AbstractMavenReport {

    // ~ Instance fields -------------------------------------------------------
    /**
     * The generated qalab.xml file.
     * 
     * @parameter expression="${project.basedir}/qalab.xml"
     */
    private File inputFile = null;

    /**
     * The xslt stylesheet bundled, either qalab-chart-xdoc.xsl or
     * qalab-chart-html.xsl.
     * 
     * @parameter default-value="qalab-chart-xdoc.xsl";
     */
    private String moversBundledXsl = null;

    /**
     * The directory where the generated html report will go. @ parameter
     * expression="${project.reporting.outputDirectory}/qalab"
     * @parameter expression="${project.build.directory}/generated-site/xdoc/qalab"
     * @required
     */
    private String outputDirectory;

    /**
     * Statistic types, defaulted to
     * 'checkstyle,pmd,pmd-cpd,findbugs,simian,cobertura-line,cobertura-branch'.
     * 
     * @parameter default-value="checkstyle,pmd,pmd-cpd,findbugs,simian,cobertura-line,cobertura-branch"
     */
    private String types = "checkstyle,pmd,pmd-cpd,findbugs,simian,cobertura-line,cobertura-branch";

    /**
     * The xml input stream.
     */
    private InputStream theXmlStream = null;

    /**
     * The xslt style sheet.
     * 
     * @parameter
     */
    private File styleSheet = null;

    /**
     * The xslt style sheet input stream.
     */
    private InputStream theStyleSheetStream = null;

    /**
     * The number of hours to define the time window from now.
     * 
     * @parameter default-value="48"
     */
    private String startTimeHoursOffset = "48";

    /**
     * The string version of the actual date computed by the offset set above.
     */
    private String theOffset;

    /**
     * Not sure what this is.
     * 
     * @component
     */
    private SiteRenderer siteRenderer;

    /**
     * The maven project.
     * 
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;

    /**
     * generate the actual report.
     * 
     * @param aLocale
     *            ignored.
     * @throws MavenReportException
     *             if anything goes wrong.
     */
    protected final void executeReport(final Locale aLocale) throws MavenReportException {

        validate();
        final File outdir = new File(getOutputDirectory());
        if (!outdir.exists()) {
            outdir.mkdirs();
        }
        assert outdir.isDirectory() : "the output directory was not a directory";

        final File output;
        try {
            output = File.createTempFile("qalab", null, outdir);
            assert output != null : "the temp file is null.";

        } catch (IOException ioex) {
            throw new MavenReportException("could not create temp file.", ioex);
        }

        final XmlTransformer t = new XmlTransformer(theXmlStream, theStyleSheetStream, output);
        t.addParameter("targetdir", getOutputDirectory());
        t.addParameter("type", types);
        t.addParameter("offset", theOffset);
        try {
            t.transform();
        } catch (TransformerException tex) {
            throw new MavenReportException("transformation failed.", tex);
        }

        // clean up the temp file when the plugin is done.
        output.deleteOnExit();
    }

    /**
     * @return "qalab/index"
     * @see org.apache.maven.reporting.MavenReport#getOutputName()
     */
    public final String getOutputName() {
        return "qalab/index";
    }

    /**
     * @return The output directory as configured in your <code>pom.xml</code>.
     * @see org.apache.maven.reporting.AbstractMavenReport#getOutputDirectory()
     */
    protected final String getOutputDirectory() {
        return outputDirectory;
    }

    /**
     * @return The Maven Project itself. Used internally to get access to Config
     *         params etc.
     * @see org.apache.maven.reporting.AbstractMavenReport#getProject()
     */
    protected final MavenProject getProject() {
        return project;
    }

    /**
     * @return a direct reference to the site renderer.
     * @see org.apache.maven.reporting.AbstractMavenReport#getSiteRenderer()
     */
    protected final SiteRenderer getSiteRenderer() {
        return siteRenderer;
    }

    /**
     * @param aLocale
     *            The locale.
     * @return The locale specific report name.
     * @see org.apache.maven.reporting.MavenReport#getName(java.util.Locale)
     */
    public final String getName(final Locale aLocale) {
        return getBundle(aLocale).getString("report.qalab.name");
    }

    /**
     * @param aLocale
     *            The locale.
     * @return The locale specific report description.
     * @see org.apache.maven.reporting.MavenReport#getDescription(java.util.Locale)
     */
    public final String getDescription(final Locale aLocale) {
        return getBundle(aLocale).getString("report.qalab.description");
    }

    /**
     * @return true.
     * @see org.apache.maven.reporting.MavenReport#isExternalReport
     */
    public final boolean isExternalReport() {
        return true;
    }

    /**
     * @param aLocale
     *            The locale.
     * @return the resource bundle for this report.
     */
    private static ResourceBundle getBundle(final Locale aLocale) {
        return ResourceBundle.getBundle("qalab-report", aLocale, MainReport.class.getClassLoader());
    }

    /**
     * Validates the parameters supplied by maven 2.
     * 
     * @throws MavenReportException
     *             if any supplied params are wrong.
     */
    private void validate() throws MavenReportException {
        try {
            Utils.checkFile(inputFile, "inputFile");
            theXmlStream = new FileInputStream(inputFile);
        } catch (IOException ioex) {
            throw new MavenReportException(ioex.getMessage(), ioex);
        }

        if (styleSheet == null) {
            try {
                theStyleSheetStream = Utils.extractAsInputStream(moversBundledXsl);
            } catch (IOException ioex) {
                throw new MavenReportException("Could not find the default stylesheet. " + ioex.getMessage(), ioex);
            }
        } else {
            try {
                Utils.checkFile(styleSheet, "styleSheet");
                theStyleSheetStream = new FileInputStream(styleSheet);
            } catch (IOException ioex) {
                throw new MavenReportException(ioex.getMessage(), ioex);
            }
        }

        theOffset = Utils.formatDateBasedOnOffset(startTimeHoursOffset);

    }
}
/*
 *                   ObjectLab is sponsoring QALab
 * 
 * Based in London, we are world leaders in the design and development 
 * of bespoke applications for the securities financing markets.
 * 
 * <a href="http://www.objectlab.co.uk/open">Click here to learn more about us</a>
 *           ___  _     _           _   _          _
 *          / _ \| |__ (_) ___  ___| |_| |    __ _| |__
 *         | | | | '_ \| |/ _ \/ __| __| |   / _` | '_ \
 *         | |_| | |_) | |  __/ (__| |_| |__| (_| | |_) |
 *          \___/|_.__// |\___|\___|\__|_____\__,_|_.__/
 *                   |__/
 *
 *                     www.ObjectLab.co.uk
 */
