/*
Copyright 2013 McDowell

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 */

package net.sf.jsonunicode;

import java.nio.charset.Charset;
import java.nio.charset.UnsupportedCharsetException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableSet;

/**
 * Type defining the intersection of Java and JSON encodings.
 * <p/>
 * WARNING: Sun <A href=
 * "http://docs.oracle.com/javase/1.5.0/docs/guide/intl/encoding.doc.html">Java
 * 5 does not support UTF-32</A>.
 * <p/>
 * Note: Java implementations are only <A href=
 * "http://docs.oracle.com/javase/1.5.0/docs/api/java/nio/charset/Charset.html"
 * >required</A> to support six encodings; of those, the encodings supported by
 * JSON are: UTF-8; UTF-16BE; UTF-16LE.
 */
public final class JsonCharsets {
  private JsonCharsets() {
  }

  /**
   * UTF-8
   */
  public static final Charset UTF_8 = Charset.forName("UTF-8");

  /**
   * UTF-16BE (big endian)
   */
  public static final Charset UTF_16BE = Charset.forName("UTF-16BE");

  /**
   * UTF-16LE (little endian)
   */
  public static final Charset UTF_16LE = Charset.forName("UTF-16LE");

  private static final Set<Charset> ENCODINGS = encodings();

  private static Set<Charset> encodings() {
    Set<Charset> encodings = set(asList(UTF_8, UTF_16BE, UTF_16LE));
    try {
      encodings.add(UTF_32LE());
    } catch (UnsupportedCharsetException e) {
      // failsafe
    }
    try {
      encodings.add(UTF_32BE());
    } catch (UnsupportedCharsetException e) {
      // failsafe
    }
    return unmodifiableSet(encodings);
  }

  /**
   * The encodings supported by both JSON and the Java runtime.
   *
   * @return UTF-8; UTF-16BE; UTF-16LE; plus UTF-32LE and UTF-32BE if they are
   * supported by the current Java runtime
   */
  public static Set<Charset> supportedJsonEncodings() {
    return ENCODINGS;
  }

  /**
   * WARNING: known to throw {@link UnsupportedCharsetException} on Sun Java 5.
   *
   * @return UTF-32LE (little endian)
   * @throws UnsupportedCharsetException
   */
  public static Charset UTF_32LE() {
    return Charset.forName("UTF-32LE");
  }

  /**
   * WARNING: known to throw {@link UnsupportedCharsetException} on Sun Java 5.
   *
   * @return UTF-32BE (big endian)
   * @throws UnsupportedCharsetException
   */
  public static Charset UTF_32BE() {
    return Charset.forName("UTF-32BE");
  }

  private static <T> Set<T> set(Collection<T> coll) {
    return new HashSet<T>(coll);
  }
}

