package org.springframework.web.servlet.view.json;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.validation.BindingResult;
import org.springframework.web.servlet.support.RequestContext;
import org.springframework.web.servlet.view.AbstractView;
import org.springframework.web.servlet.view.json.error.StandartErrorConverter;
import org.springframework.web.servlet.view.json.writer.sojo.SojoJsonStringWriter;


public class JsonView extends AbstractView {
	private static final String DEFAULT_ENCODING = "UTF-8";
	
	public static final String DEFAULT_JSON_CONTENT_TYPE = "application/json";
	private static final String REQUEST_CONTEXT_ATTRIBUTE =  RequestContext.class.toString();
	public static final String DEFAULT_HIJACKSAVE_PREFIX_POSTFIX = "JSON";
	
	private JsonViewCallback jsonViewCallback = new StandartErrorConverter();
	private List<JsonErrorHandler> jsonErrors = new ArrayList<JsonErrorHandler>();
	private JsonStringWriter jsonWriter = new SojoJsonStringWriter();
	private String encoding;
	
	private String hijackSafePrefixPostFix = DEFAULT_HIJACKSAVE_PREFIX_POSTFIX;
	private boolean hijackSafe = false;
	
	
	public JsonView(){
		super();
		setRequestContextAttribute(REQUEST_CONTEXT_ATTRIBUTE);
		setContentType(DEFAULT_JSON_CONTENT_TYPE);
		setEncoding(DEFAULT_ENCODING);
	}
	
	@SuppressWarnings("unchecked")
	public void renderMergedOutputModel(Map model, HttpServletRequest request,
			HttpServletResponse response) throws Exception {
		response.setContentType(getContentType());
		response.setCharacterEncoding(encoding);
		RequestContext rc = getRequestContext(model);
		BindingResult br = getBindingResult( model);
		JsonWriterConfiguratorTemplateRegistry configuratorTemplateRegistry = getConfiguratorTemplateRegistry(request);
		
		if(hasErrors(rc, br)){
			jsonViewCallback.onPopulateErrors(model, rc, br);
			triggerJsonErrors(model, rc, br, request, response); 
		}else{
			jsonViewCallback.onPopulateSuccess(model, rc, br);
		}
		
		if (hijackSafe)
        {
            response.getWriter().print("/*" + hijackSafePrefixPostFix);
        }
        
		jsonWriter.convertAndWrite(model, configuratorTemplateRegistry, response.getWriter(), br);

        if (hijackSafe)
        {
            response.getWriter().print(hijackSafePrefixPostFix + "*/");
        }
	}

	protected boolean hasErrors(RequestContext rc, BindingResult br) {
		if(br == null)
			return false;
		return br.hasErrors();
	}

	@SuppressWarnings("unchecked")
	protected void triggerJsonErrors(Map model, RequestContext rc,
			BindingResult br, HttpServletRequest request,
			HttpServletResponse response) throws Exception {
		
		if(jsonErrors == null || jsonErrors.size() == 0)
			return;
		
		for(JsonErrorHandler error: jsonErrors){
				error.triggerError(model, rc, br, request, response);
		}
		
		
	}
	
	@SuppressWarnings("unchecked")
	protected  BindingResult getBindingResult(Map model){
		for(Object key : model.keySet() ){
		    if(((String)key).startsWith(BindingResult.MODEL_KEY_PREFIX))
			return (BindingResult) model.remove(key);
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	protected  RequestContext getRequestContext(Map model){
		return (RequestContext) model.remove(getRequestContextAttribute());
	}
	
	protected  JsonWriterConfiguratorTemplateRegistry getConfiguratorTemplateRegistry(HttpServletRequest request){
		return JsonWriterConfiguratorTemplateRegistry.load(request);
	}

	public List<JsonErrorHandler> getJsonErrors() {
		return jsonErrors;
	}

	public void setJsonErrors(List<JsonErrorHandler> jsonErrors) {
		this.jsonErrors = jsonErrors;
	}
	
	

	public JsonViewCallback getJsonViewCallback() {
		return jsonViewCallback;
	}

	public void setJsonViewCallback(JsonViewCallback jsonViewCallback) {
		this.jsonViewCallback = jsonViewCallback;
	}

	public JsonStringWriter getJsonWriter() {
		return jsonWriter;
	}

	public void setJsonWriter(JsonStringWriter jsonWriter) {
		this.jsonWriter = jsonWriter;
	}

	public String getEncoding() {
		return encoding;
	}

	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	public boolean getHijackSafe() {
		return hijackSafe;
	}

	public void setHijackSafe(boolean hijackSafe) {
		this.hijackSafe = hijackSafe;
	}

	public String getHijackSafePrefixPostFix() {
		return hijackSafePrefixPostFix;
	}

	public void setHijackSafePrefixPostFix(String hijackSafePrefixPostFix) {
		this.hijackSafePrefixPostFix = hijackSafePrefixPostFix;
	}
	
	
	
}
