package org.springframework.web.servlet.view.path;

import java.util.HashMap;
import java.util.Map;

import org.springframework.web.servlet.view.path.CommonsBeanUtilsPathElement.Typ;

public class CommonsBeanUtilslPathTracker {
    private int pointer;
    private int capacity;
    private CommonsBeanUtilsPathElement[] pathStack;
    private Map[] indexMapStack;

    private CommonsBeanUtilsPath currentPath;

    public CommonsBeanUtilslPathTracker() {
        this(16);
    }

    /**
     * @param initialCapacity Size of the initial stack of nodes (one level per depth in the tree). Note that this is
     *                        only for optimizations - the stack will resize itself if it exceeds its capacity. If in doubt,
     *                        use the other constructor.
     */
    public CommonsBeanUtilslPathTracker(int initialCapacity) {
        this.capacity = Math.max(1, initialCapacity);
        pathStack = new CommonsBeanUtilsPathElement[capacity];
        indexMapStack = new Map[capacity];
    }

    /**
     * Notify the tracker that the stream has moved into a new element.
     *   
        Integer index = (Integer)indexMapStack[pointer].get(pathStack[pointer].getName());
        if(index == null)
        	indexMapStack[pointer].put(pathStack[pointer].getName(), new Integer(0));
        else
        	indexMapStack[pointer].put( pathStack[pointer].getName(), index + 1);
        
     * @param name Name of the element
     */
    public void pushElement(String name, CommonsBeanUtilsPathElement element) {
        
    	if (pointer + 1 >= capacity) {
            resizeStacks(capacity * 2);
        }
        pathStack[pointer] = element;
        Map indexMap = indexMapStack[pointer];
        if (indexMap == null) {
            indexMap = new HashMap();
            indexMapStack[pointer] = indexMap;
        }
        Integer index = (Integer)indexMapStack[pointer].get(pathStack[pointer].getName());
        if(index == null)
        	indexMapStack[pointer].put(pathStack[pointer].getName(), new Integer(0));
        else
        	indexMapStack[pointer].put( pathStack[pointer].getName(), index + 1);
        if(pointer > 1 && Typ.SIMPEL.equals(pathStack[pointer].getTyp()) && Typ.ARRAY.equals(pathStack[pointer -1].getTyp())){
        	index = (Integer)indexMapStack[pointer-1].get(pathStack[pointer-1].getName());
        	indexMapStack[pointer-1].put( pathStack[pointer-1].getName(), index + 1);
    	}
         
        pointer++;
        currentPath = null;
    }

    /**
     * Notify the tracker that the stream has moved out of an element.
     */
    public void popElement() {
        indexMapStack[pointer] = null;
        currentPath = null;
        pointer--;
    }


    private void resizeStacks(int newCapacity) {
    	CommonsBeanUtilsPathElement[] newPathStack = new CommonsBeanUtilsPathElement[newCapacity];
        Map[] newIndexMapStack = new Map[newCapacity];
        int min = Math.min(capacity, newCapacity);
        System.arraycopy(pathStack, 0, newPathStack, 0, min);
        System.arraycopy(indexMapStack, 0, newIndexMapStack, 0, min);
        pathStack = newPathStack;
        indexMapStack = newIndexMapStack;
        capacity = newCapacity;
    }

    /**
     * Current Path in stream.
     */
    public CommonsBeanUtilsPath getPath() {
        if (currentPath == null) {
        	CommonsBeanUtilsPathElement[] chunks = new CommonsBeanUtilsPathElement[pointer + 1];
            for (int i = 0; i < pointer; i++) {
                Integer integer = ((Integer) indexMapStack[i].get(pathStack[i].getName()));
                pathStack[i].setIndex(integer);
                chunks[i] = pathStack[i];
            }
            currentPath = new CommonsBeanUtilsPath(chunks, CommonsBeanUtilsPath.SHORT);
        }
        return currentPath;
    }

    protected int getCapacity() {
		return capacity;
	}

    protected CommonsBeanUtilsPathElement[] getPathStack() {
		return pathStack;
	}


    protected Map[] getIndexMapStack() {
		return indexMapStack;
	}

	
    public int getPointer() {
		return pointer;
	}

    
	
    
}
