package org.springframework.web.servlet.view.json.writer.xstream.io;

import java.io.Writer;
import java.util.Collection;
import java.util.Map;

import org.springframework.web.servlet.view.json.JsonStringWriter;
import org.springframework.web.servlet.view.path.CommonsBeanUtilsPathElement;
import org.springframework.web.servlet.view.path.CommonsBeanUtilslPathTracker;

import com.thoughtworks.xstream.core.util.FastStack;
import com.thoughtworks.xstream.core.util.Primitives;
import com.thoughtworks.xstream.core.util.QuickWriter;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;


public class JsonMapHierarchicalStreamWriter implements CustomPathExtendedHierarchicalStreamWriter {
		private CommonsBeanUtilslPathTracker pathTracker = new CommonsBeanUtilslPathTracker();
	
		private final QuickWriter writer;
		private String keepValueTypesMode;
		private final FastStack elementStack = new FastStack(16);
	    private final char[] lineIndenter;

	    private int depth;
	    private boolean readyForNewLine;
	    private boolean tagIsEmpty;
	    private String newLine;

	    public JsonMapHierarchicalStreamWriter(Writer writer, String keepValueTypesMode, char[] lineIndenter, String newLine) {
	        this.writer = new QuickWriter(writer);
	        this.keepValueTypesMode = keepValueTypesMode;
	        this.lineIndenter = lineIndenter;
	        this.newLine = newLine;
	    }
	    
	    public JsonMapHierarchicalStreamWriter(Writer writer, String keepValueTypesMode, String newLine) {
	        this(writer, keepValueTypesMode, new char[]{' ', ' '}, newLine);
	    }

	    public JsonMapHierarchicalStreamWriter(Writer writer, String newLine) {
	        this(writer, JsonStringWriter.MODE_KEEP_VALUETYPES_NONE, new char[]{' ', ' '}, newLine);
	    }
	  
	    

	    public JsonMapHierarchicalStreamWriter(Writer writer) {
	        this(writer, JsonStringWriter.MODE_KEEP_VALUETYPES_NONE, new char[]{}, "");
	    }

	    /**
	     * @deprecated Use startNode(String name, Class clazz) instead.
	     */

	    public void startNode(String name) {
	        startNode(name, null);
	    }

	    public void startNode(String name, Class clazz) {
	    	 Node currNode = (Node)elementStack.peek();
	    	 createPathElement(name, clazz, currNode);
	    	 
	        if (currNode == null) {
	            writer.write("{");
	        }
	        if (currNode != null && currNode.fieldAlready) {
	            writer.write(",");
	            readyForNewLine = true;
	        }
	        tagIsEmpty = false;
	        finishTag();
	        if (currNode == null
	            || currNode.clazz == null
	            || (currNode.clazz != null && !currNode.isCollection  && !currNode.isMapEntry)) {
	            if (currNode != null && !currNode.fieldAlready) {
	                writer.write("{");
	                readyForNewLine = true;
	                finishTag();
	            }
	            writer.write("\"");
	            writer.write(name);
	            writer.write("\": ");
	        }
	        if (isCollection(clazz)) {
	            writer.write("[");
	            readyForNewLine = true;
	        }
	        if (currNode != null) {
	            currNode.fieldAlready = true;
	        }
	        elementStack.push(new Node(name, clazz));
	        depth++ ;
	        tagIsEmpty = true;
	    }

		private void createPathElement(String name, Class clazz, Node currNode) {
			CommonsBeanUtilsPathElement element = null;
	    	 if(currNode != null)
	    		 element = new CommonsBeanUtilsPathElement(name);
	    	 else
	    		 element = new CommonsBeanUtilsPathElement("map");
	    	 
	    	 
	    	 if(isMap( clazz))
	    		 element.setTyp(CommonsBeanUtilsPathElement.Typ.MAP);
	    	 else if(isCollection( clazz)){
	    		 element.setTyp(CommonsBeanUtilsPathElement.Typ.ARRAY);
	    	 }else
	    		 element.setTyp(CommonsBeanUtilsPathElement.Typ.SIMPEL);
	    	 
	
	    
	    	 pathTracker.pushElement(name, element);
		}

	    public class Node {
	        public final String name;
	        public final Class clazz;
	        public boolean fieldAlready;
	        public boolean isCollection;
	        public boolean isMap;
	        public boolean isMapEntry;

	        public Node(String name, Class clazz) {
	            this.name = name;
	            this.clazz = clazz;
	            isCollection = isCollection(clazz);
	            isMap = isMap(clazz);
	            isMapEntry = isMapEntry(clazz);
	        }
	    }

	    public void setValue(String text) {
	        readyForNewLine = false;
	        tagIsEmpty = false;
	        finishTag();
	        writeText(writer, text);
	    }
	    
	    public void setValue(String text,  boolean converted) {
	    	readyForNewLine = false;
	        tagIsEmpty = false;
	        finishTag();
	    	writeText(text, String.class);
		}

	   public void addAttribute(String key, String value) {
	       	throw new UnsupportedOperationException("There is no need to add an Attribute!");
	    }

	    protected void writeAttributeValue(QuickWriter writer, String text) {
	    	throw new UnsupportedOperationException("There is no need to add an Attribute!");
	    }

	    private void writeText(QuickWriter writer, String text) {
	        Node foo = (Node)elementStack.peek();

	        writeText(text, foo.clazz);
	    }

	    protected void writeText(String text, Class clazz) {
	        if (needsQuotes(clazz)) {
	            writer.write("\"");
	        }
	        if ((clazz == Character.class || clazz == Character.TYPE) && "".equals(text)) {
	            text = "\0";
	        }

	        int length = text.length();
	        for (int i = 0; i < length; i++ ) {
	            char c = text.charAt(i);
	            switch (c) {
	            case '"':
	                this.writer.write("\\\"");
	                break;
	            case '\\':
	                this.writer.write("\\\\");
	                break;
	            default:
	                if (c > 0x1f) {
	                    this.writer.write(c);
	                } else {
	                    this.writer.write("\\u");
	                    String hex = "000" + Integer.toHexString(c);
	                    this.writer.write(hex.substring(hex.length() - 4));
	                }
	            }
	        }
	       
	        if (needsQuotes(clazz)) {
	            writer.write("\"");
	        }
	    }

	    protected boolean isMapEntry(Class clazz) {
	    	return clazz != null && 
            Map.Entry.class.isAssignableFrom(clazz) ;
	    }
	    
	    protected boolean isMap(Class clazz) {
	    	return clazz != null && 
           Map.class.isAssignableFrom(clazz);
	    }
	    
	    protected boolean isCollection(Class clazz) {
	    	return clazz != null
            && (Collection.class.isAssignableFrom(clazz)
                || clazz.isArray()
                 );
	    }

	    private boolean needsQuotes(Class clazz) {

	        if(JsonStringWriter.MODE_KEEP_VALUETYPES_BOOLEANS.equals(keepValueTypesMode)){
	        	clazz = clazz != null && clazz.isPrimitive() ? clazz : Primitives.unbox(clazz);
	 	        if(Boolean.class.equals(clazz))
	        		return false;
	        	else
	        		return true;
	 	        
	    	}else  if(JsonStringWriter.MODE_KEEP_VALUETYPES_ALL.equals(keepValueTypesMode)){
	    		clazz = clazz != null && clazz.isPrimitive() ? clazz : Primitives.unbox(clazz);
	    		return clazz == null || clazz == Character.TYPE;
	    	}else{
	    		
	    		return true;
	    		
	    	}
	    }

	    public void endNode() {
	        depth-- ;
	        Node node = (Node)elementStack.pop();
	        if (node.clazz != null && node.isCollection) {
	            if (node.fieldAlready) {
	                readyForNewLine = true;
	            }
	            finishTag();
	            writer.write("]");
	        } else if (node.clazz != null && node.isMapEntry) {
	            if (node.fieldAlready) {
	                readyForNewLine = true;
	            }
	            finishTag();
	            writer.write("}");
	        } else if (tagIsEmpty) {
	            readyForNewLine = false;
	            writer.write("null");
	            finishTag();
	        } else {
	            finishTag();
	            if (node.fieldAlready) {
	                writer.write("}");
	            }
	        }
	        readyForNewLine = true;
	        if (depth == 0) {
	            writer.write("}");
	            writer.flush();
	        }
	        pathTracker.popElement();
	    }

	    private void finishTag() {
	        if (readyForNewLine) {
	            endOfLine();
	        }
	        readyForNewLine = false;
	        tagIsEmpty = false;
	    }

	    protected void endOfLine() {
	        writer.write(newLine);
	        for (int i = 0; i < depth; i++ ) {
	            writer.write(lineIndenter);
	        }
	    }

	    public void flush() {
	        writer.flush();
	    }

	    public void close() {
	        writer.close();
	    }

	    public HierarchicalStreamWriter underlyingWriter() {
	        return this;
	    }

		public CommonsBeanUtilslPathTracker getCommonsBeanUtilslPathTracker() {
			return pathTracker;
		}

		public void setCommonsBeanUtilslPathTracker(CommonsBeanUtilslPathTracker pathTracker) {
			this.pathTracker = pathTracker;
		}

		public char[] getLineIndenter() {
			return lineIndenter;
		}

		public String getNewLine() {
			return newLine;
		}

		
	    
	    
	    
	}

