/*
 * Copyright (c) 2002-2022 Gargoyle Software Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.gargoylesoftware.htmlunit.html;

import static com.gargoylesoftware.htmlunit.BrowserVersionFeatures.HTMLOUTPUT_WILL_VALIDATE_ALWAYS_TRUE;

import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import com.gargoylesoftware.htmlunit.SgmlPage;

/**
 * Wrapper for the HTML element "output".
 *
 * @author Ronald Brill
 * @author Frank Danek
 */
public class HtmlOutput extends HtmlElement implements LabelableElement, ValidatableElement {

    /** The HTML tag represented by this element. */
    public static final String TAG_NAME = "output";

    private String customValidity_;

    /**
     * Creates a new instance.
     *
     * @param qualifiedName the qualified name of the element type to instantiate
     * @param page the page that contains this element
     * @param attributes the initial attributes
     */
    HtmlOutput(final String qualifiedName, final SgmlPage page,
            final Map<String, DomAttr> attributes) {
        super(qualifiedName, page, attributes);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public DisplayStyle getDefaultStyleDisplay() {
        return DisplayStyle.INLINE;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean willValidate() {
        return hasFeature(HTMLOUTPUT_WILL_VALIDATE_ALWAYS_TRUE);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setCustomValidity(final String message) {
        customValidity_ = message;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isValid() {
        return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasBadInputValidityState() {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isCustomErrorValidityState() {
        return !StringUtils.isEmpty(customValidity_);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasPatternMismatchValidityState() {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isStepMismatchValidityState() {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isTooLongValidityState() {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isTooShortValidityState() {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasTypeMismatchValidityState() {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasRangeOverflowValidityState() {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasRangeUnderflowValidityState() {
        return false;
    }

    @Override
    public boolean isValidValidityState() {
        return !isCustomErrorValidityState();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isValueMissingValidityState() {
        return false;
    }
}
