/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang.Validate;

import net.sourceforge.wurfl.core.request.DefaultWURFLRequestFactory;
import net.sourceforge.wurfl.core.request.WURFLRequest;
import net.sourceforge.wurfl.core.request.WURFLRequestFactory;

/**
 * This is the default WURFLManager implementation.
 * 
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * 
 * @version $Id: DefaultWURFLManager.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public class DefaultWURFLManager implements WURFLManager {

	/**
	 * The WURFLService implementation used to perform the WURFL main operation.
	 */
	private WURFLService service;

	/**
	 * The WURFLRequestFactory implementation used to build a WURFLRequest given
	 * to <code>wurflService</code> to perform main operation.
	 */
	private WURFLRequestFactory requestFactory;

	// Constructors *******************************************************

	/**
	 * This constructor build a WURFLManagerImpl from a WURFLService instance
	 * and WURFLRequestFactory instance.
	 * 
	 * @param service
	 *            The service used to match WURFLRequest
	 * @param requestFactory
	 *            The WURFLRequestFactory used to create WURFLRequest
	 */
	public DefaultWURFLManager(WURFLService service,
			WURFLRequestFactory requestFactory) {

		Validate.notNull(service, "The service must be not null");
		Validate.notNull(requestFactory, "The requestFactory must be not null");

		this.service = service;
		this.requestFactory = requestFactory;
	}

	/**
	 * This constructor build a WURFLManagerImpl from a WURFLService instance
	 * using DefaultWURFLRequestFactory as WURFLRequestFactory.
	 * 
	 * @param service
	 *            The service used to match WURFLRequest
	 */
	public DefaultWURFLManager(WURFLService service) {

		this(service, new DefaultWURFLRequestFactory());
	}

	// Business methods ***************************************************

	/**
	 * {@inheritDoc}
	 */
	public Device getDeviceForRequest(HttpServletRequest request) {

		Validate.notNull(request, "The request must be not null");

		WURFLRequest wurflRequest = requestFactory.createRequest(request);

		return getDeviceForRequest(wurflRequest);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <p>
	 * This method skip the use of WURFLRequestFactory.
	 * <p>
	 */
	public Device getDeviceForRequest(WURFLRequest request) {

		Validate.notNull(request, "The request must be not null");

		return service.getDeviceForRequest(request);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <p>
	 * This method uses {@link WURFLRequestFactory#createRequest(String)}.
	 * </p>
	 */
	public Device getDeviceForRequest(String userAgent) {

		Validate.notNull(userAgent, "The userAgent must be not null");

		WURFLRequest wurflRequest = requestFactory.createRequest(userAgent);

		return getDeviceForRequest(wurflRequest);
	}

}
