/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core.cache;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.Validate;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import edu.emory.mathcs.backport.java.util.concurrent.ConcurrentHashMap;

/**
 * {@link CacheProvider} HashMap implementation.
 * 
 * <p>
 * The items are cached by mean of {@link HashMap}, they never evict.
 * </p>
 * 
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * 
 * @version $Id: HashMapCacheProvider.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public class HashMapCacheProvider implements CacheProvider {

	/** The hashMap containing the cached objects */
	private java.util.Map cache;

	/** Initial capacity of hashMap */
	private int initialCapacity = 6000;

	/** load factor of hashmap */
	private float loadFactor = 0.75f;

	/** load factor of hashmap */
	private int concurrentWrites = 16;

	/** Logger */
	private static final Log LOG = LogFactory.getLog(HashMapCacheProvider.class);

    // Constructors *******************************************************

	/** Default constructor */
	public HashMapCacheProvider() {
		this(60000);
	}

	/**
	 * Constructor by the initial capacity.
	 * 
	 * @see ConcurrentHashMap#ConcurrentHashMap(int).
	 * 
	 * @param The
	 *            initial capacity of HashMap.
	 */
	public HashMapCacheProvider(int initialCapacity) {
		this(initialCapacity, .75f);
	}

	/**
	 * Constructor by the initial capacity and load factor.
	 * 
	 * @see ConcurrentHashMap#ConcurrentHashMap(int, float).
	 * 
	 * @param initialCapacity
	 *            The hashMap initial capacity.
	 * @param loadFactor
	 *            The hashMap load factor.
	 */
	public HashMapCacheProvider(int initialCapacity, float loadFactor) {
		this(initialCapacity, loadFactor, 16);
	}

	/**
	 * Constructor by the initial capacity, loadFactor and maximum concurrent
	 * writes
	 * 
	 * @see ConcurrentHashMap#ConcurrentHashMap(int, float, int).
	 * 
	 * @param initialCapacity
	 *            The hashMap initial capacity.
	 * @param loadFactor
	 *            The hashMap load factor.
	 * @param concurrentWrites
	 *            The maximum thread can write the cache.
	 */
	public HashMapCacheProvider(int initialCapacity, float loadFactor,
			int concurrentWrites) {

		this.initialCapacity = initialCapacity;
		this.loadFactor = loadFactor;
		this.concurrentWrites = concurrentWrites;

		cache = createCache();

		if (LOG.isInfoEnabled()) {
			StringBuffer logBuffer = new StringBuffer(
					"Created HashMapCacheProvider with initial capacity: ");
			logBuffer.append(initialCapacity);
			logBuffer.append(" load factor: ");
			logBuffer.append(loadFactor);
			logBuffer.append(" concurrent writes: ");
			logBuffer.append(concurrentWrites);

			LOG.info(logBuffer.toString());
		}
	}

	// Access methods *****************************************************

	/**
	 * Return the initial capacity of the hashMap.
	 * 
	 * @return the initial capacity of the hashMap.
	 * @see ConcurrentHashMap#ConcurrentHashMap(int, float, int)
	 */
	public int getInitialCapacity() {
		return initialCapacity;
	}

	/**
	 * Return the hashMap load factor.
	 * 
	 * @return the hashMap load factor.
	 * @see ConcurrentHashMap#ConcurrentHashMap(int, float, int)
	 */
	public float getLoadFactor() {
		return loadFactor;
	}

	/**
	 * Return the maximum threads can write the cache.
	 * 
	 * @return a int representing the maximum thread can write the cache.
	 * @see ConcurrentHashMap#ConcurrentHashMap(int, float, int)
	 */
	public int getConcurrentWrites() {
		return concurrentWrites;
	}

	// CacheProvider implementation ***************************************

	/**
	 * {@inheritDoc}
	 */
	public void clear() {
		cache.clear();

		LOG.debug("Cache erased");
	}

	/**
	 * {@inheritDoc}
	 */
	public Object get(Object key) {

		Validate.notNull(key, "The key is null");

		Object target = cache.get(key);

		if (LOG.isDebugEnabled()) {
			if (target == null) {
				LOG.debug("The key: " + key + " is not in cache");
			} else {
				LOG.debug("Retrieving from cache object: " + target
						+ " by key: " + key);
			}
		}

		return target;
	}

	/**
	 * {@inheritDoc}
	 */
	public void put(Object key, Object target) {
		
		Validate.notNull(key, "The key is null");

		if (LOG.isDebugEnabled()) {
			LOG.debug("Storing object: " + target + " with key: " + key);
		}

		cache.put(key, target);
	}

	/**
	 * Factory method to create cache Map. Subclass can override this method to
	 * change the backing HashMap implementation.
	 */
	protected Map createCache() {
		Map cacheMap = new ConcurrentHashMap(initialCapacity, loadFactor,
				concurrentWrites);

		return cacheMap;
	}
	
	// Commons methods ****************************************************

	public String toString() {
		return new ToStringBuilder(this).append(cache).toString();
	}
}
