/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core.cache;

import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections.map.LRUMap;

/**
 * CacheProvider {@link LRUMap} implementation.
 * 
 * <p>
 * The cached items are stored using LRUMap. LRUMap is a map using LRU algorithm to evict items.
 * </p>
 * 
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * 
 * @version $Id: LRUMapCacheProvider.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public class LRUMapCacheProvider implements CacheProvider {

	/** The {@link LRUMap} instance */
	private Map cache;
	
	

	/**
	 * Constructor by LRUMap parameters. To more detail look at LRUMap documentation.
	 * 
	 * @param maxSize
	 *            The max size of cache.
	 * @param scanUntilRemovable
	 *            scan until a removeable entry is found, default false.
	 * @see LRUMap#LRUMap(int, boolean)
	 */
	public LRUMapCacheProvider(int maxSize, boolean scanUntilRemovable) {
		this.cache = MapUtils.synchronizedMap(new LRUMap(maxSize, scanUntilRemovable));
	
	}

	/**
	 * Constructor by maxSize parameter.
	 * 
	 * @param maxSize
	 *            The max size of cache.
	 * @see LRUMap#LRUMap(int)
	 */
	public LRUMapCacheProvider(int maxSize) {
		this.cache = MapUtils.synchronizedMap(new LRUMap(maxSize));
	}

	/**
	 * Default constructor.
	 */
	public LRUMapCacheProvider() {
		this.cache =  MapUtils.synchronizedMap(new LRUMap());
	}

	public void clear() {
		cache.clear();
	}

	public Object get(Object key) {
		return cache.get(key);
	}

	public void put(Object key, Object value) {
		cache.put(key, value);
	}

}
