/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core.handlers.classifiers;

import org.apache.commons.lang.text.StrBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import net.sourceforge.wurfl.core.handlers.Handler;

/**
 * {@link Filter} base implementation.
 * 
 * <p>
 * The {@link Handler} decides if this object can handle the given user-agent.
 * Stores the classified device in held {@link FilteredDevices} instance.
 * </p>
 * 
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * 
 * @version $Id: BaseFilter.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public class BaseFilter implements Filter {

	/** The held Handler */
	private final Handler handler;
	
	/** Classified device identifiers */
	private FilteredDevices filteredDevices;

	/** Log */
	private static final Log LOG = LogFactory.getLog(BaseFilter.class);

	/**
	 * Constructor by {@link Handler} and {@link FilteredDevices}.
	 * 
	 * @param handler
	 *            The handler used by this object.
	 *
	 */
	public BaseFilter(Handler handler) {
		this.handler = handler;
		this.filteredDevices = new FilteredDevices();
	}

	/**
	 * Use the Handler to know if is possible to handle the user-agent.
	 */
	public boolean canHandle(String userAgent) {
		return handler.canHandle(userAgent);
	}

	/**
	 * Add the given device id and user-agent to collected devices.
	 */
	public void filter(String userAgent, String deviceID) {

		if (LOG.isTraceEnabled()) {
			StrBuilder logBuilder = new StrBuilder();
			logBuilder.append("Putting Device: ").append(deviceID).append(
					" into filter");
			LOG.trace(logBuilder.toString());
		}
		
		this.filteredDevices = filteredDevices.putDevice(handler.normalize(userAgent), deviceID);
	}

	/**
	 * Return the held {@link FilteredDevices}.
	 */
	public FilteredDevices getFilteredDevices() {
		return filteredDevices;
	}
		
}
