/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core.handlers.matchers;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.SortedSet;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import net.sourceforge.wurfl.core.Constants;
import net.sourceforge.wurfl.core.handlers.Handler;
import net.sourceforge.wurfl.core.handlers.classifiers.FilteredDevices;
import net.sourceforge.wurfl.core.request.WURFLRequest;
import net.sourceforge.wurfl.core.utils.StringUtils;
import net.sourceforge.wurfl.core.utils.UserAgentUtils;

/**
 * Abstract implementation of {@link Matcher}.
 * <p/>
 * <p>
 * It provides basic functionalities of matching. The matching strategy is:
 * <ul>
 * <li>Direct matching</li>
 * <li>Conclusive matching</li>
 * <li>Recovery matching</li>
 * <li>CatchAll matching</li>
 * </ul>
 * </p>
 *
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * @version $Id: AbstractMatcher.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public abstract class AbstractMatcher implements Matcher {

    /**
     * The handler used to determine if this matcher can handle the request
     */
    private final Handler handler;

    /**
     * Log
     */
    protected final Log logger = LogFactory.getLog(getClass());

    /**
     * Log undetected
     */
    protected final Log undetected = LogFactory
            .getLog(Constants.UNDETECTED_WURFL_DEVICES);

    /**
     * Log detected
     */
    protected final Log detected = LogFactory
            .getLog(Constants.DETECTED_WURFL_DEVICES);

    /**
     * Build AbstractMatcher by handler and collected devices by user-agent.
     *
     * @param handler The handler used to decide if this class can handle the
     *                request.
     */
    public AbstractMatcher(Handler handler) {
        this.handler = handler;
    }

    public boolean canHandle(WURFLRequest request) {
        return this.handler.canHandle(request.getUserAgent());
    }

    /**
     * Applies the matching strategy. this strategy is valid for every subclass.
     * <p>
     * The strategy is:
     * <ul>
     * <li>Direct matching</li>
     * <li>Conclusive matching</li>
     * <li>Recovery matching</li>
     * <li>CatchAll matching</li>
     * </ul>
     * </p>
     */
    public final String match(WURFLRequest request,
                              FilteredDevices filteredDevices) {

        String userAgent = request.getUserAgent();
        if (org.apache.commons.lang.StringUtils.isBlank(userAgent)) {
            return Constants.GENERIC;
        }

        // 1) Try A direct Match
        log("Applying Direct Match on UA: " + userAgent);
        String deviceID = filteredDevices.getDevice(userAgent);
        if (!isBlankOrGeneric(deviceID)) {
            return deviceID;
        }

        // 2) Try A Conclusive Match
        log("Applying Conclusive Match on UA: " + userAgent);
        deviceID = applyConclusiveMatch(request, filteredDevices);

        if (!isBlankOrGeneric(deviceID)) {
            return deviceID;
        }

        // 3) Try A Recovery Match
        log("Applying Recovery Match on UA: " + userAgent);
        deviceID = applyRecoveryMatch(request, filteredDevices);

        if (!isBlankOrGeneric(deviceID)) {
            return deviceID;
        }

        // 4) CatchAll Match
        log("Applying CatchAll Recovery Match on UA: "  + userAgent);

        deviceID = applyCatchAllRecoveryMatch(request, filteredDevices);

        assert deviceID != null;

        return deviceID;
    }

    private void log(String message) {
        if (logger.isDebugEnabled()) {
            logger.debug(message);
        }
    }

    /**
     * Apply conclusive matching strategy, at least returns <code>generic</code>.
     *
     * @param request The request used to applying matching strategy.
     * @return The matched device identifiers.
     */
    protected String applyConclusiveMatch(WURFLRequest request,
                                          FilteredDevices filteredDevices) {

        // NORMALIZE USER AGENT;
        final String normalizedUserAgent = handler.normalize(request.getUserAgent());

        if (logger.isTraceEnabled()) {
            logger.trace("Matching UA: " + normalizedUserAgent
                    + " against devices: " + filteredDevices.getUserAgents());
        }

        String match = lookForMatchingUserAgent(
                filteredDevices.getUserAgents(), normalizedUserAgent);

        // logger undetected
        if (undetected.isDebugEnabled() && match == null) {
            undetected.debug(request.getUserAgent() + ":"
                    + request.getUserAgentProfile());
        }

        String deviceID = Constants.GENERIC;

        if (null != match) {

            if (detected.isDebugEnabled()) {
                detected.debug(request.getUserAgent() + ":"
                        + request.getUserAgentProfile() + ":" + match);
            }

            deviceID = filteredDevices.getDevice(match);
        }

        // deviceID maybe null if filteredDevices contains null values
        if (deviceID == null) {

            if (logger.isWarnEnabled()) {
                logger.warn("filteredDevices do not contain UA: " + match
                        + " return generic");
            }

            deviceID = Constants.GENERIC;
        }

        return deviceID;

    }

    /**
     * Match the given user-agent to collected user-agents.
     * <p/>
     * <p>
     * Apply RIS(FS) to default.
     * </p>
     *
     * @param userAgentsSet A SortedSet of String containing user-agent string naturally
     *                      ordered.
     * @param userAgent     The user-agent to match against.
     * @return The collected user-agent matching with given user-agent.
     */
    protected String lookForMatchingUserAgent(SortedSet userAgentsSet,
                                              String userAgent) {

        int tollerance = StringUtils.firstSlash(userAgent);

        if (logger.isDebugEnabled()) {
            logger.debug("Applying RIS(FS) UA: " + userAgent);
        }

        return StringUtils.risMatch(userAgentsSet, userAgent, tollerance);

    }

    /**
     * Match the given request using the recovery matching strategy.
     *
     * @param request The request to match.
     * @return The device identifier matched from request, at least returns
     *         <code>generic</code>.
     */
    protected String applyRecoveryMatch(WURFLRequest request,
                                        FilteredDevices filteredDevices) {
        return Constants.GENERIC;
    }


    /**
     * <p>
     * Match the given request using the CatchAll strategy. The subclass can
     * override this method, but must returns "generic" device al least.
     * </p>
     * <p>
     * <b>
     * TODO this method assumes the wurfl.xml + patch files contains the
     * hard coded device's ids.
     * </b>
     * </p>
     *
     * @param request The request to match.
     * @return the matched device's id, at least "generic".
     */
    protected String applyCatchAllRecoveryMatch(WURFLRequest request,
                                                FilteredDevices filteredDevices) {

        return CatchAllRecoveryMap.deviceIdOf(request.getUserAgent());
    }

    private static final class CatchAllRecoveryMap {
        private CatchAllRecoveryMap() {
        }

        static final Map CATH_ALL_RECOVERY_MAP = new LinkedHashMap();

        static {
            // Openwave
            CATH_ALL_RECOVERY_MAP.put("UP.Browser/7.2", "opwv_v72_generic");
            CATH_ALL_RECOVERY_MAP.put("UP.Browser/7", "opwv_v7_generic");
            CATH_ALL_RECOVERY_MAP.put("UP.Browser/6.2", "opwv_v62_generic");
            CATH_ALL_RECOVERY_MAP.put("UP.Browser/6", "opwv_v6_generic");
            CATH_ALL_RECOVERY_MAP.put("UP.Browser/5", "upgui_generic");
            CATH_ALL_RECOVERY_MAP.put("UP.Browser/4", "uptext_generic");
            CATH_ALL_RECOVERY_MAP.put("UP.Browser/3", "uptext_generic");


            // Series 60
            CATH_ALL_RECOVERY_MAP.put("Series60", "nokia_generic_series60");

            // Access/Net Front
            CATH_ALL_RECOVERY_MAP.put("NetFront/3.0", "netfront_ver3");
            CATH_ALL_RECOVERY_MAP.put("ACS-NF/3.0", "netfront_ver3");
            CATH_ALL_RECOVERY_MAP.put("NetFront/3.1", "netfront_ver3_1");
            CATH_ALL_RECOVERY_MAP.put("ACS-NF/3.1", "netfront_ver3_1");
            CATH_ALL_RECOVERY_MAP.put("NetFront/3.2", "netfront_ver3_2");
            CATH_ALL_RECOVERY_MAP.put("ACS-NF/3.2", "netfront_ver3_2");
            CATH_ALL_RECOVERY_MAP.put("NetFront/3.3", "netfront_ver3_3");
            CATH_ALL_RECOVERY_MAP.put("ACS-NF/3.3", "netfront_ver3_3");
            CATH_ALL_RECOVERY_MAP.put("NetFront/3.4", "netfront_ver3_4");
            CATH_ALL_RECOVERY_MAP.put("NetFront/3.5", "netfront_ver3_5");

            // Windows CE
            CATH_ALL_RECOVERY_MAP.put("Windows CE", "ms_mobile_browser_ver1");


            // web browsers?
            CATH_ALL_RECOVERY_MAP.put("Mozilla/4.0", "generic_web_browser");
            CATH_ALL_RECOVERY_MAP.put("Mozilla/5.0", "generic_web_browser");
            CATH_ALL_RECOVERY_MAP.put("Mozilla/5.0", "generic_web_browser");

            // Generic XHTML
            CATH_ALL_RECOVERY_MAP.put("Mozilla/", Constants.GENERIC_XHTML);
            CATH_ALL_RECOVERY_MAP.put("ObigoInternetBrowser/Q03C", Constants.GENERIC_XHTML);
            CATH_ALL_RECOVERY_MAP.put("AU-MIC/2", Constants.GENERIC_XHTML);
            CATH_ALL_RECOVERY_MAP.put("AU-MIC-", Constants.GENERIC_XHTML);
            CATH_ALL_RECOVERY_MAP.put("AU-OBIGO/", Constants.GENERIC_XHTML);
            CATH_ALL_RECOVERY_MAP.put("Obigo/Q03", Constants.GENERIC_XHTML);
            CATH_ALL_RECOVERY_MAP.put("Obigo/Q04", Constants.GENERIC_XHTML);
            CATH_ALL_RECOVERY_MAP.put("ObigoInternetBrowser/2", Constants.GENERIC_XHTML);
            CATH_ALL_RECOVERY_MAP.put("Teleca Q03B1", Constants.GENERIC_XHTML);


            // Opera Mini
            CATH_ALL_RECOVERY_MAP.put("Opera Mini/1", "opera_mini_ver1");
            CATH_ALL_RECOVERY_MAP.put("Opera Mini/2", "opera_mini_ver2");
            CATH_ALL_RECOVERY_MAP.put("Opera Mini/3", "opera_mini_ver3");
            CATH_ALL_RECOVERY_MAP.put("Opera Mini/4", "opera_mini_ver4");

            // DoCoMo
            CATH_ALL_RECOVERY_MAP.put("DoCoMo", "docomo_generic_jap_ver1");
            CATH_ALL_RECOVERY_MAP.put("KDDI", "docomo_generic_jap_ver1");
        }

        /**
         * Return the hardcoded device id or generic if the given user agent contains one of
         * the keys
         *
         * @param userAgent
         * @return
         */
        public static String deviceIdOf(String userAgent) {
            String key = (String) CollectionUtils.find(CATH_ALL_RECOVERY_MAP.keySet(), UserAgentUtils.isContainedIn(userAgent));
            if (key != null) {
                return (String) CATH_ALL_RECOVERY_MAP.get(key);
            }
            return Constants.GENERIC;
        }
    }

    /**
     * Return if the given device identifier is null or <code>generic</code>.
     *
     * @param deviceID The device identifier to check.
     * @return True if <code>deviceID</code> is null or <code>generic</code>.
     */
    private boolean isBlankOrGeneric(String deviceID) {
        return org.apache.commons.lang.StringUtils.isBlank(deviceID) || Constants.GENERIC.equals(deviceID);
    }

}
