/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core.handlers.matchers;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import net.sourceforge.wurfl.core.handlers.HandlersFactory;
import net.sourceforge.wurfl.core.handlers.classifiers.Filter;
import net.sourceforge.wurfl.core.handlers.classifiers.FilterChain;
import net.sourceforge.wurfl.core.handlers.classifiers.FilterChainFactory;
import net.sourceforge.wurfl.core.handlers.classifiers.FilteredDevices;
import net.sourceforge.wurfl.core.request.WURFLRequest;
import net.sourceforge.wurfl.core.resource.ModelDevice;
import net.sourceforge.wurfl.core.resource.WURFLModel;

/**
 * This class is responsible to manage the matching process. It is a facade
 * class of {@link Matcher}s an {@link Filter}s.
 * 
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * 
 * @version $Id: MatcherManager.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public class MatcherManager {

	/** The matcher chain-of-responsibility used to match user-agents */
	private final MatcherChain matcherChain;

	/** The Filter chain used to classify devices */
	private final FilterChain filterChain;


	// Constructors *******************************************************

	/**
	 * Build a MatcherManager by model, matcher chain and filter chain.
	 * 
	 * @param model
	 *            The WURFLModel used to get devices from.
	 * @param filterChain
	 *            The used FilterChain.
	 * @param matcherChain
	 *            The used MatcherChain.
	 */
	public MatcherManager(WURFLModel model, FilterChain filterChain,
			MatcherChain matcherChain) {

		this.filterChain = filterChain;
		this.matcherChain = matcherChain;

		addFilteredDevices(model.getAllDevices());
	}

	/**
	 * Build a MatcherManager by model.
	 * 
	 * @param model
	 *            The WURFLModel used to get devices from.
	 */
	public MatcherManager(WURFLModel model) {

		Map handlers = new HandlersFactory().create();

		filterChain = new FilterChainFactory(handlers).create();
		matcherChain = new MatcherChainFactory(handlers).create();

		addFilteredDevices(model.getAllDevices());
	}

	// Business methods ***************************************************

	/**
	 * Match the request to obtain a device identifier. For more info look at
	 * {@link Matcher#match(WURFLRequest)}.
	 * 
	 * @param request
	 *            The {@link WURFLRequest} to match.
	 * @return A valid device identifier, <code>generic</code> device at least.
	 */
	public String matchRequest(WURFLRequest request) {

		FilteredDevices filteredDevices = filterChain
				.getFilteredDevices(request.getUserAgent());

		return matcherChain.match(request, filteredDevices);

	}

	/**
	 * Add devices to be collected.
	 * 
	 * @param devices
	 *            Set of ModelDevice to be collected by matchers.
	 */
	protected void addFilteredDevices(Set/* ModelDevice */devices) {

		for (Iterator iterator = devices.iterator(); iterator.hasNext();) {

			ModelDevice device = (ModelDevice) iterator.next();
			String userAgent = device.getUserAgent();
			String deviceId = device.getID();

			filterChain.filter(userAgent, deviceId);
		}

	}

}
