/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core.resource;

import java.util.List;

import org.apache.commons.lang.text.StrBuilder;

import net.sourceforge.wurfl.core.utils.StringUtils;

/**
 * Generic hierarchy consistency exception.
 * 
 * <p>
 * This Exception is thrown when a inconsistent hierarchy is detected in
 * {@link WURFLModel}.
 * </p>
 * 
 * <p>
 * Device hierarchy is defined as ordered list from <i>generic</i> to root
 * device.<br/>
 * Example:<br/>
 * <i>generic</i> -> ... -> <i>root</i>
 * </p>
 * 
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * 
 * @version $Id: HierarchyConsistencyException.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public abstract class HierarchyConsistencyException extends
		WURFLConsistencyException {

	/** Serial */
	private static final long serialVersionUID = 10L;

	/** The ModelDevice hierarchy source of this Exception */
	private List hierarchy;

	/**
	 * Build this exception from a ModelDevice hierarchy and message.
	 * 
	 * @param hierarchy
	 *            a {@link List} representing the object hierarchy.
	 * @param message
	 *            The message to display.
	 */
	public HierarchyConsistencyException(List/* ModelDevice */hierarchy,
			String message) {
		super(message);

		this.hierarchy = hierarchy;
	}

	/**
	 * Build this exception from a ModelDevice hierarchy.
	 * 
	 * @param hierarchy
	 *            a {@link List} representing the object hierarchy.
	 * @param message
	 *            The message to display.
	 */
	public HierarchyConsistencyException(List/* ModelDevice */hierarchy) {
		super(new StrBuilder("Consistency exception in hierarchy: [").append(
				StringUtils.hierarchyAsString(hierarchy)).append("]")
				.toString());

		this.hierarchy = hierarchy;
	}

	/**
	 * Returns the hierarchy source of this exception.
	 * 
	 * @return The held hierarchy source of exception.
	 */
	public List getHierarchy() {
		return hierarchy;
	}

}
