/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core.resource;

import java.util.EventObject;
import java.util.Set;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;

/**
 * The event fired by the {@link WURFLModel}.
 * 
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * 
 * @version $Id: WURFLModelEvent.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public class WURFLModelEvent extends EventObject {

	/** Serial */
	private static final long serialVersionUID = 10L;

	/** Constant for device added event */
	public static final int DEVICE_ADDED = 0;

	/** Constant for device updated event */
	public static final int DEVICE_UPDATED = 2;

	/** type of event : in DEVICE_ADDED or DEVICE_UPDATED */
	private int type;

	/** The devices target of this event: the devices added, removed or updated */
	private Set devices;

	/**
	 * Build a {@link WURFLModelEvent} from {@link WURFLModel}, type and target
	 * devices.
	 * 
	 * @param source
	 *            The model firing this event.
	 * @param type
	 *            The type of event fired in DEVICE_ADDED, DEVICE_REMOVED,
	 *            DEVICE_UPDATED.
	 * @param devices
	 *            The {@link ModelDevice}s target of this event.
	 */
	public WURFLModelEvent(WURFLModel source, int type, Set/*ModelDevice*/ devices) {
		super(source);

		this.type = type;
		this.devices = devices;
	}

	/**
	 * Return the event type in DEVICE_ADDED, DEVICE_REMOVED or DEVICE_UPDATED.
	 * 
	 * @return int Type of this event
	 */
	public int getType() {
		return type;
	}

	/**
	 * Return the target devices of this event.
	 * 
	 * @return a {@link Set} of {@link ModelDevice} target of this event.
	 */
	public Set/*ModelDevice*/ getDevices() {
		return devices;
	}

	// Commons methods ****************************************************

	public String toString() {
		return new ToStringBuilder(this).append(type).append(devices)
				.toString();
	}

	public boolean equals(Object obj) {
		EqualsBuilder eb = new EqualsBuilder();

		eb.appendSuper(getClass().isInstance(obj));

		if (eb.isEquals()) {
			WURFLModelEvent other = (WURFLModelEvent) obj;
			eb.append(source, other.source).append(type, other.type).append(
					devices, other.devices);
		}

		return eb.isEquals();
	}

	public int hashCode() {

		return new HashCodeBuilder(53, 77).append(getClass()).append(source)
				.append(type).append(devices).toHashCode();
	}

}
