/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

/* $Id: StringUtils.java 432 2010-05-06 12:12:53Z filippo.deluca $ 
 * This file is released under the GNU General Public License. 
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2009 WURFL-Pro S.r.l.
 */
package net.sourceforge.wurfl.core.utils;

import java.util.List;
import java.util.SortedSet;

import org.apache.commons.lang.text.StrBuilder;

import net.sourceforge.wurfl.core.handlers.matchers.strategy.LDMatcher;
import net.sourceforge.wurfl.core.handlers.matchers.strategy.RISMatcher;

/**
 * Strings matching utility class.
 * <p/>
 * <p>
 * This class contains the utility methods for the strings.
 * </p>
 *
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * @version $Id: StringUtils.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public final class StringUtils {

    private StringUtils() {
    }

    /**
     * Return the position of first slash of the userAgent string.
     *
     * @param target A string to search first slash
     * @return A index of first slash in the string or, if the string does not
     *         contain a slash, the string length
     */
    public static int firstSlash(String target) {

        return indexOfOrLength(target, "/");
    }

    /**
     * Return the index of second of the given string.
     *
     * @param target A string on which find second slash.
     * @return A index of second slash in the string.
     */
    public static int secondSlash(String target) {

        return ordinalIndexOfOrLength(target, "/", 2);

    }

    public static int firstSemiColon(String target) {
        return ordinalIndexOfOrLength(target, ";", 1);
    }

    /**
     * Return the position of first space of the given string.
     *
     * @param target A string to search first slash
     * @return A index of first space in the string or, if the string not
     *         containing a space, the whole string length
     */
    public static int firstSpace(String target) {

        return indexOfOrLength(target, " ");
    }

    public static int indexOfOrLength(String target, String needle) {

        return indexOfOrLength(target, needle, 0);
    }

    public static int indexOfOrLength(String target, String needle, int startIndex) {

        return ordinalIndexOfOrLength(target, needle, 1, startIndex);
    }

    public static int ordinalIndexOfOrLength(String target, String needle, int ordinal) {

        return ordinalIndexOfOrLength(target, needle, ordinal, 0);
    }

    public static int ordinalIndexOfOrLength(String target, String needle, int ordinal, int startIndex) {


        String workingTarget = org.apache.commons.lang.StringUtils.defaultString(target);
        workingTarget = org.apache.commons.lang.StringUtils.substring(workingTarget, startIndex + 1);

        int position = org.apache.commons.lang.StringUtils.ordinalIndexOf(workingTarget, needle, ordinal);

        if (position > 0) {
            position = position + startIndex + 1;
        } else {
            position = target.length();
        }

        return position;
    }


    /**
     * This method use the {@link RISMatcher} to search the given needle.
     *
     * @param needle     String to search
     * @param candidates Strings to search against
     * @param tolerance  The tolerance to use in matching.
     * @see RISMatcher#match(SortedSet, String, int)
     */
    public static String risMatch(SortedSet candidates, String needle,
                                  int tolerance) {

        return RISMatcher.INSTANCE.match(candidates, needle, tolerance);
    }

    /**
     * This method use the {@link LDMatcher} to search the given needle.
     *
     * @param needle     String to search
     * @param candidates Strings to search against
     * @param tolerance  The tolerance to use in matching.
     * @see LDMatcher#match(SortedSet, String, int)
     */
    public static String ldMatch(SortedSet candidates, String needle,
                                 int tolerance) {
        return LDMatcher.INSTANCE.match(candidates, needle, tolerance);
    }

    /**
     * Display a device hierarchy as String: generic -> ... -> root.
     *
     * @param hierarchy The hierarchy to convert.
     * @return A string representing the given hierarchy.
     */
    public static String hierarchyAsString(List/* ModelDevice */hierarchy) {

        StrBuilder hierarchyBuilder = new StrBuilder();

        for (java.util.Iterator hIt = hierarchy.iterator(); hIt.hasNext();) {
            hierarchyBuilder.append(hIt.next());
            if (hIt.hasNext()) {
                hierarchyBuilder.append(" -> ");
            }
        }

        return hierarchyBuilder.toString();
    }

    public static int indexOf(String userAgent, String key) {
        return org.apache.commons.lang.StringUtils.indexOf(userAgent, key);
    }


}
