/**
 * This file is released under the GNU General Public License.
 * Refer to the COPYING file distributed with this package.
 *
 * Copyright (c) 2008-2010 WURFL-Pro srl
 */

package net.sourceforge.wurfl.core.web;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;

import javax.servlet.ServletContext;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

import net.sourceforge.wurfl.core.CustomWURFLHolder;
import net.sourceforge.wurfl.core.WURFLException;
import net.sourceforge.wurfl.core.WURFLHolder;
import net.sourceforge.wurfl.core.WURFLManager;
import net.sourceforge.wurfl.core.WURFLUtils;
import net.sourceforge.wurfl.core.resource.WURFLResource;
import net.sourceforge.wurfl.core.resource.WURFLResources;
import net.sourceforge.wurfl.core.resource.XMLResource;

/**
 * ServletContext WURFLHolder decorator.
 * <p/>
 * <p>
 * This class is used to instantiate a WURFLHolder into web application. It is a
 * WURFLHolder Decorator build the delegate by the ServletContext parameters.
 * </p>
 *
 * @author Fantayeneh Asres Gizaw
 * @author Filippo De Luca
 * @version $Id: ServletContextWURFLHolder.java 432 2010-05-06 12:12:53Z filippo.deluca $
 */
public class ServletContextWURFLHolder implements WURFLHolder {

    /**
     * ServletContext wurfl main file parameter
     */
    public static final String WURFL = "wurfl";

    /**
     * ServletContext wurfl patch files parameter
     */
    public static final String WURFL_PATCH = "wurflPatch";

    /**
     * WURFLHolder delegate
     */
    private WURFLHolder delegate;

    /**
     * Build a ServletContextHolder by containing ServletContext.
     *
     * @param servletContext
     */
    public ServletContextWURFLHolder(ServletContext servletContext) {

        delegate = createDelegate(servletContext);
    }

    // Factory methods ****************************************************

    /**
     * Create WURFLHolder delegate from servlet context.
     *
     * @param servletContext The ServletContext used to init this WURFLHolder.
     * @return delegate tuWURFLHolder instance.
     */
    private WURFLHolder createDelegate(ServletContext servletContext) {

        // main file
        String mainPath = servletContext.getInitParameter(WURFL);
        Validate.notEmpty(mainPath, "Please Specify a Valid Location for "
                + WURFL + " context parameter");

        String[] patchesPath = StringUtils.split(StringUtils.defaultString(servletContext.getInitParameter(WURFL_PATCH)), " ,");

        URI uri = toURI(servletContext, mainPath);

        WURFLResource root = new XMLResource(uri);
        WURFLResources patches = new WURFLResources();
        for (int index = 0; index < patchesPath.length; index++) {
            WURFLResource patch = new XMLResource(toURI(servletContext, patchesPath[index]));
            patches.add(patch);
        }

        WURFLHolder wurflHolder = new CustomWURFLHolder(root, patches);

        return wurflHolder;
    }

    private URI toURI(ServletContext servletContext, String mainPath) {
        URI uri = null;
        try {
            uri = servletContext.getResource(mainPath).toURI();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
        return uri;
    }

    // WURFLHolder implementation *****************************************

    /**
     * {@inheritDoc}
     */
    public WURFLManager getWURFLManager() {

        return delegate.getWURFLManager();
    }

    /**
     * {@inheritDoc}
     */
    public WURFLUtils getWURFLUtils() {

		return delegate.getWURFLUtils();
	}

}
