package org.apache.archiva.consumers.lucene;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.maven.archiva.configuration.ManagedRepositoryConfiguration;
import org.apache.maven.archiva.consumers.AbstractMonitoredConsumer;
import org.apache.maven.archiva.consumers.ConsumerException;
import org.apache.maven.archiva.database.updater.DatabaseCleanupConsumer;
import org.apache.maven.archiva.model.ArchivaArtifact;
import org.apache.maven.archiva.repository.ManagedRepositoryContent;
import org.apache.maven.archiva.repository.RepositoryContentFactory;
import org.apache.maven.archiva.repository.RepositoryException;
import org.apache.maven.archiva.scheduled.ArchivaTaskScheduler;
import org.apache.maven.archiva.scheduled.tasks.ArtifactIndexingTask;
import org.apache.maven.archiva.scheduled.tasks.TaskCreator;
import org.codehaus.plexus.taskqueue.TaskQueueException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.List;

/**
 * LuceneCleanupRemoveIndexedConsumer
 * 
 * Clean up the index of artifacts that are no longer existing in the file system (managed repositories).  
 * 
 * @version $Id$
 */
public class LuceneCleanupRemoveIndexedConsumer
    extends AbstractMonitoredConsumer
    implements DatabaseCleanupConsumer
{
    private static final Logger log = LoggerFactory.getLogger( LuceneCleanupRemoveIndexedConsumer.class );

    private RepositoryContentFactory repoFactory;
    
    private ArchivaTaskScheduler scheduler;
    
    public LuceneCleanupRemoveIndexedConsumer( RepositoryContentFactory repoFactory, ArchivaTaskScheduler scheduler )
    {
        this.repoFactory = repoFactory;
        this.scheduler = scheduler;     
    }
      
    public void beginScan()
    {
    }

    public void completeScan()
    {
        
    }

    public List<String> getIncludedTypes()
    {        
        return null;
    }

    public void processArchivaArtifact( ArchivaArtifact artifact )
        throws ConsumerException
    {
        ManagedRepositoryContent repoContent = null;
        
        try
        {
           repoContent =
                repoFactory.getManagedRepositoryContent( artifact.getModel().getRepositoryId() );
        }
        catch ( RepositoryException e )
        {
            throw new ConsumerException( "Can't run index cleanup consumer: " + e.getMessage() );
        }

        ManagedRepositoryConfiguration repository = repoContent.getRepository();
       
        try
        {
            File artifactFile = new File( repoContent.getRepoRoot(), repoContent.toPath( artifact ) );
            
            if ( !artifactFile.exists() )
            {
                ArtifactIndexingTask task =                
                        TaskCreator.createIndexingTask( repository.getId(), artifactFile, ArtifactIndexingTask.DELETE );
                
                log.debug( "Queueing indexing task '" + task.getName() + "' to remove the artifact from the index." );
                scheduler.queueIndexingTask( task );
            }
                   
        } 
        catch ( TaskQueueException e )
        {
            throw new ConsumerException( e.getMessage() );
        }
    }

    public String getDescription()
    {
        return "Remove indexed content if not present on filesystem.";
    }

    public String getId()
    {
        return "not-present-remove-indexed";
    }

    public boolean isPermanent()
    {
        return false;
    }

    public void setRepositoryContentFactory( RepositoryContentFactory repoFactory )
    {
        this.repoFactory = repoFactory;
    }
}
