/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cxf.xkms.x509.repo.file;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.net.URISyntaxException;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;
import org.apache.cxf.xkms.exception.XKMSConfigurationException;
import org.apache.cxf.xkms.handlers.Applications;
import org.apache.cxf.xkms.model.xkms.ResultMajorEnum;
import org.apache.cxf.xkms.model.xkms.ResultMinorEnum;
import org.apache.cxf.xkms.model.xkms.UseKeyWithType;
import org.apache.cxf.xkms.x509.repo.CertificateRepo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCertificateRepo
implements CertificateRepo {
    private static final Logger LOG = LoggerFactory.getLogger(FileCertificateRepo.class);
    private static final String CN_PREFIX = "cn=";
    private static final String TRUSTED_CAS_PATH = "trusted_cas";
    private static final String CRLS_PATH = "crls";
    private static final String CAS_PATH = "cas";
    private static final String SPLIT_REGEX = "\\s*,\\s*";
    private final File storageDir;
    private final CertificateFactory certFactory;

    public FileCertificateRepo(String path) {
        this.storageDir = new File(path);
        try {
            this.certFactory = CertificateFactory.getInstance("X.509");
        }
        catch (Exception e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    public void saveCertificate(X509Certificate cert, UseKeyWithType id) {
        this.saveCategorizedCertificate(cert, id, false, false);
    }

    public void saveTrustedCACertificate(X509Certificate cert, UseKeyWithType id) {
        this.saveCategorizedCertificate(cert, id, true, false);
    }

    public void saveCACertificate(X509Certificate cert, UseKeyWithType id) {
        this.saveCategorizedCertificate(cert, id, false, true);
    }

    public void saveCRL(X509CRL crl, UseKeyWithType id) {
        String name = crl.getIssuerX500Principal().getName();
        try {
            String path = this.convertIdForFileSystem(name) + ".cer";
            Pattern p = Pattern.compile("[a-zA-Z_0-9-_]");
            if (!p.matcher(path).find()) {
                throw new URISyntaxException(path, "Input did not match [a-zA-Z_0-9-_].");
            }
            File certFile = new File(this.storageDir + "/" + CRLS_PATH, path);
            certFile.getParentFile().mkdirs();
            try (FileOutputStream fos = new FileOutputStream(certFile);
                 BufferedOutputStream bos = new BufferedOutputStream(fos);){
                bos.write(crl.getEncoded());
                bos.close();
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Error saving CRL " + name + ": " + e.getMessage(), e);
        }
    }

    private boolean saveCategorizedCertificate(X509Certificate cert, UseKeyWithType id, boolean isTrustedCA, boolean isCA) {
        String category = "";
        if (isTrustedCA) {
            category = TRUSTED_CAS_PATH;
        }
        if (isCA) {
            category = CAS_PATH;
        }
        try {
            File certFile = new File(this.storageDir + "/" + category, this.getCertPath(cert, id));
            certFile.getParentFile().mkdirs();
            try (FileOutputStream fos = new FileOutputStream(certFile);
                 BufferedOutputStream bos = new BufferedOutputStream(fos);){
                bos.write(cert.getEncoded());
                bos.close();
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Error saving certificate " + cert.getSubjectDN() + ": " + e.getMessage(), e);
        }
        return true;
    }

    public String convertIdForFileSystem(String dn) {
        String result = dn.replace("=", "-");
        result = result.replace(", ", "_");
        result = result.replace(",", "_");
        result = result.replace("/", "_");
        result = result.replace("\\", "_");
        result = result.replace("{", "_");
        result = result.replace("}", "_");
        result = result.replace(":", "_");
        return result;
    }

    public String getCertPath(X509Certificate cert, UseKeyWithType id) throws URISyntaxException {
        Applications application = null;
        String path = null;
        if (id != null) {
            application = Applications.fromUri((String)id.getApplication());
        }
        path = application == Applications.SERVICE_ENDPOINT ? id.getIdentifier() : cert.getSubjectDN().getName();
        path = this.convertIdForFileSystem(path) + ".cer";
        this.validateCertificatePath(path);
        return path;
    }

    private void validateCertificatePath(String path) throws URISyntaxException {
        Pattern p = Pattern.compile("[a-zA-Z_0-9-_]");
        if (!p.matcher(path).find()) {
            throw new URISyntaxException(path, "Input did not match [a-zA-Z_0-9-_].");
        }
    }

    private File[] getX509Files() {
        ArrayList<File> certificateFiles = new ArrayList<File>();
        try {
            certificateFiles.addAll(Arrays.asList(this.storageDir.listFiles()));
            certificateFiles.addAll(Arrays.asList(new File(this.storageDir + "/" + TRUSTED_CAS_PATH).listFiles()));
            certificateFiles.addAll(Arrays.asList(new File(this.storageDir + "/" + CAS_PATH).listFiles()));
            certificateFiles.addAll(Arrays.asList(new File(this.storageDir + "/" + CRLS_PATH).listFiles()));
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        if (certificateFiles.isEmpty()) {
            throw new XKMSConfigurationException(ResultMajorEnum.HTTP_WWW_W_3_ORG_2002_03_XKMS_RECEIVER, ResultMinorEnum.HTTP_WWW_W_3_ORG_2002_03_XKMS_FAILURE, "File base persistence storage is not found: " + this.storageDir.getPath());
        }
        return certificateFiles.toArray(new File[certificateFiles.size()]);
    }

    public X509Certificate readCertificate(File certFile) throws CertificateException, FileNotFoundException, IOException {
        try (FileInputStream fis = new FileInputStream(certFile);){
            X509Certificate x509Certificate = (X509Certificate)this.certFactory.generateCertificate(fis);
            return x509Certificate;
        }
    }

    public X509CRL readCRL(File crlFile) throws FileNotFoundException, CRLException, IOException {
        try (FileInputStream fis = new FileInputStream(crlFile);){
            X509CRL x509CRL = (X509CRL)this.certFactory.generateCRL(fis);
            return x509CRL;
        }
    }

    public List<X509Certificate> getTrustedCaCerts() {
        File[] list;
        ArrayList<X509Certificate> results = new ArrayList<X509Certificate>();
        for (File certFile : list = this.getX509Files()) {
            try {
                if (certFile.isDirectory() || !certFile.getParent().endsWith(TRUSTED_CAS_PATH)) continue;
                X509Certificate cert = this.readCertificate(certFile);
                results.add(cert);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load certificate from file: %s. Error: %s", certFile, e.getMessage()));
            }
        }
        return results;
    }

    public List<X509Certificate> getCaCerts() {
        File[] list;
        ArrayList<X509Certificate> results = new ArrayList<X509Certificate>();
        for (File certFile : list = this.getX509Files()) {
            try {
                if (certFile.isDirectory() || !certFile.getParent().endsWith(CAS_PATH)) continue;
                X509Certificate cert = this.readCertificate(certFile);
                results.add(cert);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load certificate from file: %s. Error: %s", certFile, e.getMessage()));
            }
        }
        return results;
    }

    public List<X509CRL> getCRLs() {
        File[] list;
        ArrayList<X509CRL> results = new ArrayList<X509CRL>();
        for (File crlFile : list = this.getX509Files()) {
            try {
                if (crlFile.isDirectory() || !crlFile.getParent().endsWith(CRLS_PATH)) continue;
                X509CRL crl = this.readCRL(crlFile);
                results.add(crl);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load CRL from file: %s. Error: %s", crlFile, e.getMessage()));
            }
        }
        return results;
    }

    public X509Certificate findByServiceName(String serviceName) {
        return this.findBySubjectDn(CN_PREFIX + serviceName);
    }

    public X509Certificate findByEndpoint(String endpoint) {
        try {
            String path = this.convertIdForFileSystem(endpoint) + ".cer";
            this.validateCertificatePath(path);
            File certFile = new File(this.storageDir.getAbsolutePath() + "/" + path);
            if (!certFile.exists()) {
                LOG.warn(String.format("Certificate not found for endpoint %s, path %s", endpoint, certFile.getAbsolutePath()));
                return null;
            }
            return (X509Certificate)this.certFactory.generateCertificate(new FileInputStream(certFile));
        }
        catch (Exception e) {
            LOG.warn(String.format("Cannot load certificate by endpoint: %s. Error: %s", endpoint, e.getMessage()), (Throwable)e);
            return null;
        }
    }

    public X509Certificate findBySubjectDn(String subjectDn) {
        ArrayList<X509Certificate> result = new ArrayList<X509Certificate>();
        File[] list = this.getX509Files();
        Object[] sDnArray = subjectDn.split(SPLIT_REGEX);
        Arrays.sort(sDnArray);
        for (File certFile : list) {
            try {
                if (certFile.isDirectory()) continue;
                X509Certificate cert = this.readCertificate(certFile);
                LOG.debug("Searching for " + subjectDn + ". Checking cert " + cert.getSubjectDN().getName() + ", " + cert.getSubjectX500Principal().getName());
                Object[] csDnArray = cert.getSubjectDN().getName().split(SPLIT_REGEX);
                Arrays.sort(csDnArray);
                Object[] csX500Array = cert.getSubjectX500Principal().getName().split(SPLIT_REGEX);
                Arrays.sort(csX500Array);
                if (!this.arraysEqualsIgnoreCaseIgnoreWhiteSpace((String[])sDnArray, (String[])csDnArray) && !this.arraysEqualsIgnoreCaseIgnoreWhiteSpace((String[])sDnArray, (String[])csX500Array)) continue;
                result.add(cert);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load certificate from file: %s. Error: %s", certFile, e.getMessage()));
            }
        }
        if (!result.isEmpty()) {
            return (X509Certificate)result.get(0);
        }
        return null;
    }

    private boolean arraysEqualsIgnoreCaseIgnoreWhiteSpace(String[] s1, String[] s2) {
        if (s1 == null || s2 == null || s1.length != s2.length) {
            return false;
        }
        for (int i = 0; i < s1.length; ++i) {
            if (s1[i].trim().equalsIgnoreCase(s2[i].trim())) continue;
            return false;
        }
        return true;
    }

    public X509Certificate findByIssuerSerial(String issuer, String serial) {
        File[] list;
        ArrayList<X509Certificate> result = new ArrayList<X509Certificate>();
        for (File certFile : list = this.getX509Files()) {
            try {
                if (certFile.isDirectory()) continue;
                X509Certificate cert = this.readCertificate(certFile);
                BigInteger cs = cert.getSerialNumber();
                BigInteger ss = new BigInteger(serial, 16);
                if (!issuer.equalsIgnoreCase(cert.getIssuerX500Principal().getName()) || !cs.equals(ss)) continue;
                result.add(cert);
            }
            catch (Exception e) {
                LOG.warn(String.format("Cannot load certificate from file: %s. Error: %s", certFile, e.getMessage()));
            }
        }
        if (!result.isEmpty()) {
            return (X509Certificate)result.get(0);
        }
        return null;
    }
}

