/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.server.core.authn;

import java.io.IOException;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.apache.directory.api.ldap.extras.controls.ppolicy.PasswordPolicyErrorEnum;
import org.apache.directory.api.ldap.extras.controls.ppolicy.PasswordPolicyResponseImpl;
import org.apache.directory.api.ldap.model.constants.AuthenticationLevel;
import org.apache.directory.api.ldap.model.constants.LdapSecurityConstants;
import org.apache.directory.api.ldap.model.constants.SchemaConstants;
import org.apache.directory.api.ldap.model.entry.Attribute;
import org.apache.directory.api.ldap.model.entry.DefaultAttribute;
import org.apache.directory.api.ldap.model.entry.DefaultModification;
import org.apache.directory.api.ldap.model.entry.Entry;
import org.apache.directory.api.ldap.model.entry.Modification;
import org.apache.directory.api.ldap.model.entry.ModificationOperation;
import org.apache.directory.api.ldap.model.entry.Value;
import org.apache.directory.api.ldap.model.exception.LdapAuthenticationException;
import org.apache.directory.api.ldap.model.exception.LdapException;
import org.apache.directory.api.ldap.model.exception.LdapNoPermissionException;
import org.apache.directory.api.ldap.model.exception.LdapOperationException;
import org.apache.directory.api.ldap.model.exception.LdapOtherException;
import org.apache.directory.api.ldap.model.exception.LdapUnwillingToPerformException;
import org.apache.directory.api.ldap.model.message.Control;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;
import org.apache.directory.api.ldap.model.name.Dn;
import org.apache.directory.api.ldap.model.password.PasswordUtil;
import org.apache.directory.api.ldap.model.schema.AttributeType;
import org.apache.directory.api.util.DateUtils;
import org.apache.directory.api.util.Strings;
import org.apache.directory.api.util.TimeProvider;
import org.apache.directory.server.core.api.CoreSession;
import org.apache.directory.server.core.api.DirectoryService;
import org.apache.directory.server.core.api.InterceptorEnum;
import org.apache.directory.server.core.api.LdapPrincipal;
import org.apache.directory.server.core.api.authn.ppolicy.CheckQualityEnum;
import org.apache.directory.server.core.api.authn.ppolicy.DefaultPasswordValidator;
import org.apache.directory.server.core.api.authn.ppolicy.PasswordPolicyConfiguration;
import org.apache.directory.server.core.api.authn.ppolicy.PasswordPolicyException;
import org.apache.directory.server.core.api.authn.ppolicy.PasswordValidator;
import org.apache.directory.server.core.api.filtering.EntryFilteringCursor;
import org.apache.directory.server.core.api.interceptor.BaseInterceptor;
import org.apache.directory.server.core.api.interceptor.context.AddOperationContext;
import org.apache.directory.server.core.api.interceptor.context.BindOperationContext;
import org.apache.directory.server.core.api.interceptor.context.CompareOperationContext;
import org.apache.directory.server.core.api.interceptor.context.DeleteOperationContext;
import org.apache.directory.server.core.api.interceptor.context.GetRootDseOperationContext;
import org.apache.directory.server.core.api.interceptor.context.HasEntryOperationContext;
import org.apache.directory.server.core.api.interceptor.context.LookupOperationContext;
import org.apache.directory.server.core.api.interceptor.context.ModifyOperationContext;
import org.apache.directory.server.core.api.interceptor.context.MoveAndRenameOperationContext;
import org.apache.directory.server.core.api.interceptor.context.MoveOperationContext;
import org.apache.directory.server.core.api.interceptor.context.OperationContext;
import org.apache.directory.server.core.api.interceptor.context.RenameOperationContext;
import org.apache.directory.server.core.api.interceptor.context.SearchOperationContext;
import org.apache.directory.server.core.api.interceptor.context.UnbindOperationContext;
import org.apache.directory.server.core.api.partition.Partition;
import org.apache.directory.server.core.api.partition.PartitionTxn;
import org.apache.directory.server.core.api.partition.PartitionWriteTxn;
import org.apache.directory.server.core.authn.AnonymousAuthenticator;
import org.apache.directory.server.core.authn.Authenticator;
import org.apache.directory.server.core.authn.PasswordHistory;
import org.apache.directory.server.core.authn.SimpleAuthenticator;
import org.apache.directory.server.core.authn.StrongAuthenticator;
import org.apache.directory.server.core.authn.ppolicy.PpolicyConfigContainer;
import org.apache.directory.server.core.shared.DefaultCoreSession;
import org.apache.directory.server.i18n.I18n;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AuthenticationInterceptor
extends BaseInterceptor {
    private static final Logger LOG = LoggerFactory.getLogger(AuthenticationInterceptor.class);
    private static final boolean IS_DEBUG = LOG.isDebugEnabled();
    private Set<Authenticator> authenticators = new HashSet<Authenticator>();
    private final EnumMap<AuthenticationLevel, Collection<Authenticator>> authenticatorsMapByType = new EnumMap(AuthenticationLevel.class);
    private CoreSession adminSession;
    private AttributeType pwdResetAT;
    private AttributeType pwdChangedTimeAT;
    private AttributeType pwdHistoryAT;
    private AttributeType pwdFailurTimeAT;
    private AttributeType pwdAccountLockedTimeAT;
    private AttributeType pwdLastSuccessAT;
    private AttributeType pwdGraceUseTimeAT;
    private AttributeType pwdPolicySubentryAT;
    private AttributeType pwdStartTimeAT;
    private AttributeType pwdEndTimeAT;
    private PpolicyConfigContainer pwdPolicyContainer;

    public AuthenticationInterceptor() {
        super(InterceptorEnum.AUTHENTICATION_INTERCEPTOR);
    }

    public void init(DirectoryService directoryService) throws LdapException {
        super.init(directoryService);
        this.adminSession = directoryService.getAdminSession();
        if (this.authenticators == null || this.authenticators.isEmpty()) {
            this.setDefaultAuthenticators();
        }
        for (Authenticator authenticator : this.authenticators) {
            this.register(authenticator, directoryService);
        }
        this.loadPwdPolicyStateAttributeTypes();
    }

    private void setDefaultAuthenticators() {
        if (this.authenticators == null) {
            this.authenticators = new HashSet<Authenticator>();
        }
        this.authenticators.clear();
        this.authenticators.add(new AnonymousAuthenticator(Dn.ROOT_DSE));
        this.authenticators.add(new SimpleAuthenticator(Dn.ROOT_DSE));
        this.authenticators.add(new StrongAuthenticator(Dn.ROOT_DSE));
    }

    public Set<Authenticator> getAuthenticators() {
        return this.authenticators;
    }

    public void setAuthenticators(Set<Authenticator> authenticators) {
        if (authenticators == null) {
            this.authenticators.clear();
        } else {
            this.authenticators = authenticators;
        }
    }

    public void setAuthenticators(Authenticator[] authenticators) {
        if (authenticators == null) {
            throw new IllegalArgumentException("The given authenticators set is null");
        }
        this.authenticators.clear();
        this.authenticatorsMapByType.clear();
        for (Authenticator authenticator : authenticators) {
            try {
                this.register(authenticator, this.directoryService);
            }
            catch (LdapException le) {
                LOG.error("Cannot register authenticator {}", (Object)authenticator);
            }
        }
    }

    public void destroy() {
        this.authenticatorsMapByType.clear();
        HashSet<Authenticator> copy = new HashSet<Authenticator>(this.authenticators);
        this.authenticators = new HashSet<Authenticator>();
        for (Authenticator authenticator : copy) {
            authenticator.destroy();
        }
    }

    private void register(Authenticator authenticator, DirectoryService directoryService) throws LdapException {
        authenticator.init(directoryService);
        this.authenticators.add(authenticator);
        Collection<Authenticator> authenticatorList = this.getAuthenticators(authenticator.getAuthenticatorType());
        if (authenticatorList == null) {
            authenticatorList = new ArrayList<Authenticator>();
            this.authenticatorsMapByType.put(authenticator.getAuthenticatorType(), authenticatorList);
        }
        if (!authenticatorList.contains(authenticator)) {
            authenticatorList.add(authenticator);
        }
    }

    private Collection<Authenticator> getAuthenticators(AuthenticationLevel type) {
        Collection<Authenticator> result = this.authenticatorsMapByType.get(type);
        if (result != null && !result.isEmpty()) {
            return result;
        }
        return null;
    }

    public void add(AddOperationContext addContext) throws LdapException {
        Attribute userPasswordAttribute;
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)addContext);
        }
        this.checkAuthenticated((OperationContext)addContext);
        Entry entry = addContext.getEntry();
        if (!this.directoryService.isPwdPolicyEnabled() || addContext.isReplEvent()) {
            this.next(addContext);
            return;
        }
        PasswordPolicyConfiguration policyConfig = this.getPwdPolicy(entry);
        boolean isPPolicyReqCtrlPresent = addContext.hasRequestControl("1.3.6.1.4.1.42.2.27.8.5.1");
        this.checkPwdReset((OperationContext)addContext);
        String passwordAttribute = "userPassword";
        if (isPPolicyReqCtrlPresent) {
            passwordAttribute = policyConfig.getPwdAttribute();
        }
        if ((userPasswordAttribute = entry.get(passwordAttribute)) != null) {
            Value userPassword = userPasswordAttribute.get();
            try {
                this.check((OperationContext)addContext, entry, userPassword.getBytes(), policyConfig);
            }
            catch (PasswordPolicyException e) {
                if (isPPolicyReqCtrlPresent) {
                    PasswordPolicyResponseImpl responseControl = new PasswordPolicyResponseImpl();
                    responseControl.setPasswordPolicyError(PasswordPolicyErrorEnum.get((int)e.getErrorCode()));
                    addContext.addResponseControl((Control)responseControl);
                }
                throw new LdapOperationException(ResultCodeEnum.CONSTRAINT_VIOLATION, e.getMessage(), (Throwable)e);
            }
            String pwdChangedTime = DateUtils.getGeneralizedTime((TimeProvider)this.directoryService.getTimeProvider());
            if (!(policyConfig.getPwdMinAge() <= 0 && policyConfig.getPwdMaxAge() <= 0 || addContext.getSession().isAnAdministrator() && entry.get(this.pwdChangedTimeAT) != null)) {
                DefaultAttribute pwdChangedTimeAt = new DefaultAttribute(this.pwdChangedTimeAT);
                pwdChangedTimeAt.add(new String[]{pwdChangedTime});
                entry.add(new Attribute[]{pwdChangedTimeAt});
            }
            if (policyConfig.isPwdMustChange() && addContext.getSession().isAnAdministrator()) {
                DefaultAttribute pwdResetAt = new DefaultAttribute(this.pwdResetAT);
                pwdResetAt.add(new String[]{"TRUE"});
                entry.add(new Attribute[]{pwdResetAt});
            }
            if (policyConfig.getPwdInHistory() > 0) {
                DefaultAttribute pwdHistoryAt = new DefaultAttribute(this.pwdHistoryAT);
                byte[] pwdHistoryVal = new PasswordHistory(pwdChangedTime, userPassword.getBytes()).getHistoryValue();
                pwdHistoryAt.add((byte[][])new byte[][]{pwdHistoryVal});
                entry.add(new Attribute[]{pwdHistoryAt});
            }
        }
        this.next(addContext);
    }

    private Authenticator selectAuthenticator(Dn bindDn, AuthenticationLevel level) throws LdapUnwillingToPerformException, LdapAuthenticationException {
        Iterator<Authenticator> iterator;
        Authenticator selectedAuthenticator = null;
        Collection<Authenticator> levelAuthenticators = this.authenticatorsMapByType.get(level);
        if (levelAuthenticators == null || levelAuthenticators.isEmpty()) {
            throw new LdapAuthenticationException("Cannot Bind for Dn " + bindDn.getName() + ", no authenticator for the requested level " + level);
        }
        if (levelAuthenticators.size() == 1 && (iterator = levelAuthenticators.iterator()).hasNext()) {
            Authenticator authenticator = iterator.next();
            if (authenticator.isValid(bindDn)) {
                return authenticator;
            }
            throw new LdapUnwillingToPerformException(ResultCodeEnum.UNWILLING_TO_PERFORM, "Cannot Bind for Dn " + bindDn.getName() + ", its not a descendant of the authenticator base DN '" + authenticator.getBaseDn() + "'");
        }
        Dn innerDn = Dn.ROOT_DSE;
        for (Authenticator authenticator : levelAuthenticators) {
            if (!authenticator.isValid(bindDn) || !innerDn.isAncestorOf(authenticator.getBaseDn())) continue;
            innerDn = authenticator.getBaseDn();
            selectedAuthenticator = authenticator;
        }
        if (selectedAuthenticator == null) {
            throw new LdapUnwillingToPerformException(ResultCodeEnum.UNWILLING_TO_PERFORM, "Cannot Bind for Dn " + bindDn.getName() + ", there is no authenticator for it");
        }
        return selectedAuthenticator;
    }

    private void internalModify(OperationContext opContext, ModifyOperationContext bindModCtx) throws LdapException {
        Partition partition = opContext.getPartition();
        bindModCtx.setPartition(partition);
        PartitionWriteTxn partitionTxn = null;
        try {
            partitionTxn = partition.beginWriteTransaction();
            bindModCtx.setTransaction((PartitionTxn)partitionTxn);
            this.directoryService.getPartitionNexus().modify(bindModCtx);
            partitionTxn.commit();
        }
        catch (LdapException le) {
            try {
                if (partitionTxn != null) {
                    partitionTxn.abort();
                }
                throw le;
            }
            catch (IOException ioe) {
                throw new LdapOtherException(ioe.getMessage(), (Throwable)ioe);
            }
        }
        catch (IOException ioe) {
            try {
                partitionTxn.abort();
                throw new LdapOtherException(ioe.getMessage(), (Throwable)ioe);
            }
            catch (IOException ioe2) {
                throw new LdapOtherException(ioe2.getMessage(), (Throwable)ioe2);
            }
        }
    }

    public void bind(BindOperationContext bindContext) throws LdapException {
        AuthenticationLevel level;
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)bindContext);
        }
        CoreSession session = bindContext.getSession();
        Dn bindDn = bindContext.getDn();
        if (session != null && session.getEffectivePrincipal() != null && !session.isAnonymous() && !session.isAdministrator()) {
            bindContext.setCredentials(null);
        }
        if ((level = bindContext.getAuthenticationLevel()) == AuthenticationLevel.UNAUTHENT) {
            throw new LdapUnwillingToPerformException(ResultCodeEnum.UNWILLING_TO_PERFORM, "Cannot Bind for Dn " + bindDn.getName());
        }
        PasswordPolicyException ppe = null;
        boolean isPPolicyReqCtrlPresent = bindContext.hasRequestControl("1.3.6.1.4.1.42.2.27.8.5.1");
        PasswordPolicyResponseImpl pwdRespCtrl = new PasswordPolicyResponseImpl();
        boolean authenticated = false;
        Authenticator authenticator = this.selectAuthenticator(bindDn, level);
        try {
            LdapPrincipal principal = authenticator.authenticate(bindContext);
            if (principal != null) {
                LdapPrincipal clonedPrincipal = (LdapPrincipal)principal.clone();
                bindContext.setCredentials(null);
                clonedPrincipal.setUserPassword((byte[][])new byte[][]{Strings.EMPTY_BYTES});
                DefaultCoreSession newSession = new DefaultCoreSession(clonedPrincipal, this.directoryService);
                bindContext.setSession((CoreSession)newSession);
                authenticated = true;
            }
        }
        catch (PasswordPolicyException e) {
            ppe = e;
        }
        catch (LdapAuthenticationException e) {
            LOG.info("Authenticator {} failed to authenticate: {}", (Object)authenticator, (Object)bindContext.getDn());
        }
        catch (Exception e) {
            LOG.info("Unexpected failure for Authenticator {} : {}", (Object)authenticator, (Object)bindContext.getDn());
        }
        if (ppe != null) {
            if (isPPolicyReqCtrlPresent) {
                pwdRespCtrl.setPasswordPolicyError(PasswordPolicyErrorEnum.get((int)ppe.getErrorCode()));
                bindContext.addResponseControl((Control)pwdRespCtrl);
            }
            throw ppe;
        }
        Entry userEntry = bindContext.getEntry();
        PasswordPolicyConfiguration policyConfig = this.getPwdPolicy(userEntry);
        if (policyConfig != null) {
            LookupOperationContext lookupContext = new LookupOperationContext(this.adminSession, bindDn, SchemaConstants.ALL_ATTRIBUTES_ARRAY);
            lookupContext.setPartition(bindContext.getPartition());
            lookupContext.setTransaction(bindContext.getTransaction());
            userEntry = this.directoryService.getPartitionNexus().lookup(lookupContext);
        }
        if (authenticated && userEntry == null && this.directoryService.isAllowAnonymousAccess()) {
            return;
        }
        if (!authenticated) {
            if (LOG.isInfoEnabled()) {
                LOG.info("Cannot bind to the server ");
            }
            if (policyConfig != null && userEntry != null) {
                Attribute pwdFailTimeAt = userEntry.get(this.pwdFailurTimeAT);
                if (pwdFailTimeAt == null) {
                    pwdFailTimeAt = new DefaultAttribute(this.pwdFailurTimeAT);
                } else {
                    this.purgeFailureTimes(policyConfig, pwdFailTimeAt);
                }
                String failureTime = DateUtils.getGeneralizedTime((TimeProvider)this.directoryService.getTimeProvider());
                pwdFailTimeAt.add(new String[]{failureTime});
                DefaultModification pwdFailTimeMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, pwdFailTimeAt);
                ArrayList<DefaultModification> mods = new ArrayList<DefaultModification>();
                mods.add(pwdFailTimeMod);
                int numFailures = pwdFailTimeAt.size();
                if (policyConfig.isPwdLockout() && numFailures >= policyConfig.getPwdMaxFailure()) {
                    if (!userEntry.getDn().equals((Object)new Dn(this.schemaManager, new String[]{"uid=admin,ou=system"}))) {
                        DefaultAttribute pwdAccountLockedTimeAt = new DefaultAttribute(this.pwdAccountLockedTimeAT);
                        if (policyConfig.getPwdLockoutDuration() == 0) {
                            pwdAccountLockedTimeAt.add(new String[]{"000001010000Z"});
                        } else {
                            pwdAccountLockedTimeAt.add(new String[]{failureTime});
                        }
                        DefaultModification pwdAccountLockedMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, (Attribute)pwdAccountLockedTimeAt);
                        mods.add(pwdAccountLockedMod);
                        pwdRespCtrl.setPasswordPolicyError(PasswordPolicyErrorEnum.ACCOUNT_LOCKED);
                    }
                } else if (policyConfig.getPwdMinDelay() > 0) {
                    int maxDelay;
                    int numDelay = numFailures * policyConfig.getPwdMinDelay();
                    if (numDelay > (maxDelay = policyConfig.getPwdMaxDelay())) {
                        numDelay = maxDelay;
                    }
                    try {
                        Thread.sleep((long)numDelay * 1000L);
                    }
                    catch (InterruptedException e) {
                        LOG.warn("Interrupted while delaying to send the failed authentication response for the user {}", (Object)bindDn, (Object)e);
                    }
                }
                if (!mods.isEmpty()) {
                    String csnVal = this.directoryService.getCSN().toString();
                    DefaultModification csnMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, this.directoryService.getAtProvider().getEntryCSN(), new String[]{csnVal});
                    mods.add(csnMod);
                    ModifyOperationContext bindModCtx = new ModifyOperationContext(this.adminSession);
                    bindModCtx.setDn(bindDn);
                    bindModCtx.setEntry(userEntry);
                    bindModCtx.setModItems(mods);
                    bindModCtx.setPushToEvtInterceptor(true);
                    this.internalModify((OperationContext)bindContext, bindModCtx);
                }
            }
            String upDn = bindDn == null ? "" : bindDn.getName();
            throw new LdapAuthenticationException(I18n.err((I18n)I18n.ERR_229, (Object[])new Object[]{upDn}));
        }
        if (policyConfig != null) {
            boolean expired;
            Attribute pwdChangeTimeAttr;
            Attribute pwdAccLockedTimeAt;
            Attribute pwdFailTimeAt;
            ArrayList<DefaultModification> mods = new ArrayList<DefaultModification>();
            if (policyConfig.getPwdMaxIdle() > 0) {
                DefaultAttribute pwdLastSuccesTimeAt = new DefaultAttribute(this.pwdLastSuccessAT);
                pwdLastSuccesTimeAt.add(new String[]{DateUtils.getGeneralizedTime((TimeProvider)this.directoryService.getTimeProvider())});
                DefaultModification pwdLastSuccesTimeMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, (Attribute)pwdLastSuccesTimeAt);
                mods.add(pwdLastSuccesTimeMod);
            }
            if ((pwdFailTimeAt = userEntry.get(this.pwdFailurTimeAT)) != null) {
                DefaultModification pwdFailTimeMod = new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, pwdFailTimeAt);
                mods.add(pwdFailTimeMod);
            }
            if ((pwdAccLockedTimeAt = userEntry.get(this.pwdAccountLockedTimeAT)) != null) {
                DefaultModification pwdAccLockedTimeMod = new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, pwdAccLockedTimeAt);
                mods.add(pwdAccLockedTimeMod);
            }
            if (policyConfig.getPwdMaxAge() > 0 && policyConfig.getPwdGraceAuthNLimit() > 0 && (pwdChangeTimeAttr = userEntry.get(this.pwdChangedTimeAT)) != null && (expired = PasswordUtil.isPwdExpired((String)pwdChangeTimeAttr.getString(), (int)policyConfig.getPwdMaxAge(), (TimeProvider)this.directoryService.getTimeProvider()))) {
                int numGraceAuth;
                Attribute pwdGraceUseAttr = userEntry.get(this.pwdGraceUseTimeAT);
                if (pwdGraceUseAttr != null) {
                    numGraceAuth = policyConfig.getPwdGraceAuthNLimit() - (pwdGraceUseAttr.size() + 1);
                } else {
                    pwdGraceUseAttr = new DefaultAttribute(this.pwdGraceUseTimeAT);
                    numGraceAuth = policyConfig.getPwdGraceAuthNLimit() - 1;
                }
                pwdRespCtrl.setGraceAuthNRemaining(numGraceAuth);
                pwdGraceUseAttr.add(new String[]{DateUtils.getGeneralizedTime((TimeProvider)this.directoryService.getTimeProvider())});
                DefaultModification pwdGraceUseMod = new DefaultModification(ModificationOperation.ADD_ATTRIBUTE, pwdGraceUseAttr);
                mods.add(pwdGraceUseMod);
            }
            if (!mods.isEmpty()) {
                String csnVal = this.directoryService.getCSN().toString();
                DefaultModification csnMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, this.directoryService.getAtProvider().getEntryCSN(), new String[]{csnVal});
                mods.add(csnMod);
                ModifyOperationContext bindModCtx = new ModifyOperationContext(this.adminSession);
                bindModCtx.setDn(bindDn);
                bindModCtx.setEntry(userEntry);
                bindModCtx.setModItems(mods);
                bindModCtx.setPushToEvtInterceptor(true);
                this.internalModify((OperationContext)bindContext, bindModCtx);
            }
            if (isPPolicyReqCtrlPresent) {
                int expiryWarnTime = this.getPwdTimeBeforeExpiry(userEntry, policyConfig);
                if (expiryWarnTime > 0) {
                    pwdRespCtrl.setTimeBeforeExpiration(expiryWarnTime);
                }
                if (this.isPwdMustReset(userEntry)) {
                    pwdRespCtrl.setPasswordPolicyError(PasswordPolicyErrorEnum.CHANGE_AFTER_RESET);
                    bindContext.getSession().setPwdMustChange(true);
                }
                bindContext.addResponseControl((Control)pwdRespCtrl);
            }
        }
    }

    public boolean compare(CompareOperationContext compareContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)compareContext);
        }
        this.checkAuthenticated((OperationContext)compareContext);
        this.checkPwdReset((OperationContext)compareContext);
        return this.next(compareContext);
    }

    public void delete(DeleteOperationContext deleteContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)deleteContext);
        }
        this.checkAuthenticated((OperationContext)deleteContext);
        this.next(deleteContext);
        this.invalidateAuthenticatorCaches(deleteContext.getDn());
    }

    public Entry getRootDse(GetRootDseOperationContext getRootDseContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)getRootDseContext);
        }
        this.checkAuthenticated((OperationContext)getRootDseContext);
        this.checkPwdReset((OperationContext)getRootDseContext);
        return this.next(getRootDseContext);
    }

    public boolean hasEntry(HasEntryOperationContext hasEntryContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)hasEntryContext);
        }
        this.checkAuthenticated((OperationContext)hasEntryContext);
        this.checkPwdReset((OperationContext)hasEntryContext);
        return this.next(hasEntryContext);
    }

    public Entry lookup(LookupOperationContext lookupContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)lookupContext);
        }
        this.checkAuthenticated((OperationContext)lookupContext);
        this.checkPwdReset((OperationContext)lookupContext);
        return this.next(lookupContext);
    }

    private void invalidateAuthenticatorCaches(Dn principalDn) {
        for (AuthenticationLevel authMech : this.authenticatorsMapByType.keySet()) {
            for (Authenticator authenticator : this.getAuthenticators(authMech)) {
                authenticator.invalidateCache(principalDn);
            }
        }
    }

    public void modify(ModifyOperationContext modifyContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)modifyContext);
        }
        this.checkAuthenticated((OperationContext)modifyContext);
        if (!this.directoryService.isPwdPolicyEnabled() || modifyContext.isReplEvent()) {
            this.processStandardModify(modifyContext);
        } else {
            this.processPasswordPolicydModify(modifyContext);
        }
    }

    private void processStandardModify(ModifyOperationContext modifyContext) throws LdapException {
        this.next(modifyContext);
        List modifications = modifyContext.getModItems();
        for (Modification modification : modifications) {
            if (!this.directoryService.getAtProvider().getUserPassword().equals((Object)modification.getAttribute().getAttributeType())) continue;
            this.invalidateAuthenticatorCaches(modifyContext.getDn());
            break;
        }
    }

    private void processPasswordPolicydModify(ModifyOperationContext modifyContext) throws LdapException {
        PasswordPolicyConfiguration policyConfig = this.getPwdPolicy(modifyContext.getEntry());
        PwdModDetailsHolder pwdModDetails = this.getPwdModDetails(modifyContext, policyConfig);
        if (!pwdModDetails.isPwdModPresent()) {
            this.next(modifyContext);
        } else {
            CoreSession userSession = modifyContext.getSession();
            boolean isPPolicyReqCtrlPresent = modifyContext.hasRequestControl("1.3.6.1.4.1.42.2.27.8.5.1");
            this.checkPwdMustChange(modifyContext, userSession, pwdModDetails, isPPolicyReqCtrlPresent);
            this.checkOldPwdRequired(modifyContext, policyConfig, pwdModDetails, isPPolicyReqCtrlPresent);
            this.checkChangePwdAllowed(modifyContext, policyConfig, isPPolicyReqCtrlPresent);
            Entry entry = modifyContext.getEntry();
            boolean removePwdReset = false;
            ArrayList<Modification> mods = new ArrayList<Modification>();
            if (pwdModDetails.isAddOrReplace()) {
                if (this.isPwdTooYoung((OperationContext)modifyContext, entry, policyConfig)) {
                    if (isPPolicyReqCtrlPresent) {
                        PasswordPolicyResponseImpl responseControl = new PasswordPolicyResponseImpl();
                        responseControl.setPasswordPolicyError(PasswordPolicyErrorEnum.PASSWORD_TOO_YOUNG);
                        modifyContext.addResponseControl((Control)responseControl);
                    }
                    throw new LdapOperationException(ResultCodeEnum.CONSTRAINT_VIOLATION, "password is too young to update");
                }
                byte[] newPassword = pwdModDetails.getNewPwd();
                try {
                    this.check((OperationContext)modifyContext, entry, newPassword, policyConfig);
                }
                catch (PasswordPolicyException e) {
                    if (isPPolicyReqCtrlPresent) {
                        PasswordPolicyResponseImpl responseControl = new PasswordPolicyResponseImpl();
                        responseControl.setPasswordPolicyError(PasswordPolicyErrorEnum.get((int)e.getErrorCode()));
                        modifyContext.addResponseControl((Control)responseControl);
                    }
                    throw new LdapOperationException(ResultCodeEnum.CONSTRAINT_VIOLATION, e.getMessage(), (Throwable)e);
                }
                int histSize = policyConfig.getPwdInHistory();
                Modification pwdRemHistMod = null;
                DefaultModification pwdAddHistMod = null;
                String pwdChangedTime = DateUtils.getGeneralizedTime((TimeProvider)this.directoryService.getTimeProvider());
                if (histSize > 0) {
                    Attribute pwdHistoryAt = entry.get(this.pwdHistoryAT);
                    if (pwdHistoryAt == null) {
                        pwdHistoryAt = new DefaultAttribute(this.pwdHistoryAT);
                    }
                    pwdRemHistMod = this.buildPwdHistory(modifyContext, pwdHistoryAt, histSize, newPassword, isPPolicyReqCtrlPresent);
                    PasswordHistory newPwdHist = new PasswordHistory(pwdChangedTime, newPassword);
                    pwdHistoryAt.add((byte[][])new byte[][]{newPwdHist.getHistoryValue()});
                    pwdAddHistMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, pwdHistoryAt);
                }
                this.next(modifyContext);
                this.invalidateAuthenticatorCaches(modifyContext.getDn());
                LookupOperationContext lookupContext = new LookupOperationContext(this.adminSession, modifyContext.getDn(), SchemaConstants.ALL_ATTRIBUTES_ARRAY);
                lookupContext.setPartition(modifyContext.getPartition());
                lookupContext.setTransaction(modifyContext.getTransaction());
                entry = this.directoryService.getPartitionNexus().lookup(lookupContext);
                if (policyConfig.getPwdMinAge() > 0 || policyConfig.getPwdMaxAge() > 0) {
                    DefaultAttribute pwdChangedTimeAt = new DefaultAttribute(this.pwdChangedTimeAT);
                    pwdChangedTimeAt.add(new String[]{pwdChangedTime});
                    DefaultModification pwdChangedTimeMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, (Attribute)pwdChangedTimeAt);
                    mods.add((Modification)pwdChangedTimeMod);
                }
                if (pwdAddHistMod != null) {
                    mods.add((Modification)pwdAddHistMod);
                }
                if (pwdRemHistMod != null) {
                    mods.add(pwdRemHistMod);
                }
                if (policyConfig.isPwdMustChange()) {
                    DefaultModification pwdMustChangeMod;
                    DefaultAttribute pwdMustChangeAt = new DefaultAttribute(this.pwdResetAT);
                    if (modifyContext.getSession().isAnAdministrator()) {
                        pwdMustChangeAt.add(new String[]{"TRUE"});
                        pwdMustChangeMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, (Attribute)pwdMustChangeAt);
                    } else {
                        pwdMustChangeMod = new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, (Attribute)pwdMustChangeAt);
                        removePwdReset = true;
                    }
                    mods.add((Modification)pwdMustChangeMod);
                }
            }
            this.processModifyAddPwdAttributes(entry, mods, pwdModDetails);
            String csnVal = this.directoryService.getCSN().toString();
            DefaultModification csnMod = new DefaultModification(ModificationOperation.REPLACE_ATTRIBUTE, this.directoryService.getAtProvider().getEntryCSN(), new String[]{csnVal});
            mods.add((Modification)csnMod);
            ModifyOperationContext internalModifyCtx = new ModifyOperationContext(this.adminSession);
            internalModifyCtx.setPushToEvtInterceptor(true);
            internalModifyCtx.setDn(modifyContext.getDn());
            internalModifyCtx.setEntry(entry);
            internalModifyCtx.setModItems(mods);
            this.internalModify((OperationContext)modifyContext, internalModifyCtx);
            if (removePwdReset || pwdModDetails.isDelete()) {
                userSession.setPwdMustChange(false);
            }
        }
    }

    Modification buildPwdHistory(ModifyOperationContext modifyContext, Attribute pwdHistoryAt, int histSize, byte[] newPassword, boolean isPPolicyReqCtrlPresent) throws LdapOperationException {
        ArrayList<PasswordHistory> pwdHistLst = new ArrayList<PasswordHistory>();
        for (Value value : pwdHistoryAt) {
            boolean matched;
            PasswordHistory pwdh = new PasswordHistory(Strings.utf8ToString((byte[])value.getBytes()));
            if (!modifyContext.getSession().isAnAdministrator() && (matched = MessageDigest.isEqual(newPassword, pwdh.getPassword()))) {
                if (isPPolicyReqCtrlPresent) {
                    PasswordPolicyResponseImpl responseControl = new PasswordPolicyResponseImpl();
                    responseControl.setPasswordPolicyError(PasswordPolicyErrorEnum.PASSWORD_IN_HISTORY);
                    modifyContext.addResponseControl((Control)responseControl);
                }
                throw new LdapOperationException(ResultCodeEnum.CONSTRAINT_VIOLATION, "invalid reuse of password present in password history");
            }
            pwdHistLst.add(pwdh);
        }
        DefaultModification pwdRemHistMod = null;
        if (pwdHistLst.size() >= histSize) {
            Collections.sort(pwdHistLst);
            PasswordHistory remPwdHist = (PasswordHistory)pwdHistLst.toArray()[histSize - 1];
            DefaultAttribute tempAt = new DefaultAttribute(this.pwdHistoryAT);
            tempAt.add((byte[][])new byte[][]{remPwdHist.getHistoryValue()});
            pwdRemHistMod = new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, (Attribute)tempAt);
        }
        return pwdRemHistMod;
    }

    private void processModifyAddPwdAttributes(Entry entry, List<Modification> mods, PwdModDetailsHolder pwdModDetails) {
        Attribute pwdGraceUseTimeAt;
        Attribute pwdFailureTimeAt = entry.get(this.pwdFailurTimeAT);
        if (pwdFailureTimeAt != null) {
            mods.add((Modification)new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, pwdFailureTimeAt));
        }
        if ((pwdGraceUseTimeAt = entry.get(this.pwdGraceUseTimeAT)) != null) {
            mods.add((Modification)new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, pwdGraceUseTimeAt));
        }
        if (pwdModDetails.isDelete()) {
            Attribute pwdAccountLockedTimeAt;
            Attribute pwdMustChangeAt;
            Attribute pwdChangedTimeAt;
            Attribute pwdHistory = entry.get(this.pwdHistoryAT);
            if (pwdHistory != null) {
                mods.add((Modification)new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, pwdHistory));
            }
            if ((pwdChangedTimeAt = entry.get(this.pwdChangedTimeAT)) != null) {
                mods.add((Modification)new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, pwdChangedTimeAt));
            }
            if ((pwdMustChangeAt = entry.get(this.pwdResetAT)) != null) {
                mods.add((Modification)new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, pwdMustChangeAt));
            }
            if ((pwdAccountLockedTimeAt = entry.get(this.pwdAccountLockedTimeAT)) != null) {
                mods.add((Modification)new DefaultModification(ModificationOperation.REMOVE_ATTRIBUTE, pwdAccountLockedTimeAt));
            }
        }
    }

    private void checkPwdMustChange(ModifyOperationContext modifyContext, CoreSession userSession, PwdModDetailsHolder pwdModDetails, boolean isPPolicyReqCtrlPresent) throws LdapNoPermissionException {
        if (userSession.isPwdMustChange() && !pwdModDetails.isDelete() && pwdModDetails.isOtherModExists()) {
            if (isPPolicyReqCtrlPresent) {
                PasswordPolicyResponseImpl responseControl = new PasswordPolicyResponseImpl();
                responseControl.setPasswordPolicyError(PasswordPolicyErrorEnum.CHANGE_AFTER_RESET);
                modifyContext.addResponseControl((Control)responseControl);
            }
            throw new LdapNoPermissionException("Password should be reset before making any changes to this entry");
        }
    }

    private void checkOldPwdRequired(ModifyOperationContext modifyContext, PasswordPolicyConfiguration policyConfig, PwdModDetailsHolder pwdModDetails, boolean isPPolicyReqCtrlPresent) throws LdapNoPermissionException {
        if (policyConfig.isPwdSafeModify() && !pwdModDetails.isDelete() && pwdModDetails.isAddOrReplace()) {
            String msg = "trying to update password attribute without the supplying the old password";
            LOG.debug(msg);
            if (isPPolicyReqCtrlPresent) {
                PasswordPolicyResponseImpl responseControl = new PasswordPolicyResponseImpl();
                responseControl.setPasswordPolicyError(PasswordPolicyErrorEnum.MUST_SUPPLY_OLD_PASSWORD);
                modifyContext.addResponseControl((Control)responseControl);
            }
            throw new LdapNoPermissionException(msg);
        }
    }

    private void checkChangePwdAllowed(ModifyOperationContext modifyContext, PasswordPolicyConfiguration policyConfig, boolean isPPolicyReqCtrlPresent) throws LdapNoPermissionException {
        if (!policyConfig.isPwdAllowUserChange() && !modifyContext.getSession().isAnAdministrator()) {
            if (isPPolicyReqCtrlPresent) {
                PasswordPolicyResponseImpl responseControl = new PasswordPolicyResponseImpl();
                responseControl.setPasswordPolicyError(PasswordPolicyErrorEnum.PASSWORD_MOD_NOT_ALLOWED);
                modifyContext.addResponseControl((Control)responseControl);
            }
            throw new LdapNoPermissionException();
        }
    }

    public void move(MoveOperationContext moveContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)moveContext);
        }
        this.checkAuthenticated((OperationContext)moveContext);
        this.checkPwdReset((OperationContext)moveContext);
        this.next(moveContext);
        this.invalidateAuthenticatorCaches(moveContext.getDn());
    }

    public void moveAndRename(MoveAndRenameOperationContext moveAndRenameContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)moveAndRenameContext);
        }
        this.checkAuthenticated((OperationContext)moveAndRenameContext);
        this.checkPwdReset((OperationContext)moveAndRenameContext);
        this.next(moveAndRenameContext);
        this.invalidateAuthenticatorCaches(moveAndRenameContext.getDn());
    }

    public void rename(RenameOperationContext renameContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)renameContext);
        }
        this.checkAuthenticated((OperationContext)renameContext);
        this.checkPwdReset((OperationContext)renameContext);
        this.next(renameContext);
        this.invalidateAuthenticatorCaches(renameContext.getDn());
    }

    public EntryFilteringCursor search(SearchOperationContext searchContext) throws LdapException {
        if (IS_DEBUG) {
            LOG.debug("Operation Context: {}", (Object)searchContext);
        }
        this.checkAuthenticated((OperationContext)searchContext);
        this.checkPwdReset((OperationContext)searchContext);
        return this.next(searchContext);
    }

    public void unbind(UnbindOperationContext unbindContext) throws LdapException {
        this.next(unbindContext);
    }

    private void checkAuthenticated(OperationContext operation) throws LdapException {
        if (operation.getSession().isAnonymous() && !this.directoryService.isAllowAnonymousAccess() && !operation.getDn().isEmpty()) {
            String msg = I18n.err((I18n)I18n.ERR_5, (Object[])new Object[]{operation.getName()});
            LOG.error(msg);
            throw new LdapNoPermissionException(msg);
        }
    }

    public void loadPwdPolicyStateAttributeTypes() throws LdapException {
        this.pwdResetAT = this.schemaManager.lookupAttributeTypeRegistry("pwdReset");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdResetAT);
        this.pwdChangedTimeAT = this.schemaManager.lookupAttributeTypeRegistry("pwdChangedTime");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdChangedTimeAT);
        this.pwdHistoryAT = this.schemaManager.lookupAttributeTypeRegistry("pwdHistory");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdHistoryAT);
        this.pwdFailurTimeAT = this.schemaManager.lookupAttributeTypeRegistry("pwdFailureTime");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdFailurTimeAT);
        this.pwdAccountLockedTimeAT = this.schemaManager.lookupAttributeTypeRegistry("pwdAccountLockedTime");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdAccountLockedTimeAT);
        this.pwdLastSuccessAT = this.schemaManager.lookupAttributeTypeRegistry("pwdLastSuccess");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdLastSuccessAT);
        this.pwdGraceUseTimeAT = this.schemaManager.lookupAttributeTypeRegistry("pwdGraceUseTime");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdGraceUseTimeAT);
        this.pwdPolicySubentryAT = this.schemaManager.lookupAttributeTypeRegistry("pwdPolicySubentry");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdPolicySubentryAT);
        this.pwdStartTimeAT = this.schemaManager.lookupAttributeTypeRegistry("pwdStartTime");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdStartTimeAT);
        this.pwdEndTimeAT = this.schemaManager.lookupAttributeTypeRegistry("pwdEndTime");
        PWD_POLICY_STATE_ATTRIBUTE_TYPES.add(this.pwdEndTimeAT);
    }

    private void check(OperationContext operationContext, Entry entry, byte[] password, PasswordPolicyConfiguration policyConfig) throws LdapException {
        if (operationContext.getSession().isAnAdministrator()) {
            return;
        }
        CheckQualityEnum qualityVal = policyConfig.getPwdCheckQuality();
        if (qualityVal == CheckQualityEnum.NO_CHECK) {
            return;
        }
        LdapSecurityConstants secConst = PasswordUtil.findAlgorithm((byte[])password);
        if (secConst != null) {
            if (qualityVal == CheckQualityEnum.CHECK_ACCEPT) {
                return;
            }
            throw new PasswordPolicyException("cannot verify the quality of the non-cleartext passwords", PasswordPolicyErrorEnum.INSUFFICIENT_PASSWORD_QUALITY.getValue());
        }
        String strPassword = Strings.utf8ToString((byte[])password);
        this.validatePasswordLength(strPassword, policyConfig);
        PasswordValidator passwordValidator = policyConfig.getPwdValidator();
        if (passwordValidator == null) {
            passwordValidator = new DefaultPasswordValidator();
        }
        passwordValidator.validate(strPassword, entry);
    }

    private void validatePasswordLength(String password, PasswordPolicyConfiguration policyConfig) throws PasswordPolicyException {
        int maxLen = policyConfig.getPwdMaxLength();
        int minLen = policyConfig.getPwdMinLength();
        int pwdLen = password.length();
        if (maxLen > 0 && pwdLen > maxLen) {
            throw new PasswordPolicyException("Password should not have more than " + maxLen + " characters", PasswordPolicyErrorEnum.INSUFFICIENT_PASSWORD_QUALITY.getValue());
        }
        if (minLen > 0 && pwdLen < minLen) {
            throw new PasswordPolicyException("Password should have a minimum of " + minLen + " characters", PasswordPolicyErrorEnum.PASSWORD_TOO_SHORT.getValue());
        }
    }

    private int getPwdTimeBeforeExpiry(Entry userEntry, PasswordPolicyConfiguration policyConfig) throws LdapException {
        if (policyConfig.getPwdMaxAge() == 0) {
            return 0;
        }
        int warningAge = policyConfig.getPwdExpireWarning();
        if (warningAge <= 0) {
            return 0;
        }
        Attribute pwdChangedTimeAt = userEntry.get(this.pwdChangedTimeAT);
        if (pwdChangedTimeAt == null) {
            pwdChangedTimeAt = userEntry.get(this.directoryService.getAtProvider().getCreateTimestamp());
        }
        long changedTime = DateUtils.getDate((String)pwdChangedTimeAt.getString()).getTime();
        long currentTime = this.directoryService.getTimeProvider().currentIimeMillis();
        long pwdAge = (currentTime - changedTime) / 1000L;
        if (pwdAge > (long)policyConfig.getPwdMaxAge()) {
            return 0;
        }
        warningAge = policyConfig.getPwdMaxAge() - warningAge;
        if (pwdAge >= (long)warningAge) {
            long timeBeforeExpiration = (long)policyConfig.getPwdMaxAge() - pwdAge;
            if (timeBeforeExpiration > Integer.MAX_VALUE) {
                timeBeforeExpiration = Integer.MAX_VALUE;
            }
            return (int)timeBeforeExpiration;
        }
        return 0;
    }

    private boolean isPwdTooYoung(OperationContext operationContext, Entry userEntry, PasswordPolicyConfiguration policyConfig) throws LdapException {
        if (operationContext.getSession().isAnAdministrator()) {
            return false;
        }
        if (policyConfig.getPwdMinAge() == 0) {
            return false;
        }
        CoreSession userSession = operationContext.getSession();
        if (policyConfig.isPwdMustChange() && userSession.isPwdMustChange()) {
            return false;
        }
        Attribute pwdChangedTimeAt = userEntry.get(this.pwdChangedTimeAT);
        if (pwdChangedTimeAt != null) {
            long currentTime;
            long changedTime = DateUtils.getDate((String)pwdChangedTimeAt.getString()).getTime();
            if ((changedTime += (long)policyConfig.getPwdMinAge() * 1000L) > (currentTime = this.directoryService.getTimeProvider().currentIimeMillis())) {
                return true;
            }
        }
        return false;
    }

    private boolean isPwdMustReset(Entry userEntry) throws LdapException {
        boolean mustChange = false;
        Attribute pwdResetAt = userEntry.get(this.pwdResetAT);
        if (pwdResetAt != null) {
            mustChange = Boolean.parseBoolean(pwdResetAt.getString());
        }
        return mustChange;
    }

    private PwdModDetailsHolder getPwdModDetails(ModifyOperationContext modifyContext, PasswordPolicyConfiguration policyConfig) throws LdapException {
        PwdModDetailsHolder pwdModDetails = new PwdModDetailsHolder();
        List mods = modifyContext.getModItems();
        for (Modification mod : mods) {
            Attribute at = mod.getAttribute();
            AttributeType passwordAttribute = this.schemaManager.lookupAttributeTypeRegistry(policyConfig.getPwdAttribute());
            if (at.getAttributeType().equals((Object)passwordAttribute)) {
                pwdModDetails.setPwdModPresent(true);
                ModificationOperation op = mod.getOperation();
                if (op == ModificationOperation.REMOVE_ATTRIBUTE) {
                    pwdModDetails.setDelete(true);
                } else if (op == ModificationOperation.REPLACE_ATTRIBUTE || op == ModificationOperation.ADD_ATTRIBUTE) {
                    pwdModDetails.setAddOrReplace(true);
                    pwdModDetails.setNewPwd(at.getBytes());
                }
                switch (op) {
                    case REMOVE_ATTRIBUTE: {
                        pwdModDetails.setDelete(true);
                        break;
                    }
                    case REPLACE_ATTRIBUTE: 
                    case ADD_ATTRIBUTE: {
                        pwdModDetails.setAddOrReplace(true);
                        pwdModDetails.setNewPwd(at.getBytes());
                        break;
                    }
                }
                continue;
            }
            pwdModDetails.setOtherModExists(true);
        }
        return pwdModDetails;
    }

    private void checkPwdReset(OperationContext opContext) throws LdapException {
        CoreSession session;
        if (this.directoryService.isPwdPolicyEnabled() && (session = opContext.getSession()).isPwdMustChange()) {
            boolean isPPolicyReqCtrlPresent = opContext.hasRequestControl("1.3.6.1.4.1.42.2.27.8.5.1");
            if (isPPolicyReqCtrlPresent) {
                PasswordPolicyResponseImpl responseControl = new PasswordPolicyResponseImpl();
                responseControl.setPasswordPolicyError(PasswordPolicyErrorEnum.CHANGE_AFTER_RESET);
                opContext.addResponseControl((Control)responseControl);
            }
            throw new LdapNoPermissionException("password needs to be reset before performing this operation");
        }
    }

    public PasswordPolicyConfiguration getPwdPolicy(Entry userEntry) throws LdapException {
        Attribute pwdPolicySubentry;
        if (this.pwdPolicyContainer == null) {
            return null;
        }
        if (userEntry == null) {
            return this.pwdPolicyContainer.getDefaultPolicy();
        }
        if (this.pwdPolicyContainer.hasCustomConfigs() && (pwdPolicySubentry = userEntry.get(this.pwdPolicySubentryAT)) != null) {
            Dn configDn = this.dnFactory.create(pwdPolicySubentry.getString());
            PasswordPolicyConfiguration custom = this.pwdPolicyContainer.getPolicyConfig(configDn);
            if (custom != null) {
                return custom;
            }
            LOG.warn("The custom password policy for the user entry {} is not found, returning default policy configuration", (Object)userEntry.getDn());
        }
        return this.pwdPolicyContainer.getDefaultPolicy();
    }

    public void setPwdPolicies(PpolicyConfigContainer policyContainer) {
        this.pwdPolicyContainer = policyContainer;
    }

    public boolean isPwdPolicyEnabled() {
        return this.pwdPolicyContainer != null && (this.pwdPolicyContainer.getDefaultPolicy() != null || this.pwdPolicyContainer.hasCustomConfigs());
    }

    public PpolicyConfigContainer getPwdPolicyContainer() {
        return this.pwdPolicyContainer;
    }

    private void purgeFailureTimes(PasswordPolicyConfiguration config, Attribute pwdFailTimeAt) {
        long interval = config.getPwdFailureCountInterval();
        if (interval == 0L) {
            return;
        }
        interval *= 1000L;
        long currentTime = this.directoryService.getTimeProvider().currentIimeMillis();
        Iterator itr = pwdFailTimeAt.iterator();
        while (itr.hasNext()) {
            Value value = (Value)itr.next();
            String failureTime = value.getString();
            long time = DateUtils.getDate((String)failureTime).getTime();
            if (currentTime < (time += interval)) continue;
            itr.remove();
        }
    }

    private static class PwdModDetailsHolder {
        private boolean pwdModPresent = false;
        private boolean isDelete = false;
        private boolean isAddOrReplace = false;
        private boolean otherModExists = false;
        private byte[] newPwd;

        private PwdModDetailsHolder() {
        }

        public boolean isPwdModPresent() {
            return this.pwdModPresent;
        }

        public void setPwdModPresent(boolean pwdModPresent) {
            this.pwdModPresent = pwdModPresent;
        }

        public boolean isDelete() {
            return this.isDelete;
        }

        public void setDelete(boolean isDelete) {
            this.isDelete = isDelete;
        }

        public boolean isAddOrReplace() {
            return this.isAddOrReplace;
        }

        public void setAddOrReplace(boolean isAddOrReplace) {
            this.isAddOrReplace = isAddOrReplace;
        }

        public boolean isOtherModExists() {
            return this.otherModExists;
        }

        public void setOtherModExists(boolean otherModExists) {
            this.otherModExists = otherModExists;
        }

        public byte[] getNewPwd() {
            return this.newPwd;
        }

        public void setNewPwd(byte[] newPwd) {
            this.newPwd = newPwd;
        }
    }
}

