/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dolphinscheduler.common.utils;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.math.RoundingMode;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.StringTokenizer;
import java.util.regex.Pattern;
import org.apache.commons.configuration.Configuration;
import org.apache.dolphinscheduler.common.Constants;
import org.apache.dolphinscheduler.common.shell.ShellExecutor;
import org.apache.dolphinscheduler.common.utils.LoggerUtils;
import org.apache.dolphinscheduler.common.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import oshi.SystemInfo;
import oshi.hardware.CentralProcessor;
import oshi.hardware.GlobalMemory;
import oshi.hardware.HardwareAbstractionLayer;

public class OSUtils {
    private static final Logger logger = LoggerFactory.getLogger(OSUtils.class);
    public static final ThreadLocal<Logger> taskLoggerThreadLocal = new ThreadLocal();
    private static final SystemInfo SI = new SystemInfo();
    public static final String TWO_DECIMAL = "0.00";
    private static HardwareAbstractionLayer hal = SI.getHardware();

    private OSUtils() {
    }

    public static double memoryUsage() {
        GlobalMemory memory = hal.getMemory();
        double memoryUsage = (double)(memory.getTotal() - memory.getAvailable() - memory.getSwapUsed()) * 0.1 / (double)memory.getTotal() * 10.0;
        DecimalFormat df = new DecimalFormat(TWO_DECIMAL);
        df.setRoundingMode(RoundingMode.HALF_UP);
        return Double.parseDouble(df.format(memoryUsage));
    }

    public static double availablePhysicalMemorySize() {
        GlobalMemory memory = hal.getMemory();
        double availablePhysicalMemorySize = (double)(memory.getAvailable() + memory.getSwapUsed()) / 1024.0 / 1024.0 / 1024.0;
        DecimalFormat df = new DecimalFormat(TWO_DECIMAL);
        df.setRoundingMode(RoundingMode.HALF_UP);
        return Double.parseDouble(df.format(availablePhysicalMemorySize));
    }

    public static double totalMemorySize() {
        GlobalMemory memory = hal.getMemory();
        double availablePhysicalMemorySize = (double)memory.getTotal() / 1024.0 / 1024.0 / 1024.0;
        DecimalFormat df = new DecimalFormat(TWO_DECIMAL);
        df.setRoundingMode(RoundingMode.HALF_UP);
        return Double.parseDouble(df.format(availablePhysicalMemorySize));
    }

    public static double loadAverage() {
        double loadAverage = hal.getProcessor().getSystemLoadAverage();
        DecimalFormat df = new DecimalFormat(TWO_DECIMAL);
        df.setRoundingMode(RoundingMode.HALF_UP);
        return Double.parseDouble(df.format(loadAverage));
    }

    public static double cpuUsage() {
        CentralProcessor processor = hal.getProcessor();
        double cpuUsage = processor.getSystemCpuLoad();
        DecimalFormat df = new DecimalFormat(TWO_DECIMAL);
        df.setRoundingMode(RoundingMode.HALF_UP);
        return Double.parseDouble(df.format(cpuUsage));
    }

    public static List<String> getUserList() {
        try {
            if (OSUtils.isMacOS()) {
                return OSUtils.getUserListFromMac();
            }
            if (OSUtils.isWindows()) {
                return OSUtils.getUserListFromWindows();
            }
            return OSUtils.getUserListFromLinux();
        }
        catch (Exception e) {
            logger.error(e.getMessage(), (Throwable)e);
            return Collections.emptyList();
        }
    }

    private static List<String> getUserListFromLinux() throws IOException {
        ArrayList<String> userList = new ArrayList<String>();
        try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(new FileInputStream("/etc/passwd")));){
            String line;
            while ((line = bufferedReader.readLine()) != null) {
                if (!line.contains(":")) continue;
                String[] userInfo = line.split(":");
                userList.add(userInfo[0]);
            }
        }
        return userList;
    }

    private static List<String> getUserListFromMac() throws IOException {
        String result = OSUtils.exeCmd("dscl . list /users");
        if (StringUtils.isNotEmpty(result)) {
            return Arrays.asList(result.split("\n"));
        }
        return Collections.emptyList();
    }

    private static List<String> getUserListFromWindows() throws IOException {
        String result = OSUtils.exeCmd("net user");
        String[] lines = result.split("\n");
        int startPos = 0;
        int endPos = lines.length - 2;
        for (int i = 0; i < lines.length; ++i) {
            if (lines[i].isEmpty()) continue;
            int count = 0;
            if (lines[i].charAt(0) == '-') {
                for (int j = 0; j < lines[i].length(); ++j) {
                    if (lines[i].charAt(i) != '-') continue;
                    ++count;
                }
            }
            if (count != lines[i].length()) continue;
            startPos = i + 1;
            break;
        }
        ArrayList<String> users = new ArrayList<String>();
        while (startPos <= endPos) {
            Pattern pattern = Pattern.compile("\\s+");
            users.addAll(Arrays.asList(pattern.split(lines[startPos])));
            ++startPos;
        }
        return users;
    }

    public static boolean createUser(String userName) {
        try {
            String userGroup = OSUtils.getGroup();
            if (StringUtils.isEmpty(userGroup)) {
                String errorLog = String.format("%s group does not exist for this operating system.", userGroup);
                LoggerUtils.logError(Optional.ofNullable(logger), errorLog);
                LoggerUtils.logError(Optional.ofNullable(taskLoggerThreadLocal.get()), errorLog);
                return false;
            }
            if (OSUtils.isMacOS()) {
                OSUtils.createMacUser(userName, userGroup);
            } else if (OSUtils.isWindows()) {
                OSUtils.createWindowsUser(userName, userGroup);
            } else {
                OSUtils.createLinuxUser(userName, userGroup);
            }
            return true;
        }
        catch (Exception e) {
            LoggerUtils.logError(Optional.ofNullable(logger), e);
            LoggerUtils.logError(Optional.ofNullable(taskLoggerThreadLocal.get()), e);
            return false;
        }
    }

    private static void createLinuxUser(String userName, String userGroup) throws IOException {
        String infoLog1 = String.format("create linux os user : %s", userName);
        LoggerUtils.logInfo(Optional.ofNullable(logger), infoLog1);
        LoggerUtils.logInfo(Optional.ofNullable(taskLoggerThreadLocal.get()), infoLog1);
        String cmd = String.format("sudo useradd -g %s %s", userGroup, userName);
        String infoLog2 = String.format("execute cmd : %s", cmd);
        LoggerUtils.logInfo(Optional.ofNullable(logger), infoLog2);
        LoggerUtils.logInfo(Optional.ofNullable(taskLoggerThreadLocal.get()), infoLog2);
        OSUtils.exeCmd(cmd);
    }

    private static void createMacUser(String userName, String userGroup) throws IOException {
        Optional<Logger> optionalLogger = Optional.ofNullable(logger);
        Optional<Logger> optionalTaskLogger = Optional.ofNullable(taskLoggerThreadLocal.get());
        String infoLog1 = String.format("create mac os user : %s", userName);
        LoggerUtils.logInfo(optionalLogger, infoLog1);
        LoggerUtils.logInfo(optionalTaskLogger, infoLog1);
        String createUserCmd = String.format("sudo sysadminctl -addUser %s -password %s", userName, userName);
        String infoLog2 = String.format("create user command : %s", createUserCmd);
        LoggerUtils.logInfo(optionalLogger, infoLog2);
        LoggerUtils.logInfo(optionalTaskLogger, infoLog2);
        OSUtils.exeCmd(createUserCmd);
        String appendGroupCmd = String.format("sudo dseditgroup -o edit -a %s -t user %s", userName, userGroup);
        String infoLog3 = String.format("append user to group : %s", appendGroupCmd);
        LoggerUtils.logInfo(optionalLogger, infoLog3);
        LoggerUtils.logInfo(optionalTaskLogger, infoLog3);
        OSUtils.exeCmd(appendGroupCmd);
    }

    private static void createWindowsUser(String userName, String userGroup) throws IOException {
        String infoLog1 = String.format("create windows os user : %s", userName);
        LoggerUtils.logInfo(Optional.ofNullable(logger), infoLog1);
        LoggerUtils.logInfo(Optional.ofNullable(taskLoggerThreadLocal.get()), infoLog1);
        String userCreateCmd = String.format("net user \"%s\" /add", userName);
        String infoLog2 = String.format("execute create user command : %s", userCreateCmd);
        LoggerUtils.logInfo(Optional.ofNullable(logger), infoLog2);
        LoggerUtils.logInfo(Optional.ofNullable(taskLoggerThreadLocal.get()), infoLog2);
        OSUtils.exeCmd(userCreateCmd);
        String appendGroupCmd = String.format("net localgroup \"%s\" \"%s\" /add", userGroup, userName);
        String infoLog3 = String.format("execute append user to group : %s", appendGroupCmd);
        LoggerUtils.logInfo(Optional.ofNullable(logger), infoLog3);
        LoggerUtils.logInfo(Optional.ofNullable(taskLoggerThreadLocal.get()), infoLog3);
        OSUtils.exeCmd(appendGroupCmd);
    }

    public static String getGroup() throws IOException {
        if (OSUtils.isWindows()) {
            String currentProcUserName = System.getProperty("user.name");
            String result = OSUtils.exeCmd(String.format("net user \"%s\"", currentProcUserName));
            String line = result.split("\n")[22];
            String group = Pattern.compile("\\s+").split(line)[1];
            if (group.charAt(0) == '*') {
                return group.substring(1);
            }
            return group;
        }
        String result = OSUtils.exeCmd("groups");
        if (StringUtils.isNotEmpty(result)) {
            String[] groupInfo = result.split(" ");
            return groupInfo[0];
        }
        return null;
    }

    public static String exeCmd(String command) throws IOException {
        StringTokenizer st = new StringTokenizer(command);
        String[] cmdArray = new String[st.countTokens()];
        int i = 0;
        while (st.hasMoreTokens()) {
            cmdArray[i] = st.nextToken();
            ++i;
        }
        return OSUtils.exeShell(cmdArray);
    }

    public static String exeShell(String[] command) throws IOException {
        return ShellExecutor.execCommand(command);
    }

    public static int getProcessID() {
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        return Integer.parseInt(runtimeMXBean.getName().split("@")[0]);
    }

    public static String getHost() {
        try {
            return InetAddress.getLocalHost().getHostAddress();
        }
        catch (UnknownHostException e) {
            logger.error(e.getMessage(), (Throwable)e);
            return null;
        }
    }

    public static boolean isMacOS() {
        return OSUtils.getOSName().startsWith("Mac");
    }

    public static boolean isWindows() {
        return OSUtils.getOSName().startsWith("Windows");
    }

    public static String getOSName() {
        return System.getProperty("os.name");
    }

    public static Boolean checkResource(double systemCpuLoad, double systemReservedMemory) {
        double loadAverage = OSUtils.loadAverage();
        double availablePhysicalMemorySize = OSUtils.availablePhysicalMemorySize();
        if (loadAverage > systemCpuLoad || availablePhysicalMemorySize < systemReservedMemory) {
            logger.warn("load is too high or availablePhysicalMemorySize(G) is too low, it's availablePhysicalMemorySize(G):{},loadAvg:{}", (Object)availablePhysicalMemorySize, (Object)loadAverage);
            return false;
        }
        return true;
    }

    public static Boolean checkResource(Configuration conf, Boolean isMaster) {
        double systemReservedMemory;
        double systemCpuLoad;
        if (Boolean.TRUE.equals(isMaster)) {
            systemCpuLoad = conf.getDouble("master.max.cpuload.avg", (double)Constants.DEFAULT_MASTER_CPU_LOAD);
            systemReservedMemory = conf.getDouble("master.reserved.memory", Constants.DEFAULT_MASTER_RESERVED_MEMORY);
        } else {
            systemCpuLoad = conf.getDouble("worker.max.cpuload.avg", (double)Constants.DEFAULT_WORKER_CPU_LOAD);
            systemReservedMemory = conf.getDouble("worker.reserved.memory", Constants.DEFAULT_WORKER_RESERVED_MEMORY);
        }
        return OSUtils.checkResource(systemCpuLoad, systemReservedMemory);
    }
}

