/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dolphinscheduler.common.utils;

import java.io.IOException;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Collections;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.apache.dolphinscheduler.common.Constants;
import org.apache.dolphinscheduler.common.utils.PropertyUtils;
import org.apache.http.conn.util.InetAddressUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetUtils {
    private static final String NETWORK_PRIORITY_DEFAULT = "default";
    private static final String NETWORK_PRIORITY_INNER = "inner";
    private static final String NETWORK_PRIORITY_OUTER = "outer";
    private static final Logger logger = LoggerFactory.getLogger(NetUtils.class);
    private static InetAddress LOCAL_ADDRESS = null;
    private static volatile String HOST_ADDRESS;

    private NetUtils() {
        throw new UnsupportedOperationException("Construct NetUtils");
    }

    public static String getAddr(String host, int port) {
        return String.format("%s:%d", host, port);
    }

    public static String getAddr(int port) {
        return NetUtils.getAddr(NetUtils.getHost(), port);
    }

    public static String getHost(InetAddress inetAddress) {
        if (inetAddress != null) {
            if (Constants.KUBERNETES_MODE) {
                String canonicalHost = inetAddress.getCanonicalHostName();
                String[] items = canonicalHost.split("\\.");
                if (items.length == 6 && "svc".equals(items[3])) {
                    return String.format("%s.%s", items[0], items[1]);
                }
                return canonicalHost;
            }
            return inetAddress.getHostAddress();
        }
        return null;
    }

    public static String getHost() {
        if (HOST_ADDRESS != null) {
            return HOST_ADDRESS;
        }
        InetAddress address = NetUtils.getLocalAddress();
        if (address != null) {
            HOST_ADDRESS = NetUtils.getHost(address);
            return HOST_ADDRESS;
        }
        return Constants.KUBERNETES_MODE ? "localhost" : "127.0.0.1";
    }

    private static InetAddress getLocalAddress() {
        if (null != LOCAL_ADDRESS) {
            return LOCAL_ADDRESS;
        }
        return NetUtils.getLocalAddress0();
    }

    private static synchronized InetAddress getLocalAddress0() {
        if (null != LOCAL_ADDRESS) {
            return LOCAL_ADDRESS;
        }
        InetAddress localAddress = null;
        try {
            NetworkInterface networkInterface = NetUtils.findNetworkInterface();
            if (networkInterface != null) {
                Enumeration<InetAddress> addresses = networkInterface.getInetAddresses();
                while (addresses.hasMoreElements()) {
                    Optional<InetAddress> addressOp = NetUtils.toValidAddress(addresses.nextElement());
                    if (!addressOp.isPresent()) continue;
                    try {
                        if (!addressOp.get().isReachable(200)) continue;
                        LOCAL_ADDRESS = addressOp.get();
                        return LOCAL_ADDRESS;
                    }
                    catch (IOException e) {
                        logger.warn("test address id reachable io exception", (Throwable)e);
                    }
                }
            }
            localAddress = InetAddress.getLocalHost();
        }
        catch (UnknownHostException e) {
            logger.warn("InetAddress get LocalHost exception", (Throwable)e);
        }
        Optional<InetAddress> addressOp = NetUtils.toValidAddress(localAddress);
        if (addressOp.isPresent()) {
            LOCAL_ADDRESS = addressOp.get();
        }
        return LOCAL_ADDRESS;
    }

    private static Optional<InetAddress> toValidAddress(InetAddress address) {
        if (address instanceof Inet6Address) {
            Inet6Address v6Address = (Inet6Address)address;
            if (NetUtils.isPreferIPV6Address()) {
                return Optional.ofNullable(NetUtils.normalizeV6Address(v6Address));
            }
        }
        if (NetUtils.isValidV4Address(address)) {
            return Optional.of(address);
        }
        return Optional.empty();
    }

    private static InetAddress normalizeV6Address(Inet6Address address) {
        String addr = address.getHostAddress();
        int i = addr.lastIndexOf(37);
        if (i > 0) {
            try {
                return InetAddress.getByName(addr.substring(0, i) + '%' + address.getScopeId());
            }
            catch (UnknownHostException e) {
                logger.debug("Unknown IPV6 address: ", (Throwable)e);
            }
        }
        return address;
    }

    public static boolean isValidV4Address(InetAddress address) {
        if (address == null || address.isLoopbackAddress()) {
            return false;
        }
        String name = address.getHostAddress();
        return name != null && InetAddressUtils.isIPv4Address((String)name) && !address.isAnyLocalAddress() && !address.isLoopbackAddress();
    }

    private static boolean isPreferIPV6Address() {
        return Boolean.getBoolean("java.net.preferIPv6Addresses");
    }

    private static NetworkInterface findNetworkInterface() {
        List<Object> validNetworkInterfaces = Collections.emptyList();
        try {
            validNetworkInterfaces = NetUtils.getValidNetworkInterfaces();
        }
        catch (SocketException e) {
            logger.warn("ValidNetworkInterfaces exception", (Throwable)e);
        }
        NetworkInterface result = null;
        for (NetworkInterface networkInterface : validNetworkInterfaces) {
            if (!NetUtils.isSpecifyNetworkInterface(networkInterface)) continue;
            result = networkInterface;
            break;
        }
        if (null != result) {
            return result;
        }
        return NetUtils.findAddress(validNetworkInterfaces);
    }

    private static List<NetworkInterface> getValidNetworkInterfaces() throws SocketException {
        LinkedList<NetworkInterface> validNetworkInterfaces = new LinkedList<NetworkInterface>();
        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        while (interfaces.hasMoreElements()) {
            NetworkInterface networkInterface = interfaces.nextElement();
            if (NetUtils.ignoreNetworkInterface(networkInterface)) continue;
            validNetworkInterfaces.add(networkInterface);
        }
        return validNetworkInterfaces;
    }

    public static boolean ignoreNetworkInterface(NetworkInterface networkInterface) throws SocketException {
        return networkInterface == null || networkInterface.isLoopback() || networkInterface.isVirtual() || !networkInterface.isUp();
    }

    private static boolean isSpecifyNetworkInterface(NetworkInterface networkInterface) {
        String preferredNetworkInterface = PropertyUtils.getString("dolphin.scheduler.network.interface.preferred", System.getProperty("dolphin.scheduler.network.interface.preferred"));
        return Objects.equals(networkInterface.getDisplayName(), preferredNetworkInterface);
    }

    private static NetworkInterface findAddress(List<NetworkInterface> validNetworkInterfaces) {
        if (validNetworkInterfaces.isEmpty()) {
            return null;
        }
        String networkPriority = PropertyUtils.getString("dolphin.scheduler.network.priority.strategy", NETWORK_PRIORITY_DEFAULT);
        if (NETWORK_PRIORITY_DEFAULT.equalsIgnoreCase(networkPriority)) {
            return NetUtils.findAddressByDefaultPolicy(validNetworkInterfaces);
        }
        if (NETWORK_PRIORITY_INNER.equalsIgnoreCase(networkPriority)) {
            return NetUtils.findInnerAddress(validNetworkInterfaces);
        }
        if (NETWORK_PRIORITY_OUTER.equalsIgnoreCase(networkPriority)) {
            return NetUtils.findOuterAddress(validNetworkInterfaces);
        }
        logger.error("There is no matching network card acquisition policy!");
        return null;
    }

    private static NetworkInterface findAddressByDefaultPolicy(List<NetworkInterface> validNetworkInterfaces) {
        NetworkInterface networkInterface = NetUtils.findInnerAddress(validNetworkInterfaces);
        if (networkInterface == null && (networkInterface = NetUtils.findOuterAddress(validNetworkInterfaces)) == null) {
            networkInterface = validNetworkInterfaces.get(0);
        }
        return networkInterface;
    }

    private static NetworkInterface findInnerAddress(List<NetworkInterface> validNetworkInterfaces) {
        NetworkInterface networkInterface = null;
        for (NetworkInterface ni : validNetworkInterfaces) {
            Enumeration<InetAddress> address = ni.getInetAddresses();
            while (address.hasMoreElements()) {
                InetAddress ip = address.nextElement();
                if (!ip.isSiteLocalAddress() || ip.isLoopbackAddress()) continue;
                networkInterface = ni;
            }
        }
        return networkInterface;
    }

    private static NetworkInterface findOuterAddress(List<NetworkInterface> validNetworkInterfaces) {
        NetworkInterface networkInterface = null;
        for (NetworkInterface ni : validNetworkInterfaces) {
            Enumeration<InetAddress> address = ni.getInetAddresses();
            while (address.hasMoreElements()) {
                InetAddress ip = address.nextElement();
                if (ip.isSiteLocalAddress() || ip.isLoopbackAddress()) continue;
                networkInterface = ni;
            }
        }
        return networkInterface;
    }
}

