/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dolphinscheduler.server.master.registry;

import com.google.common.collect.Sets;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.dolphinscheduler.common.IStoppable;
import org.apache.dolphinscheduler.common.enums.NodeType;
import org.apache.dolphinscheduler.common.enums.StateEvent;
import org.apache.dolphinscheduler.common.enums.StateEventType;
import org.apache.dolphinscheduler.common.model.Server;
import org.apache.dolphinscheduler.common.thread.ThreadUtils;
import org.apache.dolphinscheduler.common.utils.NetUtils;
import org.apache.dolphinscheduler.dao.entity.ProcessInstance;
import org.apache.dolphinscheduler.dao.entity.TaskInstance;
import org.apache.dolphinscheduler.plugin.task.api.TaskExecutionContext;
import org.apache.dolphinscheduler.plugin.task.api.enums.ExecutionStatus;
import org.apache.dolphinscheduler.registry.api.ConnectionState;
import org.apache.dolphinscheduler.registry.api.SubscribeListener;
import org.apache.dolphinscheduler.remote.utils.NamedThreadFactory;
import org.apache.dolphinscheduler.server.builder.TaskExecutionContextBuilder;
import org.apache.dolphinscheduler.server.master.config.MasterConfig;
import org.apache.dolphinscheduler.server.master.registry.MasterRegistryDataListener;
import org.apache.dolphinscheduler.server.master.runner.WorkflowExecuteThreadPool;
import org.apache.dolphinscheduler.server.registry.HeartBeatTask;
import org.apache.dolphinscheduler.server.utils.ProcessUtils;
import org.apache.dolphinscheduler.service.process.ProcessService;
import org.apache.dolphinscheduler.service.registry.RegistryClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class MasterRegistryClient {
    private static final Logger logger = LoggerFactory.getLogger(MasterRegistryClient.class);
    @Autowired
    private ProcessService processService;
    @Autowired
    private RegistryClient registryClient;
    @Autowired
    private MasterConfig masterConfig;
    private ScheduledExecutorService heartBeatExecutor;
    @Autowired
    private WorkflowExecuteThreadPool workflowExecuteThreadPool;
    private long startupTime;
    private String localNodePath;

    public void init() {
        this.startupTime = System.currentTimeMillis();
        this.heartBeatExecutor = Executors.newSingleThreadScheduledExecutor((ThreadFactory)new NamedThreadFactory("HeartBeatExecutor"));
    }

    public void start() {
        try {
            this.registry();
            this.registryClient.subscribe("/nodes", (SubscribeListener)new MasterRegistryDataListener());
        }
        catch (Exception e) {
            logger.error("master start up exception", (Throwable)e);
            throw new RuntimeException("master start up error", e);
        }
    }

    public void setRegistryStoppable(IStoppable stoppable) {
        this.registryClient.setStoppable(stoppable);
    }

    public void closeRegistry() {
        this.deregister();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeMasterNodePath(String path, NodeType nodeType, boolean failover) {
        logger.info("{} node deleted : {}", (Object)nodeType, (Object)path);
        if (StringUtils.isEmpty((String)path)) {
            logger.error("server down error: empty path: {}, nodeType:{}", (Object)path, (Object)nodeType);
            return;
        }
        String serverHost = this.registryClient.getHostByEventDataPath(path);
        if (StringUtils.isEmpty((String)serverHost)) {
            logger.error("server down error: unknown path: {}, nodeType:{}", (Object)path, (Object)nodeType);
            return;
        }
        String failoverPath = this.getFailoverLockPath(nodeType, serverHost);
        try {
            this.registryClient.getLock(failoverPath);
            if (!this.registryClient.exists(path)) {
                logger.info("path: {} not exists", (Object)path);
                this.registryClient.handleDeadServer(Collections.singleton(path), nodeType, "add");
            }
            if (failover) {
                this.failoverServerWhenDown(serverHost, nodeType);
            }
        }
        catch (Exception e) {
            logger.error("{} server failover failed, host:{}", new Object[]{nodeType, serverHost, e});
        }
        finally {
            this.registryClient.releaseLock(failoverPath);
        }
    }

    public void removeWorkerNodePath(String path, NodeType nodeType, boolean failover) {
        logger.info("{} node deleted : {}", (Object)nodeType, (Object)path);
        try {
            String serverHost = null;
            if (!StringUtils.isEmpty((String)path)) {
                serverHost = this.registryClient.getHostByEventDataPath(path);
                if (StringUtils.isEmpty((String)serverHost)) {
                    logger.error("server down error: unknown path: {}", (Object)path);
                    return;
                }
                if (!this.registryClient.exists(path)) {
                    logger.info("path: {} not exists", (Object)path);
                    this.registryClient.handleDeadServer(Collections.singleton(path), nodeType, "add");
                }
            }
            if (failover) {
                this.failoverServerWhenDown(serverHost, nodeType);
            }
        }
        catch (Exception e) {
            logger.error("{} server failover failed", (Object)nodeType, (Object)e);
        }
    }

    private boolean isNeedToHandleDeadServer(String host, NodeType nodeType, Duration sessionTimeout) {
        long sessionTimeoutMillis = Math.max(10000L, sessionTimeout.toMillis());
        List serverList = this.registryClient.getServerList(nodeType);
        if (CollectionUtils.isEmpty((Collection)serverList)) {
            return true;
        }
        Date startupTime = this.getServerStartupTime(serverList, host);
        if (startupTime == null) {
            return true;
        }
        return System.currentTimeMillis() - startupTime.getTime() > sessionTimeoutMillis;
    }

    private void failoverServerWhenDown(String serverHost, NodeType nodeType) {
        switch (nodeType) {
            case MASTER: {
                this.failoverMaster(serverHost);
                break;
            }
            case WORKER: {
                this.failoverWorker(serverHost);
                break;
            }
        }
    }

    public String getFailoverLockPath(NodeType nodeType, String host) {
        switch (nodeType) {
            case MASTER: {
                return "/lock/failover/masters/" + host;
            }
            case WORKER: {
                return "/lock/failover/workers/" + host;
            }
        }
        return "";
    }

    private boolean checkTaskInstanceNeedFailover(List<Server> workerServers, TaskInstance taskInstance) {
        boolean taskNeedFailover = true;
        if (taskInstance.getHost() == null) {
            return false;
        }
        if (this.checkTaskAfterWorkerStart(workerServers, taskInstance)) {
            taskNeedFailover = false;
        }
        return taskNeedFailover;
    }

    private boolean checkTaskAfterWorkerStart(List<Server> workerServers, TaskInstance taskInstance) {
        if (StringUtils.isEmpty((String)taskInstance.getHost())) {
            return false;
        }
        Date workerServerStartDate = this.getServerStartupTime(workerServers, taskInstance.getHost());
        if (workerServerStartDate != null) {
            if (taskInstance.getStartTime() == null) {
                return taskInstance.getSubmitTime().after(workerServerStartDate);
            }
            return taskInstance.getStartTime().after(workerServerStartDate);
        }
        return false;
    }

    private Date getServerStartupTime(List<Server> servers, String host) {
        if (CollectionUtils.isEmpty(servers)) {
            return null;
        }
        Date serverStartupTime = null;
        for (Server server : servers) {
            if (!host.equals(server.getHost() + ":" + server.getPort())) continue;
            serverStartupTime = server.getCreateTime();
            break;
        }
        return serverStartupTime;
    }

    private Date getServerStartupTime(NodeType nodeType, String host) {
        if (StringUtils.isEmpty((String)host)) {
            return null;
        }
        List servers = this.registryClient.getServerList(nodeType);
        return this.getServerStartupTime(servers, host);
    }

    private void failoverWorker(String workerHost) {
        if (StringUtils.isEmpty((String)workerHost)) {
            return;
        }
        List workerServers = this.registryClient.getServerList(NodeType.WORKER);
        long startTime = System.currentTimeMillis();
        List needFailoverTaskInstanceList = this.processService.queryNeedFailoverTaskInstances(workerHost);
        HashMap<Integer, ProcessInstance> processInstanceCacheMap = new HashMap<Integer, ProcessInstance>();
        logger.info("start worker[{}] failover, task list size:{}", (Object)workerHost, (Object)needFailoverTaskInstanceList.size());
        for (TaskInstance taskInstance : needFailoverTaskInstanceList) {
            ProcessInstance processInstance = (ProcessInstance)processInstanceCacheMap.get(taskInstance.getProcessInstanceId());
            if (processInstance == null) {
                processInstance = this.processService.findProcessInstanceDetailById(taskInstance.getProcessInstanceId());
                if (processInstance == null) {
                    logger.error("failover task instance error, processInstance {} of taskInstance {} is null", (Object)taskInstance.getProcessInstanceId(), (Object)taskInstance.getId());
                    continue;
                }
                processInstanceCacheMap.put(processInstance.getId(), processInstance);
            }
            if (!this.checkTaskInstanceNeedFailover(workerServers, taskInstance) || !processInstance.getHost().equalsIgnoreCase(this.getLocalAddress())) continue;
            logger.info("failover task instance id: {}, process instance id: {}", (Object)taskInstance.getId(), (Object)taskInstance.getProcessInstanceId());
            this.failoverTaskInstance(processInstance, taskInstance);
        }
        logger.info("end worker[{}] failover, useTime:{}ms", (Object)workerHost, (Object)(System.currentTimeMillis() - startTime));
    }

    public void failoverMaster(String masterHost) {
        if (StringUtils.isEmpty((String)masterHost)) {
            return;
        }
        Date serverStartupTime = this.getServerStartupTime(NodeType.MASTER, masterHost);
        List workerServers = this.registryClient.getServerList(NodeType.WORKER);
        long startTime = System.currentTimeMillis();
        List needFailoverProcessInstanceList = this.processService.queryNeedFailoverProcessInstances(masterHost);
        logger.info("start master[{}] failover, process list size:{}", (Object)masterHost, (Object)needFailoverProcessInstanceList.size());
        for (ProcessInstance processInstance : needFailoverProcessInstanceList) {
            if ("NULL".equals(processInstance.getHost())) continue;
            List validTaskInstanceList = this.processService.findValidTaskListByProcessId(Integer.valueOf(processInstance.getId()));
            for (TaskInstance taskInstance : validTaskInstanceList) {
                if ("NULL".equals(taskInstance.getHost()) || taskInstance.getState().typeIsFinished() || !this.checkTaskInstanceNeedFailover(workerServers, taskInstance)) continue;
                logger.info("failover task instance id: {}, process instance id: {}", (Object)taskInstance.getId(), (Object)taskInstance.getProcessInstanceId());
                this.failoverTaskInstance(processInstance, taskInstance);
            }
            if (serverStartupTime != null && processInstance.getRestartTime() != null && processInstance.getRestartTime().after(serverStartupTime)) continue;
            logger.info("failover process instance id: {}", (Object)processInstance.getId());
            this.processService.processNeedFailoverProcessInstances(processInstance);
        }
        logger.info("master[{}] failover end, useTime:{}ms", (Object)masterHost, (Object)(System.currentTimeMillis() - startTime));
    }

    private void failoverTaskInstance(ProcessInstance processInstance, TaskInstance taskInstance) {
        if (taskInstance == null) {
            logger.error("failover task instance error, taskInstance is null");
            return;
        }
        if (processInstance == null) {
            logger.error("failover task instance error, processInstance {} of taskInstance {} is null", (Object)taskInstance.getProcessInstanceId(), (Object)taskInstance.getId());
            return;
        }
        taskInstance.setProcessInstance(processInstance);
        TaskExecutionContext taskExecutionContext = TaskExecutionContextBuilder.get().buildTaskInstanceRelatedInfo(taskInstance).buildProcessInstanceRelatedInfo(processInstance).create();
        if (this.masterConfig.isKillYarnJobWhenTaskFailover()) {
            ProcessUtils.killYarnJob((TaskExecutionContext)taskExecutionContext);
        }
        taskInstance.setState(ExecutionStatus.NEED_FAULT_TOLERANCE);
        this.processService.saveTaskInstance(taskInstance);
        StateEvent stateEvent = new StateEvent();
        stateEvent.setTaskInstanceId(taskInstance.getId());
        stateEvent.setType(StateEventType.TASK_STATE_CHANGE);
        stateEvent.setProcessInstanceId(processInstance.getId());
        stateEvent.setExecutionStatus(taskInstance.getState());
        this.workflowExecuteThreadPool.submitStateEvent(stateEvent);
    }

    public void registry() {
        String address = NetUtils.getAddr((int)this.masterConfig.getListenPort());
        this.localNodePath = this.getMasterPath();
        int masterHeartbeatInterval = this.masterConfig.getHeartbeatInterval();
        HeartBeatTask heartBeatTask = new HeartBeatTask(this.startupTime, this.masterConfig.getMaxCpuLoadAvg(), this.masterConfig.getReservedMemory(), (Set)Sets.newHashSet((Object[])new String[]{this.getMasterPath()}), "master", this.registryClient);
        this.registryClient.remove(this.localNodePath);
        this.registryClient.persistEphemeral(this.localNodePath, heartBeatTask.getHeartBeatInfo());
        while (!this.registryClient.checkNodeExists(NetUtils.getHost(), NodeType.MASTER)) {
            ThreadUtils.sleep((long)1000L);
        }
        ThreadUtils.sleep((long)1000L);
        this.registryClient.handleDeadServer(Collections.singleton(this.localNodePath), NodeType.MASTER, "delete");
        this.registryClient.addConnectionStateListener(this::handleConnectionState);
        this.heartBeatExecutor.scheduleAtFixedRate((Runnable)heartBeatTask, masterHeartbeatInterval, masterHeartbeatInterval, TimeUnit.SECONDS);
        logger.info("master node : {} registry to ZK successfully with heartBeatInterval : {}s", (Object)address, (Object)masterHeartbeatInterval);
    }

    public void handleConnectionState(ConnectionState state) {
        switch (state) {
            case CONNECTED: {
                logger.debug("registry connection state is {}", (Object)state);
                break;
            }
            case SUSPENDED: {
                logger.warn("registry connection state is {}, ready to retry connection", (Object)state);
                break;
            }
            case RECONNECTED: {
                logger.debug("registry connection state is {}, clean the node info", (Object)state);
                this.registryClient.persistEphemeral(this.localNodePath, "");
                break;
            }
            case DISCONNECTED: {
                logger.warn("registry connection state is {}, ready to stop myself", (Object)state);
                this.registryClient.getStoppable().stop("registry connection state is DISCONNECTED, stop myself");
                break;
            }
        }
    }

    public void deregister() {
        try {
            String address = this.getLocalAddress();
            String localNodePath = this.getMasterPath();
            this.registryClient.remove(localNodePath);
            logger.info("master node : {} unRegistry to register center.", (Object)address);
            this.heartBeatExecutor.shutdown();
            logger.info("heartbeat executor shutdown");
            this.registryClient.close();
        }
        catch (Exception e) {
            logger.error("remove registry path exception ", (Throwable)e);
        }
    }

    public String getMasterPath() {
        String address = this.getLocalAddress();
        return "/nodes/master/" + address;
    }

    public String getLocalAddress() {
        return NetUtils.getAddr((int)this.masterConfig.getListenPort());
    }
}

