/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dolphinscheduler.service.registry;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import org.apache.dolphinscheduler.common.IStoppable;
import org.apache.dolphinscheduler.common.enums.NodeType;
import org.apache.dolphinscheduler.common.model.Server;
import org.apache.dolphinscheduler.common.utils.HeartBeat;
import org.apache.dolphinscheduler.common.utils.JSONUtils;
import org.apache.dolphinscheduler.common.utils.PropertyUtils;
import org.apache.dolphinscheduler.registry.api.ConnectionListener;
import org.apache.dolphinscheduler.registry.api.Registry;
import org.apache.dolphinscheduler.registry.api.RegistryException;
import org.apache.dolphinscheduler.registry.api.RegistryFactory;
import org.apache.dolphinscheduler.registry.api.RegistryFactoryLoader;
import org.apache.dolphinscheduler.registry.api.SubscribeListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class RegistryClient {
    private static final Logger logger = LoggerFactory.getLogger(RegistryClient.class);
    private static final String EMPTY = "";
    private static final String REGISTRY_PREFIX = "registry";
    private static final String REGISTRY_PLUGIN_NAME = "plugin.name";
    private static final String REGISTRY_CONFIG_FILE_PATH = "/registry.properties";
    private final AtomicBoolean isStarted = new AtomicBoolean(false);
    private Registry registry;
    private IStoppable stoppable;

    @PostConstruct
    public void afterConstruct() {
        this.start();
        this.initNodes();
    }

    public int getActiveMasterNum() {
        Collection<Object> childrenList = new ArrayList();
        try {
            if (this.exists(this.rootNodePath(NodeType.MASTER))) {
                childrenList = this.getChildrenKeys(this.rootNodePath(NodeType.MASTER));
            }
        }
        catch (Exception e) {
            logger.error("getActiveMasterNum error", (Throwable)e);
        }
        return childrenList.size();
    }

    public List<Server> getServerList(NodeType nodeType) {
        Map<String, String> serverMaps = this.getServerMaps(nodeType, false);
        String parentPath = this.rootNodePath(nodeType);
        ArrayList<Server> serverList = new ArrayList<Server>();
        for (Map.Entry<String, String> entry : serverMaps.entrySet()) {
            HeartBeat heartBeat = HeartBeat.decodeHeartBeat((String)entry.getValue());
            if (heartBeat == null) continue;
            Server server = new Server();
            server.setResInfo(JSONUtils.toJsonString((Object)heartBeat));
            server.setCreateTime(new Date(heartBeat.getStartupTime()));
            server.setLastHeartbeatTime(new Date(heartBeat.getReportTime()));
            server.setId(heartBeat.getProcessId());
            String key = entry.getKey();
            server.setZkDirectory(parentPath + "/" + key);
            String[] hostAndPort = key.split(":");
            String[] hosts = hostAndPort[0].split("/");
            server.setHost(hosts[hosts.length - 1]);
            server.setPort(Integer.parseInt(hostAndPort[1]));
            serverList.add(server);
        }
        return serverList;
    }

    public Map<String, String> getServerMaps(NodeType nodeType, boolean hostOnly) {
        HashMap<String, String> serverMap = new HashMap<String, String>();
        try {
            String path = this.rootNodePath(nodeType);
            Collection<String> serverList = this.getServerNodes(nodeType);
            Iterator<String> iterator = serverList.iterator();
            while (iterator.hasNext()) {
                String server;
                String host = server = iterator.next();
                if (nodeType == NodeType.WORKER && hostOnly) {
                    host = server.split("/")[1];
                }
                serverMap.putIfAbsent(host, this.get(path + "/" + server));
            }
        }
        catch (Exception e) {
            logger.error("get server list failed", (Throwable)e);
        }
        return serverMap;
    }

    public boolean checkNodeExists(String host, NodeType nodeType) {
        return this.getServerMaps(nodeType, true).keySet().stream().anyMatch(it -> it.contains(host));
    }

    public void handleDeadServer(Collection<String> nodes, NodeType nodeType, String opType) {
        nodes.forEach(node -> {
            String type;
            String host = this.getHostByEventDataPath((String)node);
            String string = type = nodeType == NodeType.MASTER ? "master" : "worker";
            if (opType.equals("delete")) {
                this.removeDeadServerByHost(host, type);
            } else if (opType.equals("add")) {
                String deadServerPath = "/dead-servers/" + type + "_" + host;
                this.registry.put(deadServerPath, type + "_" + host, false);
                logger.info("{} server dead , and {} added to zk dead server path success", (Object)nodeType, node);
            }
        });
    }

    public boolean checkIsDeadServer(String node, String serverType) {
        String[] zNodesPath = node.split("/");
        String ipSeqNo = zNodesPath[zNodesPath.length - 1];
        String deadServerPath = "/dead-servers/" + serverType + "_" + ipSeqNo;
        return !this.exists(node) || this.exists(deadServerPath);
    }

    public Collection<String> getMasterNodesDirectly() {
        return this.getChildrenKeys("/nodes/master");
    }

    public Collection<String> getWorkerGroupDirectly() {
        return this.getChildrenKeys("/nodes/worker");
    }

    public Collection<String> getWorkerGroupNodesDirectly(String workerGroup) {
        return this.getChildrenKeys("/nodes/worker/" + workerGroup);
    }

    public String getHostByEventDataPath(String path) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)path) ? 1 : 0) != 0, (Object)"path cannot be null or empty");
        String[] pathArray = path.split("/");
        Preconditions.checkArgument((pathArray.length >= 1 ? 1 : 0) != 0, (String)"cannot parse path: %s", (Object)path);
        return pathArray[pathArray.length - 1];
    }

    public void close() throws IOException {
        if (this.isStarted.compareAndSet(true, false) && this.registry != null) {
            this.registry.close();
        }
    }

    public void persistEphemeral(String key, String value) {
        this.registry.put(key, value, true);
    }

    public void remove(String key) {
        this.registry.delete(key);
    }

    public String get(String key) {
        return this.registry.get(key);
    }

    public void subscribe(String path, SubscribeListener listener) {
        this.registry.subscribe(path, listener);
    }

    public void addConnectionStateListener(ConnectionListener listener) {
        this.registry.addConnectionStateListener(listener);
    }

    public boolean exists(String key) {
        return this.registry.exists(key);
    }

    public boolean getLock(String key) {
        return this.registry.acquireLock(key);
    }

    public boolean releaseLock(String key) {
        return this.registry.releaseLock(key);
    }

    public void setStoppable(IStoppable stoppable) {
        this.stoppable = stoppable;
    }

    public IStoppable getStoppable() {
        return this.stoppable;
    }

    public boolean isMasterPath(String path) {
        return path != null && path.startsWith("/nodes/master");
    }

    public boolean isWorkerPath(String path) {
        return path != null && path.startsWith("/nodes/worker");
    }

    public Collection<String> getChildrenKeys(String key) {
        return this.registry.children(key);
    }

    public Set<String> getServerNodeSet(NodeType nodeType, boolean hostOnly) {
        try {
            return this.getServerNodes(nodeType).stream().map(server -> {
                if (nodeType == NodeType.WORKER && hostOnly) {
                    return server.split("/")[1];
                }
                return server;
            }).collect(Collectors.toSet());
        }
        catch (Exception e) {
            throw new RegistryException("Failed to get server node: " + nodeType, (Throwable)e);
        }
    }

    private void start() {
        if (this.isStarted.compareAndSet(false, true)) {
            PropertyUtils.loadPropertyFile((String[])new String[]{REGISTRY_CONFIG_FILE_PATH});
            Map registryConfig = PropertyUtils.getPropertiesByPrefix((String)REGISTRY_PREFIX);
            if (null == registryConfig || registryConfig.isEmpty()) {
                throw new RegistryException("registry config param is null");
            }
            String pluginName = (String)registryConfig.get(REGISTRY_PLUGIN_NAME);
            Map factories = RegistryFactoryLoader.load();
            if (!factories.containsKey(pluginName)) {
                throw new RegistryException("No such registry plugin: " + pluginName);
            }
            this.registry = ((RegistryFactory)factories.get(pluginName)).create();
            this.registry.start(registryConfig);
        }
    }

    private void initNodes() {
        this.registry.put("/nodes/master", EMPTY, false);
        this.registry.put("/nodes/worker", EMPTY, false);
        this.registry.put("/dead-servers", EMPTY, false);
    }

    private String rootNodePath(NodeType type) {
        switch (type) {
            case MASTER: {
                return "/nodes/master";
            }
            case WORKER: {
                return "/nodes/worker";
            }
            case DEAD_SERVER: {
                return "/dead-servers";
            }
        }
        throw new IllegalStateException("Should not reach here");
    }

    private Collection<String> getServerNodes(NodeType nodeType) {
        String path = this.rootNodePath(nodeType);
        Collection<String> serverList = this.getChildrenKeys(path);
        if (nodeType != NodeType.WORKER) {
            return serverList;
        }
        return serverList.stream().flatMap(group -> this.getChildrenKeys(path + "/" + group).stream().map(it -> group + "/" + it)).collect(Collectors.toList());
    }

    private void removeDeadServerByHost(String host, String serverType) {
        Collection<String> deadServers = this.getChildrenKeys("/dead-servers");
        for (String serverPath : deadServers) {
            if (!serverPath.startsWith(serverType + "_" + host)) continue;
            String server = "/dead-servers/" + serverPath;
            this.remove(server);
            logger.info("{} server {} deleted from zk dead server path success", (Object)serverType, (Object)host);
        }
    }
}

