/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dolphinscheduler.service.registry;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import lombok.NonNull;
import org.apache.commons.lang3.StringUtils;
import org.apache.dolphinscheduler.common.IStoppable;
import org.apache.dolphinscheduler.common.enums.NodeType;
import org.apache.dolphinscheduler.common.model.MasterHeartBeat;
import org.apache.dolphinscheduler.common.model.Server;
import org.apache.dolphinscheduler.common.model.WorkerHeartBeat;
import org.apache.dolphinscheduler.common.utils.JSONUtils;
import org.apache.dolphinscheduler.registry.api.ConnectionListener;
import org.apache.dolphinscheduler.registry.api.Registry;
import org.apache.dolphinscheduler.registry.api.RegistryException;
import org.apache.dolphinscheduler.registry.api.SubscribeListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class RegistryClient {
    private static final Logger logger = LoggerFactory.getLogger(RegistryClient.class);
    private static final String EMPTY = "";
    private IStoppable stoppable;
    private final Registry registry;

    public RegistryClient(Registry registry) {
        this.registry = registry;
    }

    @PostConstruct
    public void afterConstruct() {
        this.initNodes();
    }

    public void connectUntilTimeout(@NonNull Duration duration) throws RegistryException {
        if (duration == null) {
            throw new NullPointerException("duration is marked non-null but is null");
        }
        this.registry.connectUntilTimeout(duration);
    }

    public int getActiveMasterNum() {
        Collection<Object> childrenList = new ArrayList();
        try {
            if (this.exists(this.rootNodePath(NodeType.MASTER))) {
                childrenList = this.getChildrenKeys(this.rootNodePath(NodeType.MASTER));
            }
        }
        catch (Exception e) {
            logger.error("getActiveMasterNum error", (Throwable)e);
        }
        return childrenList.size();
    }

    public List<Server> getServerList(NodeType nodeType) {
        Map<String, String> serverMaps = this.getServerMaps(nodeType, false);
        String parentPath = this.rootNodePath(nodeType);
        ArrayList<Server> serverList = new ArrayList<Server>();
        for (Map.Entry<String, String> entry : serverMaps.entrySet()) {
            String serverPath = entry.getKey();
            String heartBeatJson = entry.getValue();
            if (StringUtils.isEmpty((CharSequence)heartBeatJson)) {
                logger.error("The heartBeatJson is empty, serverPath: {}", (Object)serverPath);
                continue;
            }
            Server server = new Server();
            switch (nodeType) {
                case MASTER: {
                    MasterHeartBeat masterHeartBeat = (MasterHeartBeat)JSONUtils.parseObject((String)heartBeatJson, MasterHeartBeat.class);
                    server.setCreateTime(new Date(masterHeartBeat.getStartupTime()));
                    server.setLastHeartbeatTime(new Date(masterHeartBeat.getReportTime()));
                    server.setId(masterHeartBeat.getProcessId());
                    break;
                }
                case WORKER: {
                    WorkerHeartBeat workerHeartBeat = (WorkerHeartBeat)JSONUtils.parseObject((String)heartBeatJson, WorkerHeartBeat.class);
                    server.setCreateTime(new Date(workerHeartBeat.getStartupTime()));
                    server.setLastHeartbeatTime(new Date(workerHeartBeat.getReportTime()));
                    server.setId(workerHeartBeat.getProcessId());
                }
            }
            server.setResInfo(heartBeatJson);
            server.setZkDirectory(parentPath + "/" + serverPath);
            String[] hostAndPort = serverPath.split(":");
            String[] hosts = hostAndPort[0].split("/");
            server.setHost(hosts[hosts.length - 1]);
            server.setPort(Integer.parseInt(hostAndPort[1]));
            serverList.add(server);
        }
        return serverList;
    }

    public Map<String, String> getServerMaps(NodeType nodeType, boolean hostOnly) {
        HashMap<String, String> serverMap = new HashMap<String, String>();
        try {
            String path = this.rootNodePath(nodeType);
            Collection<String> serverList = this.getServerNodes(nodeType);
            Iterator<String> iterator = serverList.iterator();
            while (iterator.hasNext()) {
                String server;
                String host = server = iterator.next();
                if (nodeType == NodeType.WORKER && hostOnly) {
                    host = server.split("/")[1];
                }
                serverMap.putIfAbsent(host, this.get(path + "/" + server));
            }
        }
        catch (Exception e) {
            logger.error("get server list failed", (Throwable)e);
        }
        return serverMap;
    }

    public boolean checkNodeExists(String host, NodeType nodeType) {
        return this.getServerMaps(nodeType, true).keySet().stream().anyMatch(it -> it.contains(host));
    }

    public Collection<String> getMasterNodesDirectly() {
        return this.getChildrenKeys("/nodes/master");
    }

    public Collection<String> getWorkerGroupDirectly() {
        return this.getChildrenKeys("/nodes/worker");
    }

    public Collection<String> getWorkerGroupNodesDirectly(String workerGroup) {
        return this.getChildrenKeys("/nodes/worker/" + workerGroup);
    }

    public String getHostByEventDataPath(String path) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)path) ? 1 : 0) != 0, (Object)"path cannot be null or empty");
        String[] pathArray = path.split("/");
        Preconditions.checkArgument((pathArray.length >= 1 ? 1 : 0) != 0, (String)"cannot parse path: %s", (Object)path);
        return pathArray[pathArray.length - 1];
    }

    public void close() throws IOException {
        this.registry.close();
    }

    public void persistEphemeral(String key, String value) {
        this.registry.put(key, value, true);
    }

    public void remove(String key) {
        this.registry.delete(key);
    }

    public String get(String key) {
        return this.registry.get(key);
    }

    public void subscribe(String path, SubscribeListener listener) {
        this.registry.subscribe(path, listener);
    }

    public void addConnectionStateListener(ConnectionListener listener) {
        this.registry.addConnectionStateListener(listener);
    }

    public boolean exists(String key) {
        return this.registry.exists(key);
    }

    public boolean getLock(String key) {
        return this.registry.acquireLock(key);
    }

    public boolean releaseLock(String key) {
        return this.registry.releaseLock(key);
    }

    public void setStoppable(IStoppable stoppable) {
        this.stoppable = stoppable;
    }

    public IStoppable getStoppable() {
        return this.stoppable;
    }

    public boolean isMasterPath(String path) {
        return path != null && path.startsWith("/nodes/master");
    }

    public boolean isWorkerPath(String path) {
        return path != null && path.startsWith("/nodes/worker");
    }

    public Collection<String> getChildrenKeys(String key) {
        return this.registry.children(key);
    }

    public Set<String> getServerNodeSet(NodeType nodeType, boolean hostOnly) {
        try {
            return this.getServerNodes(nodeType).stream().map(server -> {
                if (nodeType == NodeType.WORKER && hostOnly) {
                    return server.split("/")[1];
                }
                return server;
            }).collect(Collectors.toSet());
        }
        catch (Exception e) {
            throw new RegistryException("Failed to get server node: " + nodeType, (Throwable)e);
        }
    }

    private void initNodes() {
        this.registry.put("/nodes/master", EMPTY, false);
        this.registry.put("/nodes/worker", EMPTY, false);
    }

    private String rootNodePath(NodeType type) {
        switch (type) {
            case MASTER: {
                return "/nodes/master";
            }
            case WORKER: {
                return "/nodes/worker";
            }
        }
        throw new IllegalStateException("Should not reach here");
    }

    private Collection<String> getServerNodes(NodeType nodeType) {
        String path = this.rootNodePath(nodeType);
        Collection<String> serverList = this.getChildrenKeys(path);
        if (nodeType != NodeType.WORKER) {
            return serverList;
        }
        return serverList.stream().flatMap(group -> this.getChildrenKeys(path + "/" + group).stream().map(it -> group + "/" + it)).collect(Collectors.toList());
    }
}

