/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Ints;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableDescriptors;
import org.apache.hadoop.hbase.TableInfoMissingException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;

@InterfaceAudience.Private
public class FSTableDescriptors
implements TableDescriptors {
    private static final Log LOG = LogFactory.getLog(FSTableDescriptors.class);
    private final FileSystem fs;
    private final Path rootdir;
    private final boolean fsreadonly;
    @VisibleForTesting
    long cachehits = 0L;
    @VisibleForTesting
    long invocations = 0L;
    static final String TABLEINFO_FILE_PREFIX = ".tableinfo";
    static final String TABLEINFO_DIR = ".tabledesc";
    static final String TMP_DIR = ".tmp";
    private final Map<TableName, TableDescriptorAndModtime> cache = new ConcurrentHashMap<TableName, TableDescriptorAndModtime>();
    @VisibleForTesting
    static final Comparator<FileStatus> TABLEINFO_FILESTATUS_COMPARATOR = new Comparator<FileStatus>(){

        @Override
        public int compare(FileStatus left, FileStatus right) {
            return -left.compareTo((Object)right);
        }
    };
    private static final PathFilter TABLEINFO_PATHFILTER = new PathFilter(){

        public boolean accept(Path p) {
            return p.getName().startsWith(FSTableDescriptors.TABLEINFO_FILE_PREFIX);
        }
    };
    @VisibleForTesting
    static final int WIDTH_OF_SEQUENCE_ID = 10;
    private static final Pattern TABLEINFO_FILE_REGEX = Pattern.compile(".tableinfo(\\.([0-9]{10}))?$");

    public FSTableDescriptors(Configuration conf) throws IOException {
        this(FSUtils.getCurrentFileSystem(conf), FSUtils.getRootDir(conf));
    }

    public FSTableDescriptors(FileSystem fs, Path rootdir) {
        this(fs, rootdir, false);
    }

    public FSTableDescriptors(FileSystem fs, Path rootdir, boolean fsreadonly) {
        this.fs = fs;
        this.rootdir = rootdir;
        this.fsreadonly = fsreadonly;
    }

    @Override
    public HTableDescriptor get(TableName tablename) throws IOException {
        ++this.invocations;
        if (HTableDescriptor.META_TABLEDESC.getTableName().equals((Object)tablename)) {
            ++this.cachehits;
            return HTableDescriptor.META_TABLEDESC;
        }
        if (HConstants.HBASE_NON_USER_TABLE_DIRS.contains(tablename.getNameAsString())) {
            throw new IOException("No descriptor found for non table = " + tablename);
        }
        TableDescriptorAndModtime cachedtdm = this.cache.get(tablename);
        if (cachedtdm != null && this.getTableInfoModtime(tablename) <= cachedtdm.getModtime()) {
            ++this.cachehits;
            return cachedtdm.getTableDescriptor();
        }
        TableDescriptorAndModtime tdmt = null;
        try {
            tdmt = this.getTableDescriptorAndModtime(tablename);
        }
        catch (NullPointerException e) {
            LOG.debug((Object)("Exception during readTableDecriptor. Current table name = " + tablename), (Throwable)e);
        }
        catch (IOException ioe) {
            LOG.debug((Object)("Exception during readTableDecriptor. Current table name = " + tablename), (Throwable)ioe);
        }
        if (tdmt != null) {
            this.cache.put(tablename, tdmt);
        }
        return tdmt == null ? null : tdmt.getTableDescriptor();
    }

    @Override
    public Map<String, HTableDescriptor> getAll() throws IOException {
        TreeMap<String, HTableDescriptor> htds = new TreeMap<String, HTableDescriptor>();
        List<Path> tableDirs = FSUtils.getTableDirs(this.fs, this.rootdir);
        for (Path d : tableDirs) {
            HTableDescriptor htd = null;
            try {
                htd = this.get(FSUtils.getTableName(d));
            }
            catch (FileNotFoundException fnfe) {
                LOG.warn((Object)"Trouble retrieving htd", (Throwable)fnfe);
            }
            if (htd == null) continue;
            htds.put(htd.getTableName().getNameAsString(), htd);
        }
        return htds;
    }

    @Override
    public Map<String, HTableDescriptor> getByNamespace(String name) throws IOException {
        TreeMap<String, HTableDescriptor> htds = new TreeMap<String, HTableDescriptor>();
        List<Path> tableDirs = FSUtils.getLocalTableDirs(this.fs, FSUtils.getNamespaceDir(this.rootdir, name));
        for (Path d : tableDirs) {
            HTableDescriptor htd = null;
            try {
                htd = this.get(FSUtils.getTableName(d));
            }
            catch (FileNotFoundException fnfe) {
                LOG.warn((Object)"Trouble retrieving htd", (Throwable)fnfe);
            }
            if (htd == null) continue;
            htds.put(FSUtils.getTableName(d).getNameAsString(), htd);
        }
        return htds;
    }

    @Override
    public void add(HTableDescriptor htd) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot add a table descriptor - in read only mode");
        }
        if (TableName.META_TABLE_NAME.equals((Object)htd.getTableName())) {
            throw new NotImplementedException();
        }
        if (HConstants.HBASE_NON_USER_TABLE_DIRS.contains(htd.getTableName().getNameAsString())) {
            throw new NotImplementedException("Cannot add a table descriptor for a reserved subdirectory name: " + htd.getNameAsString());
        }
        this.updateTableDescriptor(htd);
        long modtime = this.getTableInfoModtime(htd.getTableName());
        this.cache.put(htd.getTableName(), new TableDescriptorAndModtime(modtime, htd));
    }

    @Override
    public HTableDescriptor remove(TableName tablename) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot remove a table descriptor - in read only mode");
        }
        Path tabledir = this.getTableDir(tablename);
        if (this.fs.exists(tabledir) && !this.fs.delete(tabledir, true)) {
            throw new IOException("Failed delete of " + tabledir.toString());
        }
        TableDescriptorAndModtime tdm = this.cache.remove(tablename);
        return tdm == null ? null : tdm.getTableDescriptor();
    }

    public boolean isTableInfoExists(TableName tableName) throws IOException {
        return this.getTableInfoPath(tableName) != null;
    }

    private FileStatus getTableInfoPath(TableName tableName) throws IOException {
        Path tableDir = this.getTableDir(tableName);
        return this.getTableInfoPath(tableDir);
    }

    private FileStatus getTableInfoPath(Path tableDir) throws IOException {
        return FSTableDescriptors.getTableInfoPath(this.fs, tableDir, !this.fsreadonly);
    }

    public static FileStatus getTableInfoPath(FileSystem fs, Path tableDir) throws IOException {
        return FSTableDescriptors.getTableInfoPath(fs, tableDir, false);
    }

    private static FileStatus getTableInfoPath(FileSystem fs, Path tableDir, boolean removeOldFiles) throws IOException {
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        return FSTableDescriptors.getCurrentTableInfoStatus(fs, tableInfoDir, removeOldFiles);
    }

    static FileStatus getCurrentTableInfoStatus(FileSystem fs, Path dir, boolean removeOldFiles) throws IOException {
        FileStatus[] status = FSUtils.listStatus(fs, dir, TABLEINFO_PATHFILTER);
        if (status == null || status.length < 1) {
            return null;
        }
        FileStatus mostCurrent = null;
        for (FileStatus file : status) {
            if (mostCurrent != null && TABLEINFO_FILESTATUS_COMPARATOR.compare(file, mostCurrent) >= 0) continue;
            mostCurrent = file;
        }
        if (removeOldFiles && status.length > 1) {
            for (FileStatus file : status) {
                Path path = file.getPath();
                if (file == mostCurrent) continue;
                if (!fs.delete(file.getPath(), false)) {
                    LOG.warn((Object)("Failed cleanup of " + path));
                    continue;
                }
                LOG.debug((Object)("Cleaned up old tableinfo file " + path));
            }
        }
        return mostCurrent;
    }

    @VisibleForTesting
    Path getTableDir(TableName tableName) {
        return FSUtils.getTableDir(this.rootdir, tableName);
    }

    private static String formatTableInfoSequenceId(int number) {
        byte[] b = new byte[10];
        int d = Math.abs(number);
        for (int i = b.length - 1; i >= 0; --i) {
            b[i] = (byte)(d % 10 + 48);
            d /= 10;
        }
        return Bytes.toString((byte[])b);
    }

    @VisibleForTesting
    static int getTableInfoSequenceId(Path p) {
        if (p == null) {
            return 0;
        }
        Matcher m = TABLEINFO_FILE_REGEX.matcher(p.getName());
        if (!m.matches()) {
            throw new IllegalArgumentException(p.toString());
        }
        String suffix = m.group(2);
        if (suffix == null || suffix.length() <= 0) {
            return 0;
        }
        return Integer.parseInt(m.group(2));
    }

    @VisibleForTesting
    static String getTableInfoFileName(int sequenceid) {
        return ".tableinfo." + FSTableDescriptors.formatTableInfoSequenceId(sequenceid);
    }

    private long getTableInfoModtime(TableName tableName) throws IOException {
        FileStatus status = this.getTableInfoPath(tableName);
        return status == null ? 0L : status.getModificationTime();
    }

    public static HTableDescriptor getTableDescriptorFromFs(FileSystem fs, Path hbaseRootDir, TableName tableName) throws IOException {
        Path tableDir = FSUtils.getTableDir(hbaseRootDir, tableName);
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir);
    }

    public static HTableDescriptor getTableDescriptorFromFs(FileSystem fs, Path tableDir) throws IOException {
        FileStatus status = FSTableDescriptors.getTableInfoPath(fs, tableDir, false);
        if (status == null) {
            throw new TableInfoMissingException("No table descriptor file under " + tableDir);
        }
        return FSTableDescriptors.readTableDescriptor(fs, status, false);
    }

    private TableDescriptorAndModtime getTableDescriptorAndModtime(TableName tableName) throws IOException {
        if (tableName.equals((Object)TableName.META_TABLE_NAME)) {
            return null;
        }
        return this.getTableDescriptorAndModtime(this.getTableDir(tableName));
    }

    private TableDescriptorAndModtime getTableDescriptorAndModtime(Path tableDir) throws IOException {
        FileStatus status = this.getTableInfoPath(tableDir);
        if (status == null) {
            return null;
        }
        HTableDescriptor htd = FSTableDescriptors.readTableDescriptor(this.fs, status, !this.fsreadonly);
        return new TableDescriptorAndModtime(status.getModificationTime(), htd);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static HTableDescriptor readTableDescriptor(FileSystem fs, FileStatus status, boolean rewritePb) throws IOException {
        int len = Ints.checkedCast((long)status.getLen());
        byte[] content = new byte[len];
        FSDataInputStream fsDataInputStream = fs.open(status.getPath());
        try {
            fsDataInputStream.readFully(content);
        }
        finally {
            fsDataInputStream.close();
        }
        HTableDescriptor htd = null;
        try {
            htd = HTableDescriptor.parseFrom((byte[])content);
        }
        catch (DeserializationException e) {
            throw new IOException("content=" + Bytes.toShort((byte[])content), e);
        }
        if (rewritePb && !ProtobufUtil.isPBMagicPrefix((byte[])content)) {
            Path tableInfoDir = status.getPath().getParent();
            Path tableDir = tableInfoDir.getParent();
            FSTableDescriptors.writeTableDescriptor(fs, htd, tableDir, status);
        }
        return htd;
    }

    @VisibleForTesting
    Path updateTableDescriptor(HTableDescriptor htd) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot update a table descriptor - in read only mode");
        }
        Path tableDir = this.getTableDir(htd.getTableName());
        Path p = FSTableDescriptors.writeTableDescriptor(this.fs, htd, tableDir, this.getTableInfoPath(tableDir));
        if (p == null) {
            throw new IOException("Failed update");
        }
        LOG.info((Object)("Updated tableinfo=" + p));
        return p;
    }

    public void deleteTableDescriptorIfExists(TableName tableName) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot delete a table descriptor - in read only mode");
        }
        Path tableDir = this.getTableDir(tableName);
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        FSTableDescriptors.deleteTableDescriptorFiles(this.fs, tableInfoDir, Integer.MAX_VALUE);
    }

    private static void deleteTableDescriptorFiles(FileSystem fs, Path dir, int maxSequenceId) throws IOException {
        FileStatus[] status;
        for (FileStatus file : status = FSUtils.listStatus(fs, dir, TABLEINFO_PATHFILTER)) {
            Path path = file.getPath();
            int sequenceId = FSTableDescriptors.getTableInfoSequenceId(path);
            if (sequenceId > maxSequenceId) continue;
            boolean success = FSUtils.delete(fs, path, false);
            if (success) {
                LOG.debug((Object)("Deleted table descriptor at " + path));
                continue;
            }
            LOG.error((Object)("Failed to delete descriptor at " + path));
        }
    }

    private static Path writeTableDescriptor(FileSystem fs, HTableDescriptor htd, Path tableDir, FileStatus currentDescriptorFile) throws IOException {
        int currentSequenceId;
        Path tmpTableDir = new Path(tableDir, TMP_DIR);
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        int newSequenceId = currentSequenceId = currentDescriptorFile == null ? 0 : FSTableDescriptors.getTableInfoSequenceId(currentDescriptorFile.getPath());
        int retries = 10;
        int retrymax = currentSequenceId + retries;
        Path tableInfoDirPath = null;
        do {
            String filename;
            Path tempPath;
            if (fs.exists(tempPath = new Path(tmpTableDir, filename = FSTableDescriptors.getTableInfoFileName(++newSequenceId)))) {
                LOG.debug((Object)(tempPath + " exists; retrying up to " + retries + " times"));
                continue;
            }
            tableInfoDirPath = new Path(tableInfoDir, filename);
            try {
                FSTableDescriptors.writeHTD(fs, tempPath, htd);
                fs.mkdirs(tableInfoDirPath.getParent());
                if (!fs.rename(tempPath, tableInfoDirPath)) {
                    throw new IOException("Failed rename of " + tempPath + " to " + tableInfoDirPath);
                }
                LOG.debug((Object)("Wrote descriptor into: " + tableInfoDirPath));
                break;
            }
            catch (IOException ioe) {
                LOG.debug((Object)"Failed write and/or rename; retrying", (Throwable)ioe);
                if (!FSUtils.deleteDirectory(fs, tempPath)) {
                    LOG.warn((Object)("Failed cleanup of " + tempPath));
                }
                tableInfoDirPath = null;
            }
        } while (newSequenceId < retrymax);
        if (tableInfoDirPath != null) {
            FSTableDescriptors.deleteTableDescriptorFiles(fs, tableInfoDir, newSequenceId - 1);
        }
        return tableInfoDirPath;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeHTD(FileSystem fs, Path p, HTableDescriptor htd) throws IOException {
        FSDataOutputStream out = fs.create(p, false);
        try {
            out.write(htd.toByteArray());
        }
        finally {
            out.close();
        }
    }

    public boolean createTableDescriptor(HTableDescriptor htd) throws IOException {
        return this.createTableDescriptor(htd, false);
    }

    public boolean createTableDescriptor(HTableDescriptor htd, boolean forceCreation) throws IOException {
        Path tableDir = this.getTableDir(htd.getTableName());
        return this.createTableDescriptorForTableDirectory(tableDir, htd, forceCreation);
    }

    public boolean createTableDescriptorForTableDirectory(Path tableDir, HTableDescriptor htd, boolean forceCreation) throws IOException {
        Path p;
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot create a table descriptor - in read only mode");
        }
        FileStatus status = FSTableDescriptors.getTableInfoPath(this.fs, tableDir);
        if (status != null) {
            LOG.debug((Object)("Current tableInfoPath = " + status.getPath()));
            if (!forceCreation && this.fs.exists(status.getPath()) && status.getLen() > 0L && FSTableDescriptors.readTableDescriptor(this.fs, status, false).equals((Object)htd)) {
                LOG.debug((Object)"TableInfo already exists.. Skipping creation");
                return false;
            }
        }
        return (p = FSTableDescriptors.writeTableDescriptor(this.fs, htd, tableDir, status)) != null;
    }

    private static class TableDescriptorAndModtime {
        private final HTableDescriptor htd;
        private final long modtime;

        TableDescriptorAndModtime(long modtime, HTableDescriptor htd) {
            this.htd = htd;
            this.modtime = modtime;
        }

        long getModtime() {
            return this.modtime;
        }

        HTableDescriptor getTableDescriptor() {
            return this.htd;
        }
    }
}

