/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.vault.fs.io;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.annotation.Nonnull;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.CloseShieldInputStream;
import org.apache.jackrabbit.vault.fs.api.VaultInputSource;
import org.apache.jackrabbit.vault.fs.config.ConfigurationException;
import org.apache.jackrabbit.vault.fs.config.DefaultMetaInf;
import org.apache.jackrabbit.vault.fs.config.MetaInf;
import org.apache.jackrabbit.vault.fs.config.VaultSettings;
import org.apache.jackrabbit.vault.fs.io.AbstractArchive;
import org.apache.jackrabbit.vault.fs.io.Archive;
import org.apache.jackrabbit.vault.util.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZipStreamArchive
extends AbstractArchive {
    private static final Logger log = LoggerFactory.getLogger(ZipStreamArchive.class);
    private static final int DEFAULT_BUFFER_SIZE = 0x100000;
    private InputStream in;
    private File tmpFile;
    private RandomAccessFile raf;
    private byte[] decompressed;
    private final int maxBufferSize;
    private int pos;
    private EntryImpl root;
    private DefaultMetaInf inf;
    private final byte[] buffer = new byte[65536];

    public ZipStreamArchive(@Nonnull InputStream in) {
        this(in, 0x100000);
    }

    public ZipStreamArchive(@Nonnull InputStream in, int maxBufferSize) {
        this.in = in;
        this.maxBufferSize = maxBufferSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void open(boolean strict) throws IOException {
        if (this.raf != null || this.decompressed != null) {
            return;
        }
        this.decompressed = new byte[this.maxBufferSize];
        this.pos = 0;
        this.root = new EntryImpl("");
        this.inf = new DefaultMetaInf();
        ZipInputStream zin = new ZipInputStream(this.in);
        try {
            ZipEntry entry;
            while ((entry = zin.getNextEntry()) != null) {
                String[] names;
                String name = entry.getName();
                if (name.startsWith("META-INF/vault/")) {
                    try {
                        this.inf.load((InputStream)new CloseShieldInputStream((InputStream)zin), "inputstream:" + name);
                    }
                    catch (ConfigurationException e) {
                        throw new IOException(e);
                    }
                }
                if ((names = Text.explode(name, 47)).length <= 0) continue;
                EntryImpl je = this.root;
                for (int i = 0; i < names.length; ++i) {
                    if (i == names.length - 1 && !entry.isDirectory()) {
                        long pos = this.getPosition();
                        long len = this.copy(zin);
                        je = je.add(new EntryImpl(names[i], entry.getTime(), pos, len));
                        continue;
                    }
                    je = je.add(names[i]);
                }
                if (!log.isDebugEnabled()) continue;
                log.debug("scanning jar: {}", (Object)name);
            }
            if (this.inf.getFilter() == null) {
                log.debug("Zip stream does not contain filter definition.");
            }
            if (this.inf.getConfig() == null) {
                log.debug("Zip stream does not contain vault config.");
            }
            if (this.inf.getSettings() == null) {
                log.debug("Zip stream does not contain vault settings. using default.");
                VaultSettings settings = new VaultSettings();
                settings.getIgnoredNames().add(".svn");
                this.inf.setSettings(settings);
            }
            if (this.inf.getProperties() == null) {
                log.debug("Zip stream does not contain properties.");
            }
            if (this.inf.getNodeTypes().isEmpty()) {
                log.debug("Zip stream does not contain nodetypes.");
            }
        }
        finally {
            IOUtils.closeQuietly((InputStream)zin);
        }
    }

    private long getPosition() throws IOException {
        if (this.raf != null) {
            return this.raf.getFilePointer();
        }
        return this.pos;
    }

    private long copy(@Nonnull InputStream in) throws IOException {
        if (this.raf != null) {
            return this.copyToRaf(in);
        }
        return this.copyToBuffer(in);
    }

    private long copyToBuffer(@Nonnull InputStream in) throws IOException {
        int read;
        int total = 0;
        while ((read = in.read(this.decompressed, this.pos, this.decompressed.length - this.pos)) > 0) {
            total += read;
            this.pos += read;
            if (this.pos != this.decompressed.length) continue;
            this.tmpFile = File.createTempFile("__vlttmpbuffer", ".dat");
            this.raf = new RandomAccessFile(this.tmpFile, "rw");
            this.raf.write(this.decompressed);
            this.decompressed = null;
            return (long)total + this.copyToRaf(in);
        }
        return total;
    }

    private long copyToRaf(@Nonnull InputStream in) throws IOException {
        int read;
        int total = 0;
        while ((read = in.read(this.buffer)) > 0) {
            this.raf.write(this.buffer, 0, read);
            total += read;
        }
        return total;
    }

    @Override
    public InputStream openInputStream(Archive.Entry entry) throws IOException {
        return this.createInputStream((EntryImpl)entry);
    }

    @Override
    public VaultInputSource getInputSource(Archive.Entry entry) throws IOException {
        return new RafInputSource((EntryImpl)entry);
    }

    @Override
    public MetaInf getMetaInf() {
        return this.inf;
    }

    @Override
    public void close() {
        if (this.raf != null) {
            try {
                this.raf.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            this.raf = null;
        }
        if (this.tmpFile != null) {
            FileUtils.deleteQuietly((File)this.tmpFile);
            this.tmpFile = null;
        }
        if (this.decompressed != null) {
            this.decompressed = new byte[0];
        }
    }

    @Override
    public Archive.Entry getRoot() throws IOException {
        return this.root;
    }

    public boolean isBuffered() {
        return this.decompressed != null;
    }

    private InputStream createInputStream(@Nonnull EntryImpl entry) {
        if (this.raf == null) {
            return new ByteArrayInputStream(this.decompressed, (int)entry.pos, (int)entry.len);
        }
        return new RafInputStream(entry);
    }

    private static class EntryImpl
    implements Archive.Entry {
        public final String name;
        public final long time;
        public final long pos;
        public final long len;
        public Map<String, EntryImpl> children;

        private EntryImpl(String name) {
            this.name = name;
            this.time = 0L;
            this.pos = -1L;
            this.len = 0L;
        }

        private EntryImpl(String name, long time, long pos, long len) {
            this.name = name;
            this.time = time;
            this.pos = pos;
            this.len = len;
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public boolean isDirectory() {
            return this.pos < 0L;
        }

        public EntryImpl add(EntryImpl e) {
            if (this.children == null) {
                this.children = new LinkedHashMap<String, EntryImpl>();
            }
            this.children.put(e.getName(), e);
            return e;
        }

        public EntryImpl add(String name) {
            EntryImpl e;
            if (this.children == null) {
                this.children = new LinkedHashMap<String, EntryImpl>();
            } else {
                e = this.children.get(name);
                if (e != null) {
                    return e;
                }
            }
            e = new EntryImpl(name);
            this.children.put(name, e);
            return e;
        }

        @Override
        public Collection<? extends Archive.Entry> getChildren() {
            return this.children == null ? Collections.emptyList() : this.children.values();
        }

        @Override
        public Archive.Entry getChild(String name) {
            return this.children == null ? null : this.children.get(name);
        }
    }

    private class RafInputStream
    extends InputStream {
        private long pos;
        private long end;
        private long mark;

        private RafInputStream(EntryImpl entry) {
            this.pos = entry.pos;
            this.end = this.pos + entry.len;
        }

        @Override
        public int read() throws IOException {
            if (this.pos < this.end) {
                ZipStreamArchive.this.raf.seek(this.pos++);
                return ZipStreamArchive.this.raf.read();
            }
            return -1;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            if (this.pos >= this.end) {
                return -1;
            }
            len = Math.min(len, (int)(this.end - this.pos));
            ZipStreamArchive.this.raf.seek(this.pos);
            int read = ZipStreamArchive.this.raf.read(b, off, len);
            if (read < 0) {
                return -1;
            }
            this.pos += (long)read;
            return read;
        }

        @Override
        public int read(byte[] b) throws IOException {
            return this.read(b, 0, b.length);
        }

        @Override
        public long skip(long n) throws IOException {
            if (this.pos >= this.end) {
                return -1L;
            }
            n = Math.min(n, this.end - this.pos);
            this.pos += n;
            return n;
        }

        @Override
        public int available() throws IOException {
            return (int)(this.end - this.pos);
        }

        @Override
        public void close() throws IOException {
        }

        @Override
        public void mark(int readlimit) {
            this.mark = this.pos;
        }

        @Override
        public void reset() throws IOException {
            this.pos = this.mark;
        }

        @Override
        public boolean markSupported() {
            return true;
        }
    }

    private class RafInputSource
    extends VaultInputSource {
        private final EntryImpl entry;

        private RafInputSource(EntryImpl entry) {
            this.entry = entry;
        }

        @Override
        public InputStream getByteStream() {
            return ZipStreamArchive.this.createInputStream(this.entry);
        }

        @Override
        public long getContentLength() {
            return this.entry.len;
        }

        @Override
        public long getLastModified() {
            return this.entry.time;
        }
    }
}

