/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.producer.internals;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.kafka.clients.ClientRequest;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.KafkaClient;
import org.apache.kafka.clients.producer.internals.Metadata;
import org.apache.kafka.clients.producer.internals.RecordAccumulator;
import org.apache.kafka.clients.producer.internals.RecordBatch;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.InvalidMetadataException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.metrics.Measurable;
import org.apache.kafka.common.metrics.MetricConfig;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.metrics.stats.Rate;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.ProduceRequest;
import org.apache.kafka.common.requests.ProduceResponse;
import org.apache.kafka.common.requests.RequestSend;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Sender
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(Sender.class);
    private final KafkaClient client;
    private final RecordAccumulator accumulator;
    private final Metadata metadata;
    private final int maxRequestSize;
    private final short acks;
    private final int requestTimeout;
    private final int retries;
    private final Time time;
    private volatile boolean running;
    private final SenderMetrics sensors;

    public Sender(KafkaClient client, Metadata metadata, RecordAccumulator accumulator, int maxRequestSize, short acks, int retries, int requestTimeout, Metrics metrics, Time time) {
        this.client = client;
        this.accumulator = accumulator;
        this.metadata = metadata;
        this.maxRequestSize = maxRequestSize;
        this.running = true;
        this.requestTimeout = requestTimeout;
        this.acks = acks;
        this.retries = retries;
        this.time = time;
        this.sensors = new SenderMetrics(metrics);
    }

    @Override
    public void run() {
        log.debug("Starting Kafka producer I/O thread.");
        while (this.running) {
            try {
                this.run(this.time.milliseconds());
            }
            catch (Exception e) {
                log.error("Uncaught error in kafka producer I/O thread: ", (Throwable)e);
            }
        }
        log.debug("Beginning shutdown of Kafka producer I/O thread, sending remaining records.");
        while (this.accumulator.hasUnsent() || this.client.inFlightRequestCount() > 0) {
            try {
                this.run(this.time.milliseconds());
            }
            catch (Exception e) {
                log.error("Uncaught error in kafka producer I/O thread: ", (Throwable)e);
            }
        }
        this.client.close();
        log.debug("Shutdown of Kafka producer I/O thread has completed.");
    }

    public void run(long now) {
        Cluster cluster = this.metadata.fetch();
        RecordAccumulator.ReadyCheckResult result = this.accumulator.ready(cluster, now);
        if (result.unknownLeadersExist) {
            this.metadata.requestUpdate();
        }
        Iterator<Node> iter = result.readyNodes.iterator();
        while (iter.hasNext()) {
            Node node = iter.next();
            if (this.client.ready(node, now)) continue;
            iter.remove();
        }
        Map<Integer, List<RecordBatch>> batches = this.accumulator.drain(cluster, result.readyNodes, this.maxRequestSize, now);
        List<ClientRequest> requests = this.createProduceRequests(batches, now);
        this.sensors.updateProduceRequestMetrics(requests);
        if (result.readyNodes.size() > 0) {
            log.trace("Nodes with data ready to send: {}", result.readyNodes);
            log.trace("Created {} produce requests: {}", (Object)requests.size(), requests);
        }
        List<ClientResponse> responses = this.client.poll(requests, result.nextReadyCheckDelayMs, now);
        for (ClientResponse response : responses) {
            if (response.wasDisconnected()) {
                this.handleDisconnect(response, now);
                continue;
            }
            this.handleResponse(response, now);
        }
    }

    public void initiateClose() {
        this.running = false;
        this.accumulator.close();
        this.wakeup();
    }

    private void handleDisconnect(ClientResponse response, long now) {
        log.trace("Cancelled request {} due to node {} being disconnected", (Object)response, (Object)response.request().request().destination());
        int correlation = response.request().request().header().correlationId();
        Map responseBatches = (Map)response.request().attachment();
        for (RecordBatch batch : responseBatches.values()) {
            this.completeBatch(batch, Errors.NETWORK_EXCEPTION, -1L, correlation, now);
        }
    }

    private void handleResponse(ClientResponse response, long now) {
        int correlationId = response.request().request().header().correlationId();
        log.trace("Received produce response from node {} with correlation id {}", (Object)response.request().request().destination(), (Object)correlationId);
        Map batches = (Map)response.request().attachment();
        if (response.hasResponse()) {
            ProduceResponse produceResponse = new ProduceResponse(response.responseBody());
            for (Map.Entry<TopicPartition, ProduceResponse.PartitionResponse> entry : produceResponse.responses().entrySet()) {
                TopicPartition tp = entry.getKey();
                ProduceResponse.PartitionResponse partResp = entry.getValue();
                Errors error = Errors.forCode(partResp.errorCode);
                RecordBatch batch = (RecordBatch)batches.get(tp);
                this.completeBatch(batch, error, partResp.baseOffset, correlationId, now);
            }
            this.sensors.recordLatency(response.request().request().destination(), response.requestLatencyMs());
        } else {
            for (RecordBatch batch : batches.values()) {
                this.completeBatch(batch, Errors.NONE, -1L, correlationId, now);
            }
        }
    }

    private void completeBatch(RecordBatch batch, Errors error, long baseOffset, long correlationId, long now) {
        if (error != Errors.NONE && this.canRetry(batch, error)) {
            log.warn("Got error produce response with correlation id {} on topic-partition {}, retrying ({} attempts left). Error: {}", new Object[]{correlationId, batch.topicPartition, this.retries - batch.attempts - 1, error});
            this.accumulator.reenqueue(batch, now);
            this.sensors.recordRetries(batch.topicPartition.topic(), batch.recordCount);
        } else {
            batch.done(baseOffset, error.exception());
            this.accumulator.deallocate(batch);
            if (error != Errors.NONE) {
                this.sensors.recordErrors(batch.topicPartition.topic(), batch.recordCount);
            }
        }
        if (error.exception() instanceof InvalidMetadataException) {
            this.metadata.requestUpdate();
        }
    }

    private boolean canRetry(RecordBatch batch, Errors error) {
        return batch.attempts < this.retries && error.exception() instanceof RetriableException;
    }

    private List<ClientRequest> createProduceRequests(Map<Integer, List<RecordBatch>> collated, long now) {
        ArrayList<ClientRequest> requests = new ArrayList<ClientRequest>(collated.size());
        for (Map.Entry<Integer, List<RecordBatch>> entry : collated.entrySet()) {
            requests.add(this.produceRequest(now, entry.getKey(), this.acks, this.requestTimeout, entry.getValue()));
        }
        return requests;
    }

    private ClientRequest produceRequest(long now, int destination, short acks, int timeout, List<RecordBatch> batches) {
        HashMap<TopicPartition, ByteBuffer> produceRecordsByPartition = new HashMap<TopicPartition, ByteBuffer>(batches.size());
        HashMap<TopicPartition, RecordBatch> recordsByPartition = new HashMap<TopicPartition, RecordBatch>(batches.size());
        for (RecordBatch batch : batches) {
            TopicPartition tp = batch.topicPartition;
            ByteBuffer recordsBuffer = batch.records.buffer();
            recordsBuffer.flip();
            produceRecordsByPartition.put(tp, recordsBuffer);
            recordsByPartition.put(tp, batch);
        }
        ProduceRequest request = new ProduceRequest(acks, timeout, produceRecordsByPartition);
        RequestSend send = new RequestSend(destination, this.client.nextRequestHeader(ApiKeys.PRODUCE), request.toStruct());
        return new ClientRequest(now, acks != 0, send, recordsByPartition);
    }

    public void wakeup() {
        this.client.wakeup();
    }

    private class SenderMetrics {
        private final Metrics metrics;
        public final Sensor retrySensor;
        public final Sensor errorSensor;
        public final Sensor queueTimeSensor;
        public final Sensor requestTimeSensor;
        public final Sensor recordsPerRequestSensor;
        public final Sensor batchSizeSensor;
        public final Sensor compressionRateSensor;
        public final Sensor maxRecordSizeSensor;

        public SenderMetrics(Metrics metrics) {
            this.metrics = metrics;
            this.batchSizeSensor = metrics.sensor("batch-size");
            this.batchSizeSensor.add("batch-size-avg", "The average number of bytes sent per partition per-request.", new Avg());
            this.batchSizeSensor.add("batch-size-max", "The max number of bytes sent per partition per-request.", new Max());
            this.compressionRateSensor = metrics.sensor("compression-rate");
            this.compressionRateSensor.add("compression-rate-avg", "The average compression rate of record batches.", new Avg());
            this.queueTimeSensor = metrics.sensor("queue-time");
            this.queueTimeSensor.add("record-queue-time-avg", "The average time in ms record batches spent in the record accumulator.", new Avg());
            this.queueTimeSensor.add("record-queue-time-max", "The maximum time in ms record batches spent in the record accumulator.", new Max());
            this.requestTimeSensor = metrics.sensor("request-time");
            this.requestTimeSensor.add("request-latency-avg", "The average request latency in ms", new Avg());
            this.requestTimeSensor.add("request-latency-max", "The maximum request latency in ms", new Max());
            this.recordsPerRequestSensor = metrics.sensor("records-per-request");
            this.recordsPerRequestSensor.add("record-send-rate", "The average number of records sent per second.", new Rate());
            this.recordsPerRequestSensor.add("records-per-request-avg", "The average number of records per request.", new Avg());
            this.retrySensor = metrics.sensor("record-retries");
            this.retrySensor.add("record-retry-rate", "The average per-second number of retried record sends", new Rate());
            this.errorSensor = metrics.sensor("errors");
            this.errorSensor.add("record-error-rate", "The average per-second number of record sends that resulted in errors", new Rate());
            this.maxRecordSizeSensor = metrics.sensor("record-size-max");
            this.maxRecordSizeSensor.add("record-size-max", "The maximum record size", new Max());
            this.maxRecordSizeSensor.add("record-size-avg", "The average record size", new Avg());
            this.metrics.addMetric("requests-in-flight", "The current number of in-flight requests awaiting a response.", new Measurable(){

                @Override
                public double measure(MetricConfig config, long now) {
                    return Sender.this.client.inFlightRequestCount();
                }
            });
            metrics.addMetric("metadata-age", "The age in seconds of the current producer metadata being used.", new Measurable(){

                @Override
                public double measure(MetricConfig config, long now) {
                    return (double)(now - Sender.this.metadata.lastUpdate()) / 1000.0;
                }
            });
        }

        public void maybeRegisterTopicMetrics(String topic) {
            String topicRecordsCountName = "topic." + topic + ".records-per-batch";
            Sensor topicRecordCount = this.metrics.getSensor(topicRecordsCountName);
            if (topicRecordCount == null) {
                topicRecordCount = this.metrics.sensor(topicRecordsCountName);
                topicRecordCount.add("topic." + topic + ".record-send-rate", new Rate());
                String topicByteRateName = "topic." + topic + ".bytes";
                Sensor topicByteRate = this.metrics.sensor(topicByteRateName);
                topicByteRate.add("topic." + topic + ".byte-rate", new Rate());
                String topicCompressionRateName = "topic." + topic + ".compression-rate";
                Sensor topicCompressionRate = this.metrics.sensor(topicCompressionRateName);
                topicCompressionRate.add("topic." + topic + ".compression-rate", new Avg());
                String topicRetryName = "topic." + topic + ".record-retries";
                Sensor topicRetrySensor = this.metrics.sensor(topicRetryName);
                topicRetrySensor.add("topic." + topic + ".record-retry-rate", new Rate());
                String topicErrorName = "topic." + topic + ".record-errors";
                Sensor topicErrorSensor = this.metrics.sensor(topicErrorName);
                topicErrorSensor.add("topic." + topic + ".record-error-rate", new Rate());
            }
        }

        public void updateProduceRequestMetrics(List<ClientRequest> requests) {
            long now = Sender.this.time.milliseconds();
            for (int i = 0; i < requests.size(); ++i) {
                ClientRequest request = requests.get(i);
                int records = 0;
                if (request.attachment() == null) continue;
                Map responseBatches = (Map)request.attachment();
                for (RecordBatch batch : responseBatches.values()) {
                    String topic = batch.topicPartition.topic();
                    this.maybeRegisterTopicMetrics(topic);
                    String topicRecordsCountName = "topic." + topic + ".records-per-batch";
                    Sensor topicRecordCount = Utils.notNull(this.metrics.getSensor(topicRecordsCountName));
                    topicRecordCount.record(batch.recordCount);
                    String topicByteRateName = "topic." + topic + ".bytes";
                    Sensor topicByteRate = Utils.notNull(this.metrics.getSensor(topicByteRateName));
                    topicByteRate.record(batch.records.sizeInBytes());
                    String topicCompressionRateName = "topic." + topic + ".compression-rate";
                    Sensor topicCompressionRate = Utils.notNull(this.metrics.getSensor(topicCompressionRateName));
                    topicCompressionRate.record(batch.records.compressionRate());
                    this.batchSizeSensor.record(batch.records.sizeInBytes(), now);
                    this.queueTimeSensor.record(batch.drainedMs - batch.createdMs, now);
                    this.compressionRateSensor.record(batch.records.compressionRate());
                    this.maxRecordSizeSensor.record(batch.maxRecordSize, now);
                    records += batch.recordCount;
                }
                this.recordsPerRequestSensor.record(records, now);
            }
        }

        public void recordRetries(String topic, int count) {
            long now = Sender.this.time.milliseconds();
            this.retrySensor.record(count, now);
            String topicRetryName = "topic." + topic + ".record-retries";
            Sensor topicRetrySensor = this.metrics.getSensor(topicRetryName);
            if (topicRetrySensor != null) {
                topicRetrySensor.record(count, now);
            }
        }

        public void recordErrors(String topic, int count) {
            long now = Sender.this.time.milliseconds();
            this.errorSensor.record(count, now);
            String topicErrorName = "topic." + topic + ".record-errors";
            Sensor topicErrorSensor = this.metrics.getSensor(topicErrorName);
            if (topicErrorSensor != null) {
                topicErrorSensor.record(count, now);
            }
        }

        public void recordLatency(int node, long latency) {
            String nodeTimeName;
            Sensor nodeRequestTime;
            long now = Sender.this.time.milliseconds();
            this.requestTimeSensor.record(latency, now);
            if (node >= 0 && (nodeRequestTime = this.metrics.getSensor(nodeTimeName = "node-" + node + ".latency")) != null) {
                nodeRequestTime.record(latency, now);
            }
        }
    }
}

