/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.client;

import java.io.IOException;
import java.util.Map;
import java.util.NavigableMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.ClusterStatus;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.MasterNotRunningException;
import org.apache.hadoop.hbase.RegionException;
import org.apache.hadoop.hbase.RemoteExceptionHandler;
import org.apache.hadoop.hbase.client.HConnection;
import org.apache.hadoop.hbase.client.HConnectionManager;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.ipc.HMasterInterface;
import org.apache.hadoop.hbase.ipc.HRegionInterface;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.MetaUtils;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.io.BooleanWritable;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.ipc.RemoteException;

public class HBaseAdmin {
    private final Log LOG = LogFactory.getLog((String)this.getClass().getName());
    final HConnection connection;
    private volatile HBaseConfiguration conf;
    private final long pause;
    private final int numRetries;
    private volatile HMasterInterface master;

    public HBaseAdmin(HBaseConfiguration conf) throws MasterNotRunningException {
        this.connection = HConnectionManager.getConnection(conf);
        this.conf = conf;
        this.pause = conf.getLong("hbase.client.pause", 30000L);
        this.numRetries = conf.getInt("hbase.client.retries.number", 5);
        this.master = this.connection.getMaster();
    }

    public HConnection getConnection() {
        return this.connection;
    }

    public HMasterInterface getMaster() throws MasterNotRunningException {
        return this.connection.getMaster();
    }

    public boolean isMasterRunning() {
        return this.connection.isMasterRunning();
    }

    public boolean tableExists(String tableName) throws MasterNotRunningException {
        return this.tableExists(Bytes.toBytes(tableName));
    }

    public boolean tableExists(byte[] tableName) throws MasterNotRunningException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        return this.connection.tableExists(tableName);
    }

    public HTableDescriptor[] listTables() throws IOException {
        return this.connection.listTables();
    }

    public HTableDescriptor getTableDescriptor(byte[] tableName) throws IOException {
        return this.connection.getHTableDescriptor(tableName);
    }

    private long getPauseTime(int tries) {
        int triesCount = tries;
        if (triesCount >= HConstants.RETRY_BACKOFF.length) {
            triesCount = HConstants.RETRY_BACKOFF.length - 1;
        }
        return this.pause * (long)HConstants.RETRY_BACKOFF[triesCount];
    }

    public void createTable(HTableDescriptor desc) throws IOException {
        HTableDescriptor.isLegalTableName(desc.getName());
        this.createTableAsync(desc);
        for (int tries = 0; tries < this.numRetries; ++tries) {
            try {
                this.connection.locateRegion(desc.getName(), HConstants.EMPTY_START_ROW);
                break;
            }
            catch (RegionException e) {
                if (tries == this.numRetries - 1) {
                    throw e;
                }
                try {
                    Thread.sleep(this.getPauseTime(tries));
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                continue;
            }
        }
    }

    public void createTableAsync(HTableDescriptor desc) throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        HTableDescriptor.isLegalTableName(desc.getName());
        try {
            this.master.createTable(desc);
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
    }

    public void deleteTable(String tableName) throws IOException {
        this.deleteTable(Bytes.toBytes(tableName));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteTable(byte[] tableName) throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        HTableDescriptor.isLegalTableName(tableName);
        HRegionLocation firstMetaServer = this.getFirstMetaServerForTable(tableName);
        try {
            this.master.deleteTable(tableName);
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
        HRegionInterface server = this.connection.getHRegionConnection(firstMetaServer.getServerAddress());
        HRegionInfo info = new HRegionInfo();
        for (int tries = 0; tries < this.numRetries; ++tries) {
            long scannerId = -1L;
            try {
                Scan scan = new Scan().addColumn(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
                scannerId = server.openScanner(firstMetaServer.getRegionInfo().getRegionName(), scan);
                Result values = server.next(scannerId);
                if (values == null || values.size() == 0) break;
                boolean found = false;
                NavigableMap<byte[], byte[]> infoValues = values.getFamilyMap(HConstants.CATALOG_FAMILY);
                for (Map.Entry e : infoValues.entrySet()) {
                    if (!Bytes.equals((byte[])e.getKey(), HConstants.REGIONINFO_QUALIFIER) || !Bytes.equals((info = (HRegionInfo)Writables.getWritable((byte[])e.getValue(), (Writable)info)).getTableDesc().getName(), tableName)) continue;
                    found = true;
                }
                if (!found) {
                    break;
                }
            }
            catch (IOException ex) {
                if (tries == this.numRetries - 1) {
                    if (ex instanceof RemoteException) {
                        ex = RemoteExceptionHandler.decodeRemoteException((RemoteException)((Object)ex));
                    }
                    throw ex;
                }
            }
            finally {
                if (scannerId != -1L) {
                    try {
                        server.close(scannerId);
                    }
                    catch (Exception ex) {
                        this.LOG.warn((Object)ex);
                    }
                }
            }
            try {
                Thread.sleep(this.getPauseTime(tries));
                continue;
            }
            catch (InterruptedException e) {
                // empty catch block
            }
        }
        HConnectionManager.deleteConnectionInfo(this.conf, false);
        this.LOG.info((Object)("Deleted " + Bytes.toString(tableName)));
    }

    public void enableTable(String tableName) throws IOException {
        this.enableTable(Bytes.toBytes(tableName));
    }

    public void enableTable(byte[] tableName) throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        try {
            this.master.enableTable(tableName);
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
        for (int tries = 0; tries < this.numRetries && !this.isTableEnabled(tableName); ++tries) {
            if (this.LOG.isDebugEnabled()) {
                this.LOG.debug((Object)("Sleep. Waiting for all regions to be enabled from " + Bytes.toString(tableName)));
            }
            try {
                Thread.sleep(this.getPauseTime(tries));
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (!this.LOG.isDebugEnabled()) continue;
            this.LOG.debug((Object)("Wake. Waiting for all regions to be enabled from " + Bytes.toString(tableName)));
        }
        if (!this.isTableEnabled(tableName)) {
            throw new IOException("unable to enable table " + Bytes.toString(tableName));
        }
        this.LOG.info((Object)("Enabled table " + Bytes.toString(tableName)));
    }

    public void disableTable(String tableName) throws IOException {
        this.disableTable(Bytes.toBytes(tableName));
    }

    public void disableTable(byte[] tableName) throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        try {
            this.master.disableTable(tableName);
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
        for (int tries = 0; tries < this.numRetries && !this.isTableDisabled(tableName); ++tries) {
            if (this.LOG.isDebugEnabled()) {
                this.LOG.debug((Object)("Sleep. Waiting for all regions to be disabled from " + Bytes.toString(tableName)));
            }
            try {
                Thread.sleep(this.getPauseTime(tries));
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (!this.LOG.isDebugEnabled()) continue;
            this.LOG.debug((Object)("Wake. Waiting for all regions to be disabled from " + Bytes.toString(tableName)));
        }
        if (!this.isTableDisabled(tableName)) {
            throw new RegionException("Retries exhausted, it took too long to wait for the table " + Bytes.toString(tableName) + " to be disabled.");
        }
        this.LOG.info((Object)("Disabled " + Bytes.toString(tableName)));
    }

    public boolean isTableEnabled(String tableName) throws IOException {
        return this.isTableEnabled(Bytes.toBytes(tableName));
    }

    public boolean isTableEnabled(byte[] tableName) throws IOException {
        return this.connection.isTableEnabled(tableName);
    }

    public boolean isTableDisabled(byte[] tableName) throws IOException {
        return this.connection.isTableDisabled(tableName);
    }

    public void addColumn(String tableName, HColumnDescriptor column) throws IOException {
        this.addColumn(Bytes.toBytes(tableName), column);
    }

    public void addColumn(byte[] tableName, HColumnDescriptor column) throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        HTableDescriptor.isLegalTableName(tableName);
        try {
            this.master.addColumn(tableName, column);
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
    }

    public void deleteColumn(String tableName, String columnName) throws IOException {
        this.deleteColumn(Bytes.toBytes(tableName), Bytes.toBytes(columnName));
    }

    public void deleteColumn(byte[] tableName, byte[] columnName) throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        HTableDescriptor.isLegalTableName(tableName);
        try {
            this.master.deleteColumn(tableName, columnName);
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
    }

    public void modifyColumn(String tableName, String columnName, HColumnDescriptor descriptor) throws IOException {
        this.modifyColumn(Bytes.toBytes(tableName), Bytes.toBytes(columnName), descriptor);
    }

    public void modifyColumn(byte[] tableName, byte[] columnName, HColumnDescriptor descriptor) throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        HTableDescriptor.isLegalTableName(tableName);
        try {
            this.master.modifyColumn(tableName, columnName, descriptor);
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
    }

    public void closeRegion(String regionname, Object ... args) throws IOException {
        this.closeRegion(Bytes.toBytes(regionname), args);
    }

    public void closeRegion(byte[] regionname, Object ... args) throws IOException {
        int len = args == null ? 0 : args.length;
        int xtraArgsCount = 1;
        Object[] newargs = new Object[len + xtraArgsCount];
        newargs[0] = regionname;
        if (args != null) {
            for (int i = 0; i < len; ++i) {
                newargs[i + xtraArgsCount] = args[i];
            }
        }
        this.modifyTable(HConstants.META_TABLE_NAME, HConstants.Modify.CLOSE_REGION, newargs);
    }

    public void flush(String tableNameOrRegionName) throws IOException {
        this.flush(Bytes.toBytes(tableNameOrRegionName));
    }

    public void flush(byte[] tableNameOrRegionName) throws IOException {
        this.modifyTable(tableNameOrRegionName, HConstants.Modify.TABLE_FLUSH);
    }

    public void compact(String tableNameOrRegionName) throws IOException {
        this.compact(Bytes.toBytes(tableNameOrRegionName));
    }

    public void compact(byte[] tableNameOrRegionName) throws IOException {
        this.modifyTable(tableNameOrRegionName, HConstants.Modify.TABLE_COMPACT);
    }

    public void majorCompact(String tableNameOrRegionName) throws IOException {
        this.majorCompact(Bytes.toBytes(tableNameOrRegionName));
    }

    public void majorCompact(byte[] tableNameOrRegionName) throws IOException {
        this.modifyTable(tableNameOrRegionName, HConstants.Modify.TABLE_MAJOR_COMPACT);
    }

    public void split(String tableNameOrRegionName) throws IOException {
        this.split(Bytes.toBytes(tableNameOrRegionName));
    }

    public void split(byte[] tableNameOrRegionName) throws IOException {
        this.modifyTable(tableNameOrRegionName, HConstants.Modify.TABLE_SPLIT);
    }

    private void modifyTable(byte[] tableNameOrRegionName, HConstants.Modify op) throws IOException {
        byte[][] byArrayArray;
        byte[] regionName;
        if (tableNameOrRegionName == null) {
            throw new IllegalArgumentException("Pass a table name or region name");
        }
        byte[] tableName = (byte[])(this.tableExists(tableNameOrRegionName) ? tableNameOrRegionName : null);
        Object object = regionName = (Object)(tableName == null ? tableNameOrRegionName : null);
        if (regionName == null) {
            byArrayArray = null;
        } else {
            byte[][] byArrayArray2 = new byte[1][];
            byArrayArray = byArrayArray2;
            byArrayArray2[0] = regionName;
        }
        byte[][] args = byArrayArray;
        this.modifyTable(tableName == null ? null : tableName, op, (Object[])args);
    }

    public void modifyTable(byte[] tableName, HTableDescriptor htd) throws IOException {
        this.modifyTable(tableName, HConstants.Modify.TABLE_SET_HTD, htd);
    }

    public void modifyTable(byte[] tableName, HConstants.Modify op, Object ... args) throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        if (tableName != null && !MetaUtils.isMetaTableName(tableName)) {
            HTableDescriptor.isLegalTableName(tableName);
        }
        Writable[] arr = null;
        try {
            switch (op) {
                case TABLE_SET_HTD: {
                    if (args == null || args.length < 1 || !(args[0] instanceof HTableDescriptor)) {
                        throw new IllegalArgumentException("SET_HTD requires a HTableDescriptor");
                    }
                    arr = new Writable[]{(HTableDescriptor)args[0]};
                    this.master.modifyTable(tableName, op, arr);
                    break;
                }
                case TABLE_COMPACT: 
                case TABLE_SPLIT: 
                case TABLE_MAJOR_COMPACT: 
                case TABLE_FLUSH: {
                    if (args != null && args.length > 0) {
                        arr = new Writable[1];
                        if (args[0] instanceof byte[]) {
                            arr[0] = new ImmutableBytesWritable((byte[])args[0]);
                        } else if (args[0] instanceof ImmutableBytesWritable) {
                            arr[0] = (ImmutableBytesWritable)args[0];
                        } else if (args[0] instanceof String) {
                            arr[0] = new ImmutableBytesWritable(Bytes.toBytes((String)args[0]));
                        } else {
                            throw new IllegalArgumentException("Requires byte[], String, orImmutableBytesWritable");
                        }
                    }
                    this.master.modifyTable(tableName, op, arr);
                    break;
                }
                case CLOSE_REGION: {
                    if (args == null || args.length < 1) {
                        throw new IllegalArgumentException("Requires at least a region name");
                    }
                    arr = new Writable[args.length];
                    for (int i = 0; i < args.length; ++i) {
                        if (args[i] instanceof byte[]) {
                            arr[i] = new ImmutableBytesWritable((byte[])args[i]);
                            continue;
                        }
                        if (args[i] instanceof ImmutableBytesWritable) {
                            arr[i] = (ImmutableBytesWritable)args[i];
                            continue;
                        }
                        if (args[i] instanceof String) {
                            arr[i] = new ImmutableBytesWritable(Bytes.toBytes((String)args[i]));
                            continue;
                        }
                        if (args[i] instanceof Boolean) {
                            arr[i] = new BooleanWritable(((Boolean)args[i]).booleanValue());
                            continue;
                        }
                        throw new IllegalArgumentException("Requires byte [] or ImmutableBytesWritable, not " + args[i]);
                    }
                    this.master.modifyTable(tableName, op, arr);
                    break;
                }
                default: {
                    throw new IOException("unknown modifyTable op " + (Object)((Object)op));
                }
            }
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
    }

    public synchronized void shutdown() throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        try {
            this.master.shutdown();
        }
        catch (RemoteException e) {
            throw RemoteExceptionHandler.decodeRemoteException(e);
        }
        finally {
            this.master = null;
        }
    }

    public ClusterStatus getClusterStatus() throws IOException {
        if (this.master == null) {
            throw new MasterNotRunningException("master has been shut down");
        }
        return this.master.getClusterStatus();
    }

    private HRegionLocation getFirstMetaServerForTable(byte[] tableName) throws IOException {
        return this.connection.locateRegion(HConstants.META_TABLE_NAME, HRegionInfo.createRegionName(tableName, null, "99999999999999"));
    }

    public static void checkHBaseAvailable(HBaseConfiguration conf) throws MasterNotRunningException {
        HBaseConfiguration copyOfConf = new HBaseConfiguration(conf);
        copyOfConf.setInt("hbase.client.retries.number", 1);
        new HBaseAdmin(copyOfConf);
    }
}

