/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.mesos;

public class MesosNativeLibrary {
  /**
   * Attempts to load the native library (if it was not previously loaded)
   * from the given path. If the path is null 'java.library.path' is used to
   * load the library.
   */
  public static void load(String path) {
    // Our JNI library will actually set 'loaded' to true once it is
    // loaded, that way the library can get loaded by a user via
    // 'System.load' in the event that they want to specify an
    // absolute path and we won't try and reload the library ourselves
    // (which would probably fail because 'java.library.path' might
    // not be set).
    if (loaded) {
      return;
    }

    // In some circumstances, such as when sandboxed class loaders are used,
    // the current thread's context class loader will not be able to see
    // MesosNativeLibrary (even when executing this code!).
    // We therefore, temporarily swap the thread's context class loader with
    // the class loader that loaded this class, for the duration of the native
    // library load.
    ClassLoader contextClassLoader =
        Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(
        MesosNativeLibrary.class.getClassLoader());
    try {
      if (path != null) {
        System.load(path);
      } else {
        // TODO(tillt): Change the default fallback to JNI specific library
        // once libmesos has been split.
        System.loadLibrary("mesos");
      }
    } catch (UnsatisfiedLinkError error) {
      System.err.println("Failed to load native Mesos library from " +
          (path != null ? path : System.getProperty("java.library.path")));
      throw error;
    } finally {
      Thread.currentThread().setContextClassLoader(contextClassLoader);
    }
  }

  public static void load() {
    // Try to get the JNI specific library path from the environment.
    String path = System.getenv("MESOS_NATIVE_JAVA_LIBRARY");

    // As a fallback, use deprecated environment variable to extract that path.
    if (path == null) {
      path = System.getenv("MESOS_NATIVE_LIBRARY");
      if (path != null) {
        System.out.println("Warning: MESOS_NATIVE_LIBRARY is deprecated, " +
            "use MESOS_NATIVE_JAVA_LIBRARY instead. Future releases will " +
            "not support JNI bindings via MESOS_NATIVE_LIBRARY.");
      }
    }

    load(path);
  }

  public static final String VERSION = "0.21.0";

  private static boolean loaded = false;
}
